/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/*      bigblock.c:                                    */
/*  All subroutine related to the bigblock (sperblock),            */
/*  including:                                          */
/*      init_bb:      initialise the bigblock structure      */
/*      read_bb:      read in ca grid and details            */
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Jul 1, 1998                                                  */
/****************************************************************/
/*       MODIFIED by:                                           */
/*       see RCS LOG at end of file                             */
/****************************************************************/
/****** To Do List **********************************************/
/****************************************************************/

/*RCS Id:$Id: bigblock.c 1491 2010-07-30 00:32:44Z  $*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>

#include "machine.h"
#include "fem.h"
#include "read_ctrl.h"
#include "blocks.h"
#include "nuc_lookup.h"
#include "gaussdev.h"
#include "ca_phy_calc.h"
#include "ff_mpflow.h"
#include "curvature.h"
#include "ext_inp.h"

extern CA_FLOAT global_pressure;

#ifdef PROCAST_CA
extern float t_final;
#endif

void init_pr_lookup (BB_struct * bp, Ctrl_str * cp);
void init_temp_lookup (BB_struct * bp, Ctrl_str * cp);
extern void alloc_ms (MultiS_struct * ms, int ele_1, int nc);

/* functions used from read_cap_ca.c */
extern int read_cap_ca (Ctrl_str *, BB_struct *);

/* functions used from subblock.c */
extern int read_geoplus (Ctrl_str *, BB_struct *);

/* functions used from readmat.c */
extern int read_matprop (Ctrl_str * cp, BB_struct * bp);

/* functions used from readphase.c */
extern int readphase (Ctrl_str * cp, BB_struct * bp);


/* functions used from thermo_trace_init.c */
extern int thermo_trace_init (TC_str * tc);

/* functions used from fidap_interp_init.c */
extern int init_output_img (RGB_struct *);

/* functions used from initface.c */
extern int init_facecode (Frame * cubeptr, int *ins, int dim);

/* functions used from initcube.c */
extern int init_cube (BB_struct * bp);

/*function from nbhd_def.c*/
extern void nbhd_def (BB_struct * bp);

/* pass in a blank BB structure, return inited        */
extern int alloc_bb (BB_struct * bp);

/**************************************************/
/*      Subroutine to initialize bigblock        */
/**************************************************/
/* pass in a blank BB structure, return inited        */
int init_bb (BB_struct * bp, Ctrl_str * cp)
{
  char command[MAX_STRING_LEN];
  int i, j, k, l, ele_num, ele_1;       /* tmp counters */
  int nc, local_index;
  int n_nuc_beut, n_nuc_teut;
  extern void srand48 (long);
  int errors = 0;
  MultiS_struct *ms;
  Nuc_str *np;
  FILE *fptr1, *fptr2;
  CA_FLOAT *cell_index_beut, *cell_index_teut;

  ms = &(bp->MultiSvals);
  np = &(bp->nprops);

   /********************set the number of components**/
  ele_num = cp->NUM_COMP;
  ele_1 = ele_num - 1;

   /************************************************/
  /* Start off the random number generator        */
   /************************************************/
  srand48 ((long) (cp->seed));
   /************************************************/
  /* If running SOLO, read geo from commands file, */
  /* or read it from CAP_CA file                  */
   /************************************************/


  /*-------------- Initialization of external T interplated from ProCast -------------*/
  
  if(bp->ctrl->extinput == 1){
    
    ext_ini_readin(cp, bp);
    fprintf(stderr, "EXT TEMP: CELL SIZE and DOMAIN SIZE are CHANGED according to the input file \n");
  }
  
  
  if (cp->solo || cp->procast || cp->physica ) {
    read_geoplus (cp, bp);
      /************************************************/
    /* If running CAP, read geo from commands file, */
    /* or read it from CAP_CA file                  */
      /************************************************/
  } else if (cp->cap) {
    read_cap_ca (cp, bp);
    bp->dim = THREE_D;
  } else {
    /* do nothing */
  }

  /* Open up output file                        */

  fprintf (stderr, "calc. all subsiduary values...\n");
   /************************************************/
  /* finished reading all of the header, now      */
  /* calc. all subsiduary values...            */
   /************************************************/
  bp->ntsb = bp->nsb[0] * bp->nsb[1] * bp->nsb[2];
  bp->ncsb = (bp->tnc[0] * bp->tnc[1] * bp->tnc[2]) / bp->ntsb;
  if (bp->ctrl->procast == 0) {
    for (i = 0; i < 3; i++)
      bp->size_bb[i] = (CA_FLOAT) bp->tnc[i] * bp->size_c[i];
  }

/*  
  if (cp->procast == 0) {
    bp->nsteps = (int) (bp->finish_time / bp->delt);
  }
  if (cp->procast == 0) {
    cp->nscrdumps = (int) bp->nsteps / cp->scr_dmp_freq;
  } else {
#ifdef PROCAST_CA
    cp->nscrdumps = (int) (t_final / (bp->delt * cp->scr_dmp_freq));
#endif
  }

  */
  
  bp->scr_dump_num = 0;
  bp->blk_dump_num = 0;
  bp->slice_dump_num = 0;

  bp->pore_err = 0;

   #ifdef CHIRAZI_MULTI
   /*******************************************/
  /* Multicomponent structure allocation     */
   /*******************************************/

  if (cp->diffuse_alloy_multi == TRUE) {
    alloc_ms (&(bp->MultiSvals), cp->NUM_COMP, bp->ncsb);
  }
  #endif /* CHIRAZI_MULTI */

   /************************************************/
  /* Read in all of the material properties.      */
   /************************************************/
  fprintf (stderr, "calling read_matprop().\n");
  read_matprop (cp, bp);

      /*****************/
      /*****************/
  /* allocate all  */
  /* the memory    */
  /* for the big   */
  /* block struct  */
      /*****************/
      /*****************/

  /***************   CA-PHISICA initialization   by LY********************/
  if(cp->physica) {
      fprintf(stderr, "Reading in the physica control file... \n");
   //   CA_Phy_readin(cp, bp);
      read_ca_phy (cp, bp) ;
  }
/*  
      fprintf(stderr, "Initialize the CA-PHYSICA mesh and Temperature. \n");
      phy_cacell_generation(bp, bp->orig_bb);
      ca_phy_interp(bp, bp->orig_bb);

      tecplotop(bp->ca_cord_x, bp->ca_cord_y, bp->ca_cord_z, bp->tncx, bp->tncy, bp->tncz, bp->ca_cell_Tar, bp->total_ca_number, "Phy_camicroInitT.plt");

  }
*/
/**************************************************************/ 
 
  
  alloc_bb (bp);

/*******************************************/
/* if there is a PRESSURE LOOKUP TABLE     */
/*******************************************/
  if (cp->pr_lookup == TRUE) {
    init_pr_lookup (bp, cp);
  }

/*******************************************/
/* if there is a TEMPERATURE LOOKUP TABLE     */
/*******************************************/
  if (cp->temp_lookup == TRUE) {
    init_temp_lookup (bp, cp);
  }

   /************************************************/
  /* Read in all of the phase diagram.      */
   /************************************************/
  if (bp->ctrl->thermocalc == TRUE) {
    fprintf (stderr, "calling readphase().\n");
    readphase (cp, bp);
  }

  /*                                         */
  /* calculate DAS from cooling rate         */
   /*******************************************/
  if (cp->coolrate == TRUE) {
    CA_FLOAT das_pre, das_exp;

    das_pre = DAS_PRE;
    das_exp = DAS_EXP;
    bp->mprops.das = bp->mprops.das_factor * das_pre * POW (((-cp->delT)), das_exp);
  }

  bp->c_sol_values->part_coef = bp->mprops.gasprops.part_coef[0]; /*  gas only  */

  /** \todo This needs to be redone for poly-component , create multiblock arrays*/
  if (cp->diffuse_alloy_poly == FALSE){
     bp->c_sol_alloy_values->part_coef = bp->mprops.alloyprops[0].part_coef[0];
  }else{
     printf("WARNING:%s: -- initialization of polycomponent values is not complete\n",__func__);
     /* ?????????????? */
  }
  bp->mprops.rhocp = bp->mprops.rho * bp->mprops.cp;
  bp->vol_bb = bp->size_bb[0] * bp->size_bb[1] * bp->size_bb[2];
  bp->area_bb = 2.0 * ((bp->size_bb[0] * bp->size_bb[1])
                       + (bp->size_bb[0] * bp->size_bb[2]) + (bp->size_bb[1] * bp->size_bb[2]));
  bp->vol_sb = bp->ncsb * bp->vol_c;
  fprintf (stderr, "vol_sb:,%.10e\n", bp->vol_sb);
  /* QSV = Q * S / V */
#ifdef HEAT_FLUX_SURF
  bp->mprops.QSV = bp->mprops.CA_Q * bp->area_bb / bp->vol_bb;
#else
  bp->mprops.QSV = bp->mprops.CA_Q;
#endif /*HEAT_FLUX_SURF */
  bp->CR = bp->mprops.QSV / bp->mprops.rhocp;

   /********************************************************/
  /* If the nuc model is DIST (distribution from lookup)  */
   /********************************************************/
  if (bp->nprops.nmodel == N_DIST) {
    errors += init_nuc_func_table (&(bp->nprops));
    errors += init_nuc_values (&(bp->nprops));
    errors += init_nuc_dist (&(bp->nprops));
    errors += init_nuc_lookup (&(bp->nprops));
    errors += output_nuc_lookup (&(bp->nprops));
#ifdef MAKE_NUC_TABLE
    {
      extern int dist_cell_nuc (BB_struct * bp, CA_FLOAT Tunder);
      CA_FLOAT testund = 0;
      int teststep;

      for (teststep = 0; teststep < 1000; teststep++) {
        dist_cell_nuc (bp, testund);
        testund += 0.01;
      }
    }

    exit (0);
#endif /*MAKE_NUC_TABLE */
  }

  #ifdef CHIRAZI_MULTI
  /************************************************************************************/
  if (cp->diffuse_alloy_multi == TRUE && cp->eut_nuc_option == TRUE) {
    /* setup the threshold for binary and ternary eutectic nucleation */

    fptr1 = fopen ("eut_thr_bin.txt", "w+");
    fptr2 = fopen ("eut_thr_ter.txt", "w+");

    n_nuc_beut = (int) (bp->nprops.gd_max_beut * bp->vol_sb);

    /* no nuclei??? */
    if (n_nuc_beut < 1) {
      fprintf (stderr, "ERROR: init_buet_nuc_thresh: no nuclei! creating one ...\n");
      n_nuc_beut = 1;
    }

    /* more nuclei than cells???  */
    if (n_nuc_beut > bp->ncsb) {
      fprintf (stderr, "WARNING: init_bin_nuc_thresh: too many nuclei! Resetting ...\n");
      n_nuc_beut = bp->ncsb;
    }

    /* create the nuclei */
    fprintf (stderr, "init_bin_nuc_thresh:  Number of ternary eutectic nuclei will be: %i\n", n_nuc_beut);

    n_nuc_teut = (int) (bp->nprops.gd_max_teut * bp->vol_sb);

    /* no nuclei??? */
    if (n_nuc_teut < 1) {
      fprintf (stderr, "ERROR: init_tuet_nuc_thresh: no nuclei! creating one ...\n");
      n_nuc_teut = 1;
    }

    /* more nuclei than cells???  */
    if (n_nuc_teut > bp->ncsb) {
      fprintf (stderr, "WARNING: init_ter_nuc_thresh: too many nuclei! Resetting ...\n");
      n_nuc_teut = bp->ncsb;
    }

    /* create the nuclei */
    fprintf (stderr, "init_ter_nuc_thresh: Number of ternary eutectic nuclei will be: %i\n", n_nuc_teut);

    if (cp->block_nuc) {

      nc = bp->nc[0] * bp->nc[1] * bp->nc[2];

      bp->nprops.rand_function_beut = gaussdev;
      bp->nprops.rand_function_teut = gaussdev;

      for (j = 0; j < n_nuc_beut; j++) {
        local_index = (int) FLOOR (drand48 () * bp->ncsb);
        cell_index_beut[j] = local_index;
        np->beut_threshold[local_index] = (*(bp->nprops.rand_function_beut)) (bp->nprops.NucParamsBeut);
        if (np->beut_threshold[local_index] < 0.0)
          np->beut_threshold[local_index] = 0.5;
      }

      for (j = 0; j < nc; j++) {
        for (k = 0, l = 0; k < n_nuc_beut; k++) {
          if (j != cell_index_beut[k])
            l++;
        }
        if (l == n_nuc_beut)
          np->beut_threshold[j] = 500.0;
        fprintf (fptr1, "%d \t %f \n", j, np->beut_threshold[j]);
      }

      for (j = 0; j < n_nuc_teut; j++) {
        local_index = (int) FLOOR (drand48 () * bp->ncsb);
        cell_index_teut[j] = local_index;
        np->teut_threshold[local_index] = (*(bp->nprops.rand_function_teut)) (bp->nprops.NucParamsTeut);
        if (np->teut_threshold[local_index] < 0.0)
          np->teut_threshold[local_index] = 0.5;
      }

      for (j = 0; j < nc; j++) {
        for (k = 0, l = 0; k < n_nuc_teut; k++) {
          if (j != cell_index_teut[k])
            l++;
        }
        if (l == n_nuc_teut)
          np->teut_threshold[j] = 500.0;
        fprintf (fptr2, "%d \t %f \n", j, np->teut_threshold[j]);
      }

    } else {
      fprintf (stderr, "You have to set the threshold manually within the diffusion routine \n");
    }
    fclose (fptr1);
    fclose (fptr2);
    free (cell_index_beut);
    free (cell_index_teut);
  }
  #endif /* CHIRAZI_MULTI */
   /********************************************************************/

   /************************************************/
  /* Init tctrace geom and results if required...   */
   /************************************************/
  if (bp->ctrl->t_input) {
    bp->tc.fn = strdup (bp->ctrl->fn_t_input);
    fprintf (stderr, "Calling thermo_trace_init for initialisation.\n");
    thermo_trace_init (&(bp->tc));
    bp->Tinit = bp->tc.Temp[0]; /* override Tinit with first temp */
    fprintf (stderr, "Finished thermo_trace_init for initialisation.\n");
    {
         /*******************************************/
      /*                                         */
      /* calculate DAS based on tc trace file    */
         /*******************************************/
      CA_FLOAT das_pre, das_exp;
      CA_FLOAT ftemp, ltemp, coolrate, ftime, ltime;

      ftemp = bp->tc.Temp[0];
      ltemp = bp->tc.Temp[bp->tc.Nlines - 1];
      ftime = bp->tc.Time[0];
      ltime = bp->tc.Time[bp->tc.Nlines - 1];
      coolrate = (ltemp - ftemp) / (ltime - ftime);
      das_pre = DAS_PRE;
      das_exp = DAS_EXP;
      bp->mprops.das = bp->mprops.das_factor * das_pre * POW (((-coolrate)), das_exp);
    }

  }

  /**********    CA-PHYSICA Initialization  by lyuan ******************/

  if (bp->ctrl->physica) {
  
     fprintf(stderr, "CA_PHY: Read in CA_PHY control file... \n");
  
     if( bp->phy_tran == 0 ){
     fprintf(stderr, "CA_PHY: Initialize the physica Marco of STEADY STATE... \n");
     CA_Phy_readin(cp, bp);
 //    read_ca_phy (cp, bp) ;

     fprintf(stderr, "CA_PHY: Initialize the CA-PHYSICA mesh and Temperature. \n");
     phy_cacell_generation(bp, bp->orig_bb);
     ca_phy_interp(bp, bp->orig_bb);

     tecplotop(bp->ca_cord_x, bp->ca_cord_y, bp->ca_cord_z, bp->tncx, bp->tncy, bp->tncz, bp->ca_cell_Tar, bp->ca_cell_uar, bp->ca_cell_var, bp->ca_cell_war, bp->total_ca_number, "Phy_camicro.plt");
				
     bp->Tinit = bp->ca_cell_Tar[0];
     bp->outflag = 0;
    }else if(bp->phy_tran == 1 ){
     fprintf(stderr, "CA_PHY: Initialize the physica Marco of TRAN MODEL... \n");
     
     bp->fcount = 1;
     phy_readin(bp, bp->fcountin);
     datarestore(bp);
     bp->fcountin++;
     phy_readin(bp, bp->fcountin);
    
     bp->macro_dt = bp->phy_sim_time - bp->phy_sim_told;  

     fprintf(stderr, "CA_PHY: MACRO..next time step time: %lf \n", bp->phy_sim_time);
     fprintf(stderr, "CA_PHY: MACRO..old time step time: %lf \n", bp->phy_sim_told);
     fprintf(stderr, "CA_PHY: MACRO..time gap between two files: %lf \n",bp-> macro_dt);
   
     ca_cord_gen(bp, bp->orig_bb);
     phy_interp_slide(bp,bp->orig_bb,bp->phy_Told,bp->ca_Told);
     phy_interp_slide(bp,bp->orig_bb,bp->phy_T,bp->ca_T);
     phy_interp_Tnow(bp, 0.0, bp->macro_dt);
     bp->Tinit = bp->ca_Tnow[1][1][1];	    
    }
     fprintf(stderr, "CA_PHY: Finished Cell&Temperature initialisaztion \n");
  }

 if(bp->ctrl->extinput == 1){
   
    fprintf(stderr, "EXT TEMP: Initialize the External Temperature input... \n");
    bp->ext_count = 0;

    ext_readin(cp, bp, bp->ext_count);
    ext_dataexchange(bp);

    bp->ext_count ++;
    ext_readin(cp, bp, bp->ext_count);

    fprintf(stderr, "EXT TEMP: Finished Initialize the External Temperature input \n");


  }


  
  /**********    Fluidflow model Initialization  by lyuan ******************/

  
  bp->ffstartflag = 0;
  if (bp->ctrl->ffheattran || bp->ctrl->fluidflow ){
    if ( ff_timestep(bp) ) {
      fprintf(stderr, "ERROR: Time step is too big, exiting....\n");
      return 1;
    }

    if( bp->ctrl->fstarttime <= 0){
      fprintf(stderr, "Initialize fluid flow and heat transfer calculation... \n");
      initvelocity(bp);
      fprintf(stderr, "\t Total Fluid Cell number is %d \n", bp->totalffcnum);
     
      /* time step check for fluid flow, ...*/
      if ( ff_timestep(bp) ) {
	fprintf(stderr, "ERROR: Time step is too big, exiting....\n");
	return 1;
      }

      if (bp->tonecyc <= 0)  boundary(bp);
      else {
        boundary_cyc(bp);
        fprintf(stderr, " cycling Velociyt BCs applied, Cycle Time : %.2f s \n", bp->tonecyc);
      }

      fprintf(stderr, " Finished Velocity&Temperature initialisaztion !\n");
     
      bp->ffstartflag = 1;
    } else fprintf(stderr, "Fluid flow will start after %f secends !!! \n", bp->ctrl->fstarttime);
  }
  
  if(bp->ctrl->curvature_2D == 3){
    curv_init(bp);
  }

  /* init implemented neighbourhoods */
  nbhd_def (bp);


   /****************************************************/
  /* If solo_mode and auto_finish time, calcualte it! */
   /****************************************************/
  if (cp->solo && bp->auto_fin) {
    bp->autofin_time = -1.4 * (bp->mprops.rhocp * (bp->Tinit - bp->mprops.Tliq)
                               + bp->mprops.latentH) / bp->mprops.QSV;
    fprintf (stderr, "Autofin_time set = %g\n", bp->autofin_time);
  }

   /************************************************/
  /* Init output variables ....                   */
   /************************************************/
  fprintf (stderr, "Calling init_output_img for initialisation.\n");
  bp->ctrl->rgbp->grey = bp->ctrl->rgbgrey;
  init_output_img (bp->ctrl->rgbp);
  fprintf (stderr, "Finished init_output_img for initialisation.\n");

   /*************************************/
  /* Print out checks on input data... */
   /*************************************/
  fprintf (stderr, "Exiting init_bb().\n");
  fprintf (stderr, "sb_mask nsb: %d, %d, %d\n", bp->nsb[0], bp->nsb[1], bp->nsb[2]);

  return (0);
}                               /* end of init_bb subroutine */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_bigblock_c ()
{
  static char const rcsid[] = "$Id: bigblock.c 1491 2010-07-30 00:32:44Z  $";

  return (rcsid);
}

/* end of rcs_id_bigblock_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:38  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.4.2.2  2006/01/11 12:12:19  rcatwood
RCS Log:Altered to use the new solute properties structure arrays
RCS Log:
RCS Log:Revision 10.4.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:52  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:Revision 10.1  2005/11/03 11:56:46  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.2.8.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.3.4.3  2004/09/02 15:34:14  rcatwood
RCS Log:Prevented overflow in random seed value!
RCS Log:
RCS Log:Revision 9.3.4.2  2004/06/18 18:26:36  rcatwood
RCS Log:Added option for output freqency at uniform sim-time instead of steps
RCS Log:
RCS Log:Revision 9.3.4.1  2004/03/09 11:30:52  rcatwood
RCS Log:Changed to read materials properties from separate files for alloy and gas
RCS Log:
RCS Log:Revision 9.3  2003/11/18 13:22:40  rcatwood
RCS Log:Added routines to find and store the interface cells between the casting and the mould.
RCS Log:Added the surface cell storage structure to the subblock.
RCS Log:Improved mould source and nucleation  routines to use the surface cell structure
RCS Log:
RCS Log:Revision 9.2  2003/09/08 17:13:33  rcatwood
RCS Log:changed ca_feedback to use the average of the nearest cells for each node.
RCS Log:Fixed som warnings (implicit declarations)
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:33  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.2.2.6  2003/08/14 14:17:56  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.2.2.5  2003/05/14 11:49:53  rcatwood
RCS Log:Fixed temperature input to conform wiht new temperature calculation methods
RCS Log:and control input file
RCS Log:
RCS Log:Revision 8.2.2.4  2003/01/22 16:53:42  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.2.2.3  2003/01/17 16:09:29  rcatwood
RCS Log:Before changing all CA_FLOAT to CA_CA_FLOAT
RCS Log:
RCS Log:Revision 8.2.2.2  2003/01/15 19:02:00  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.3  2003/01/14 19:32:04  rcatwood
RCS Log:working checkin - not for multicomponent however
RCS Log:Temeprature routine still not harmonised
RCS Log:
RCS Log:Revision 8.1.6.2  2003/01/14 16:22:25  rcatwood
RCS Log:Removed many lint warnings from sb_decentered_step
RCS Log:Added signal function to ca_procast
RCS Log:Removed some unused files
RCS Log:
RCS Log:Revision 8.1.6.1  2003/01/14 12:59:39  rcatwood
RCS Log:Merged procast version which compiles successfullly
RCS Log:Not checked for running yet
RCS Log:Including WEI decentered square but not temperature routine links
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:00  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.6  2002/10/17 16:52:36  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.5.10.4  2002/10/03 15:54:51  rcatwood
RCS Log:Quick fix: restart to read ctrl file and replace values
RCS Log:Fixed file name 'ca_restart.in'
RCS Log:
RCS Log:Revision 7.5.10.3  2002/09/03 13:31:58  rcatwood
RCS Log:Merged with reorganized allocation routines, and adjusted nucleation to compartmentalize
RCS Log:the grain information assignment.
RCS Log:
RCS Log:Revision 7.5.10.2  2002/08/30 19:18:24  rcatwood
RCS Log:split bigblock and subblock allocation into seperate subroutines
RCS Log:from bigblock.c and open_sb.c: added alloc_bb.c and alloc_sb.c
RCS Log:
RCS Log:Revision 7.5.2.2  2002/07/09 11:46:38  rcatwood
RCS Log:Split big block memory allocation and value initialization into seperate files.
RCS Log:Use alloc_bb for reading the block file.
RCS Log:Test program read_binblocks.c working (almost)  all strings in ctrl, pore lists
RCS Log:
RCS Log:Revision 7.5.2.1  2002/05/29 14:24:50  rcatwood
RCS Log:Working up to subblock header & array data
RCS Log:Revision 7.5.10.1  2002/08/22 15:13:43  ww1
RCS Log:Merge of Wei Wang's decentered octahedron/square method
RCS Log:
RCS Log:Revision 7.5  2002/05/23 17:28:01  rcatwood
RCS Log:Used dbmalloc to eliminate all memory leaks!
RCS Log:Started close-sb routine to improve multiblock
RCS Log:
RCS Log:Revision 7.4  2001/07/20 13:30:53  rcatwood
RCS Log:added realtime screen output, improved grain nuc histogram
RCS Log:
RCS Log:Revision 7.3  2001/06/23 16:08:30  rcatwood
RCS Log:added virtual K for frac. solid when not using cliq_output
RCS Log:
RCS Log:Revision 7.2  2001/02/16 20:30:49  rcatwood
RCS Log:Added some commenting for DAS and Limrad
RCS Log:Removed some JUNK
RCS Log:
RCS Log:Revision 7.2  2001/02/16 20:29:17  rcatwood
RCS Log:Added comments to show DAS and Limrad areas.
RCS Log:Removed some ifdef JUNK sections
RCS Log:
RCS Log:Revision 7.1  2000/12/06 21:10:40  rcatwood
RCS Log:fixed up heatfolw, tctrace
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.1  2000/10/16 10:37:22  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:34  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.4  2000/07/11 16:32:13  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.3  2000/06/27 11:41:19  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.2  2000/06/06 18:43:04  rcatwood
RCS Log:Changed r start and growth of pores -- pore radius equilibrium
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.5  2000/02/16 10:51:47  rcatwood
RCS Log:Fixed grain structure
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.2  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
