/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/* nucprops.h                                                   */
/* Header file defining nucleation related structures.          */
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Wed Jul  1 18:38:31 bst 1998                                 */
/****************************************************************/
/*RCS Id:$Id: nucprops.h 887 2006-03-01 18:21:01Z rcatwood $*/

#ifndef NUCPROPS_H
#define NUCPROPS_H
#define NUCPROPSREV "nucprops.h $Revision: 887 $"

#define MAX_NSP 50
#define MAX_NUCAREA 150
#define MAX_NUC_PERTURB 20
#define NUC_PERTURB_F 3

/****************************************************************/
/* Structure to define a single clasical nucleation curve.      */
/* One structure must be created for each particle type.        */
/****************************************************************/
typedef struct nucleus {
   int fin;             /* TRUE if all nuclei consumed         */
   CA_FLOAT CA_A;           	/* constant "A"				*/
   CA_FLOAT B;           	/* constant "B"				*/
   CA_FLOAT ftheta;        /* f(theta)             		*/
   CA_FLOAT rad;	         /* average size				*/
   CA_FLOAT Nrate;	    	/* current nucleation rate 		*/
   CA_FLOAT Nmax;	      	/* max number of nuclei of this type    */
   CA_FLOAT Ntot;	      	/* total num. nucleated 		*/
} Nucleus_str;

/**************
**************************************************/
/* Structure to define an area for fixed het. nucleation.       */
/****************************************************************/
typedef struct perturb_nuc {
   CA_FLOAT v_f[NUC_PERTURB_F];  /* 0: magnitude of perturbation         */
                        /* 1: time perturb starts               */
                        /* 2: time perturbation lasts           */
} Nuc_Perturb_str;

#ifdef NEW_NUC
   /*
      The structure for storing the list of potential nuclei for the grains.
   */
   typedef struct nuc_list {
      int Cellnum ;
      int Cellxyz[3];
      CA_FLOAT thresh;
      struct nuc_list *next;
   } NUCLIST_str;

#endif /*NEW_NUC*/

 
/****************************************************************/
/* Structure to define an area for fixed het. nucleation.       */
/****************************************************************/
typedef struct nuc_area {
   CA_FLOAT dens;         	/* nucl. dens. in #/m^3                */
   int sbnum;	        /* sb that area is in                   */
   int lleft[3];	/* lower lefthand corner of area [cells]*/
   int uright[3];	/* upper righthand corner of area[cells]*/
} NucArea_struct;

/*******************************************/
/*                                         */
/*******************************************/
typedef struct nucdiststruc {
   int nuc_dist_func_type;
   int n_U_bins;          /* Number of nUc-activity bins           */
   int n_Tund_bins;       /* Number of Tundercooling bins          */
   CA_FLOAT Tund_incr;       /* increment of undercooling             */
   CA_FLOAT Unuc_incr;       /* increment of U                        */
   CA_FLOAT Tund_factor;     /* factor to multiply Tund for bin number*/
   CA_FLOAT * Number_U;      /* Array of number in each U bin         */
   CA_FLOAT * Nuctable;      /* Array of nuc rate by undercooling     */
   char ** Nuc_func_id;   /* Store the names of the functions for output */
   CA_FLOAT (**Nuc_func_table)(CA_FLOAT Unuc,CA_FLOAT par_one,CA_FLOAT par_two,CA_FLOAT par_three);
}Nuc_dist_str;


/****************************************************************/
/* Nucleation properties structures, Nuc_str contains           */
/* constans used for calculation of nucleation and              */
/* do not vary from one subblock to the next;                   */
/****************************************************************/
typedef struct nucprops  {
   int nmodel;          /* type of nucleation model:            */
                        /*    N_RAPPAZ - rappaz model           */
                        /*    N_HETERO - heterogenous model     */
                        /* etc. see nucprops.h                  */
   int nhet;		/* number of different het. nuclei      */
   Nucleus_str nuc[MAX_NSP];    /* ptr to array of heterogeneous nucl.  */
   int nareanuc;	/* number of different area nucl. spec. */
   NucArea_struct nap[MAX_NUCAREA];    /* ptr to array of area nuc str.  */
   int ngr;		/* number of grains			*/
   CA_FLOAT gd_max;        /* max. grain density per [cm^3]        */
   CA_FLOAT gd_max_surf;        /* max. grain density per [cm^3]        */
   CA_FLOAT gd_max_beut;        /* max. grain density for binary eutectic nucleation sites per [cm^3] */
   CA_FLOAT gd_max_teut;        /* max. grain density for ternary eutectic nucleation sites per [cm^3] */

/*used in the multiphase extension of the CAFD model THUINET  04/05 */
   CA_FLOAT gd_max_poly[NPHAMAX];  /* max. grain density for the second solid phase nucleation sites per [cm^3] */
   CA_FLOAT NucParams_poly[NPHAMAX][4];
/*END THUINET*/

   int gd_max_total;    /* total max. # grains for array size   */
   int oriented;        /* Orientation Calc: True/Flase         */
   CA_FLOAT NucParams[4];   /* the nucleation parameters array      */
   CA_FLOAT NucParamsBeut[4];   /* the nucleation parameters array for binary eutectic      */
   CA_FLOAT NucParamsTeut[4];   /* the nucleation parameters array for ternary eutectic     */
   CA_FLOAT NucParamsSurf[4];   /* The nucleatio parameters array for the surface */

                        /* Perturbation things                  */
   int perturb_on;        /* Using Fidap? TRUE/FALSE              */
   int n_perturb;  /* Using Fidap? TRUE/FALSE              */
   Nuc_Perturb_str perturb[MAX_NUC_PERTURB];  /* structure to add perturbations       */
   Nuc_dist_str nucdist; 
   CA_FLOAT (*rand_function) (CA_FLOAT * params); /* random number function with required distribution */
   CA_FLOAT (*rand_function_beut) (CA_FLOAT * params); /* random number function for binary eutectic */
   CA_FLOAT (*rand_function_teut) (CA_FLOAT * params); /* random number function ternary eutectic */
   CA_FLOAT *beut_threshold;
   CA_FLOAT *teut_threshold;
} Nuc_str;
/****************************************************************/
/* SB_buc_str contains nucleation values calculated             */
/* for a particular subblock (at a particular iteration)        */
/****************************************************************/
typedef struct sbnuc  {
                       /* Nucleation stuff for each subblock  	*/
   int Ni_active;      /* number made active                   	*/
   CA_FLOAT N_nuc_old;    /* Number calculated for previous iter. */
   CA_FLOAT N_nuc;        /* Number calculated this iteration     	*/
   CA_FLOAT Ni_sum;       /* sum of Ni that should be active      	*/
   CA_FLOAT fract_nuc;    /* carry fractions of nuclei             */
} SB_nuc_str;


#endif /* NUCPROPS_H */
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.2.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 10.2  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.2.4.1  2004/05/24 18:36:04  rcatwood
RCS Log:Added individual surface nucleation parameters
RCS Log:
RCS Log:Revision 9.2  2003/11/18 13:22:40  rcatwood
RCS Log:Added routines to find and store the interface cells between the casting and the mould.
RCS Log:Added the surface cell storage structure to the subblock.
RCS Log:Improved mould source and nucleation  routines to use the surface cell structure
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:37  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:44  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:01  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:47  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.0.8.2  2002/11/04 11:22:05  rcatwood
RCS Log:Check-in after Ali has gone
RCS Log:
RCS Log:Revision 7.0.8.1  2002/08/27 14:18:17  chirazi
RCS Log:adding files for multi-component-Procast version of CA
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:55:38  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.2  2000/10/24 14:55:52  rcatwood
RCS Log:before changing pore
RCS Log:
RCS Log:Revision 6.1  2000/10/16 10:37:22  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:37  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.2  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.1  1999/12/16 12:37:36  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
