/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/* subblock.c:							*/
/* Most subroutines related to the subblock,			*/
/* including:							*/
/*   init_sb:	        initialise the subblock structure	*/
/*   calc_sb:	        perform on timestep on subblock		*/
/*   write_sb:	        write out a subblock to bb_out.dat file	*/
/*   init_c_elm:	read in CAP ca element numbers          */
/*   init_c_elm_solo:   set c_elm to cell # for test purposes   */
/* Other subblock subroutine NOT included are:                  */
/*   sb_nuc:            (sb_nuc.c) Set nuclei for a subblock    */
/****************************************************************/
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Jul 1, 1998                                                  */
/****************************************************************/
/* 	MODIFIED by:						*/
/*  PDL: July 2, 1998						*/
/*  PDL: Aug 22, 1998						*/
/****************************************************************/
/****** To Do List **********************************************/
/*General:							*/
/* 1) make initial conc mor general                						*/
/* 2)                						*/
/****************************************************************/
/*RCS Id:$Id: subblock.c 887 2006-03-01 18:21:01Z rcatwood $*/
#include <stdio.h>
#include <stdlib.h>

#include "machine.h"
#include "blocks.h"
#include "ca_matrix.h"
#include "writeblocks.h"
#include "read_sb.h"
#include "find_max.h"
#include "nearnode.h"
#include "interp.h"

/* functions used from sb_nuc.c */
extern int sb_nuc (BB_struct * bp, int sbnum);

/*from sb_ca_step.c*/
extern int sb_ca_step (BB_struct * bp, int sbnum);

/*from pore.w*/
extern int sb_pore (BB_struct * bp, int sbnum);

/* functions used from sb_temp_calc.c */
extern int sb_temp_calc (BB_struct * bp, int sbnum);

/* subroutines later in the file ... */
int init_c_elm (BB_struct *, int);
int init_c_elm_solo (BB_struct *, int);
int init_sb_interp (BB_struct * bp, NODENB_str ** nndp);

/*Subroutine to set up interpolation grid */
int init_sb_interp (BB_struct * bp, NODENB_str ** nndp)
{
  int nr, nz, errors = 0;

#ifdef VERTGRAD
  nr = bp->nc[XAXIS];
  nz = bp->nc[YAXIS];
#else
  nz = bp->nc[XAXIS];
  nr = bp->nc[YAXIS];
#endif

  /*First allocate the structure body */
  if (!(*nndp = (NODENB_str *) malloc (sizeof (NODENB_str)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nnd malloc failed\n");
    errors++;
  }
  /*Allocate the node number array in RADIAL (y) direction */
  if (!((*nndp)->nl = (int *) malloc (nr * sizeof (int)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nl malloc failed\n");
    errors++;
  }

  /*Allocate array for right-hand radial node weight factor */
  if (!((*nndp)->wr = (CA_FLOAT *) malloc (nr * sizeof (CA_FLOAT)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nnd malloc failed\n");
    errors++;
  }
  /*Allocate array for left-hand radial node weight factor */
  if (!((*nndp)->wl = (CA_FLOAT *) malloc (nr * sizeof (CA_FLOAT)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nnd malloc failed\n");
    errors++;
  }
  /*Allocate the node number array in AXIAL (x) direction */
  if (!((*nndp)->nd = (int *) malloc (nz * sizeof (int)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nd malloc failed\n");
    errors++;
  }
  /*Allocate array for upper axial node weight factor */
  if (!((*nndp)->wd = (CA_FLOAT *) malloc (nz * sizeof (CA_FLOAT)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nnd malloc failed\n");
    errors++;
  }
  /*Allocate array for lower axial node weight factor */
  if (!((*nndp)->wu = (CA_FLOAT *) malloc (nz * sizeof (CA_FLOAT)))) {
    fprintf (stderr, "ERROR:init_sb_interp: nnd malloc failed\n");
    errors++;
  }
  return (errors);
}                               /*end of init_sb_interp */

/********************************************************/
/********************************************************/
/*      Subroutine to initialize subblock	  	*/
/********************************************************/
/********************************************************/
/* pass in a number of sb to be inited, create	  */
/* and initialise it.				  */
int init_sb (BB_struct * bp, int sbnum)
{
  int errors = 0;
  int i, j, k;                  /* tmp counters */
  SB_struct *sp;

  fprintf (stderr, "InitSb A...\n");
   /************************************************/
  /* Malloc the SB structure                      */
   /************************************************/
  /* malloc SB structure */
  if (!(bp->sb[sbnum] = (SB_struct *) malloc (sizeof (SB_struct)))) {
    fprintf (stderr, "ERROR: SB_struct malloc failed\n");
    return (1);
  }
  sp = bp->sb[sbnum];

   /********************************************************/
  /* malloc array to hold interpolation factors         */
  /* via subroutine init_sb_interp                      */
  /* This is needed to find temperature of non-open sb's */
   /********************************************************/

  if (bp->ctrl->fgrid_input == TRUE) {  /* interpolation structure */
    if (init_sb_interp (bp, &(sp->nnd)) != 0) {
      fprintf (stderr, "ERROR: init_sb: init_sb_interp failed.\n");
      return (1);
    }
    if (init_sb_interp (bp, &(sp->nnd_next)) != 0) {
      fprintf (stderr, "ERROR: init_sb: init_sb_interp failed.\n");
      return (1);
    }

  }

  /* end init interpolation structure */
 /************************************************/
  /* Set all the general values in the SB str.    */
 /************************************************/
  sp->num = sbnum;
  sp->done = FALSE;
  sp->open = FALSE;
  /* set initial Nucleation Values... */
  sp->ngr = 0;
  sp->ncsolid = 0;              /* used as check, MUST CHANGE when working with CAP */
  sp->sbnuc.Ni_active = 0;
  sp->sbnuc.N_nuc_old = 0.0;
  sp->sbnuc.N_nuc = 0.0;
  sp->sbnuc.Ni_sum = 0.0;
  sp->sbnuc.fract_nuc = 0.0;
  /* calc the orig. of the sb */
  if (bp->ntsb == 1) {
    for (i = 0; i < 3; i++) {
      sp->orig_sb[i] = bp->orig_bb[i];
      sp->orig_bb_idx[i] = 0;
    }
  } else {
    /* multiple subblocks */
    fprintf (stderr, "DEBUG: Reminder -- Multiple subblocks not tested. Test this routine and remove this message. \n");
    fprintf (stderr, "DEBUG: File: subblock.c line 194 \n");
    exit (0);

    k = sbnum / (bp->nsb[0] * bp->nsb[1]);
    j = (sbnum - k * (bp->nsb[0] * bp->nsb[1])) / bp->nsb[0];
    i = (sbnum - k * (bp->nsb[0] * bp->nsb[1]) - j * bp->nsb[0]);
    sp->orig_sb[0] = bp->orig_bb[0] + i * (bp->size_bb[0] / ((CA_FLOAT) bp->nsb[0]));
    sp->orig_sb[1] = bp->orig_bb[1] + j * (bp->size_bb[1] / ((CA_FLOAT) bp->nsb[1]));
    sp->orig_sb[2] = bp->orig_bb[2] + k * (bp->size_bb[2] / ((CA_FLOAT) bp->nsb[2]));

    /* set up the big-block relative index of the subblock */
    sp->orig_bb_idx[0] = i;
    sp->orig_bb_idx[1] = j;
    sp->orig_bb_idx[2] = k;
  }

  if (bp->ctrl->solo) {
    /* set initial Thermal Values... */
    sp->Tvals.Tinit = bp->Tinit;
    sp->Tvals.Tavg = bp->Tinit;
    sp->Tvals.TminReached = bp->Tinit;
    sp->Tvals.Tmin = bp->Tinit;
    sp->Tvals.Tmax = bp->Tinit;
    sp->Tvals.fsavg = 0.0;
    sp->Tvals.del_fs = 0.0;
      /**************************************************/
    /* read in a binary file of grain numbers - 8 bit */
    /* or Boundaries if in RECRYSTALLIZE mode         */
      /**************************************************/
    /*Set up the pore array for the subblock */
  }

  return (errors);
}                               /* end of init_sb subroutine */

/****************************************************************/
/****************************************************************/
/* Subroutine to read in the cap element number that a          */
/* cell belongs to from CAP CA file.                            */
/* IN:                                                          */
/*    BB_struct *bp:    bigblock structure                      */
/*    int sbnum:        index for subblock to read in           */
/* OUT:                                                         */
/*    BB_struct bp->sb[sbnum].c_elm[n]:    element number that  */
/*                      each cell is in.                        */
/****************************************************************/
/****************************************************************/
int init_c_elm (BB_struct * bp, int sbnum)
{
#ifdef JUNK
  int i, j, k, itmp;            /* tmp counters */
  long offset, ltmp;
  FILE *tfd;                    /* tmp filehandle */

  ltmp = (long) bp->sb_mask[sbnum];
  tfd = bp->ibb_fd;
  fseek (tfd, ltmp, SEEK_SET);
  /*  size_t fread (void *ptr, size_t size, size_t nitems, FILE *stream); */
  itmp = fread (bp->sb[sbnum]->c_elm, sizeof (int), (long) bp->ncsb, tfd);
  if (itmp != bp->ncsb) {
    fprintf (stderr, "ERROR: only %d [should be %d] cells read from file\n", itmp, bp->ncsb);
  }

  fseek (tfd, ltmp, 0);
#endif /*JUNK*/
    return (0);
}                               /* end of init_c_elm subroutine */

/****************************************************************/
/****************************************************************/
/* Subroutine to set cell element number if in solo mode        */
/* This is just a fudge for checking output...                  */
/* IN:                                                          */
/*    BB_struct *bp:    bigblock structure                      */
/*    int sbnum:        index for subblock to read in           */
/* OUT:                                                         */
/*    BB_struct bp->sb[sbnum].c_elm[n]:    element number that  */
/*                      each cell is in.                        */
/****************************************************************/
/****************************************************************/
int init_c_elm_solo (BB_struct * bp, int sbnum)
{
  int i, j, k, itmp;            /* tmp counters */
  int *ptr_c_elm;               /* tmp counters */

  ptr_c_elm = bp->sb[sbnum]->c_elm;
  for (i = 0; i < bp->ncsb; i++) {
    *(ptr_c_elm++) = (10000 * sbnum) + i;
  }

  return (0);
}                               /* end of init_c_elm_solo subroutine */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_subblock_c ()
{
  static char const rcsid[] = "$Id: subblock.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_subblock_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.2.8.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.2.4.1  2004/01/06 15:49:37  rcatwood
RCS Log:Fixed surface nucleation to work wtih Procast - moved transfer of NOT_CASTING information into open_sb
RCS Log:
RCS Log:Revision 9.2  2003/11/18 13:22:40  rcatwood
RCS Log:Added routines to find and store the interface cells between the casting and the mould.
RCS Log:Added the surface cell storage structure to the subblock.
RCS Log:Improved mould source and nucleation  routines to use the surface cell structure
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:40  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.2.2.5  2003/02/27 23:04:40  rcatwood
RCS Log:Removed use of old temperature routines , all temperatures shoudl
RCS Log:be determined by checking the array c_temp in teh subblock, if the
RCS Log:subblock is open
RCS Log:
RCS Log:Revision 8.2.2.4  2003/01/22 16:53:47  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.2.2.3  2003/01/17 16:09:30  rcatwood
RCS Log:Before changing all CA_FLOAT to CA_CA_FLOAT
RCS Log:
RCS Log:Revision 8.2.2.2  2003/01/15 19:02:02  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.0.8.1  2002/08/27 14:18:19  chirazi
RCS Log:adding files for multi-component-Procast version of CA
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:34  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.2  2000/07/11 16:32:13  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.2  2000/03/02 13:09:59  rcatwood
RCS Log:Fixed copy-mat bug.
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.6  2000/01/27 12:18:48  rcatwood
RCS Log:Overgrowth addressed. Bindump of t, fs
RCS Log:
RCS Log:Revision 4.5  2000/01/05 12:47:29  rcatwood
RCS Log:Files as of New year 2000
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.2  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
