/* #define DEBUG_FIDAP */
/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*   email p.d.lee or r.atwood or x.xu @ic.ac.uk for details    */
/****************************************************************/

/****************************************************************/
/* T_interp_calc.c:                                             */
/* Subroutine to interpolate the temperature at a point         */
/* in the CA code using values calculated in a                  */
/* transient state solution of the heat, mass and               */
/* momentum transfer in VAR ingots.                             */
/****************************************************************/
/****************************************************************/
/* Written by X. Xu, P.D. Lee & R.C. Atwood, Imperial College   */
/* Feb. 18, 2000                                                */
/****************************************************************/
/*      MODIFIED by:                                            */
/****************************************************************/
/****** To Do List **********************************************/
/*General:                                                      */
/* 1)                                                           */
/****************************************************************/
#include <stdio.h>
#include <math.h>
/* include header files requred by subroutines */
#include "machine.h"
#include "blocks.h"
#include "fidap.h"              /* included for def. of FGrid_struct */
#include "nearnode.h"
int search_r (FGrid_str * fg, CA_FLOAT r);

/****************************************************************/
/****************************************************************/
/* wfact_r_calc.c:                                             */
/* Subroutine to calculation the weighting factor in r direction */
/****************************************************************/
/* Input Variables:                                             */
/*   *fg:		ptr to the FGrid_str structure that     */
/*       		holds all FIDAP geometry and T's data.  */
/*   *bp:+		hold location of CA cell                */
/*      		                                        */
/* Output Variables:    weight factor in r direction            */
/*      		                                        */
/* Returned Value:      none                                    */
/****************************************************************/
/****************************************************************/

void wfact_r_calc (FGrid_str * fg, NODENB_str * node_ptr, BB_struct * bp, int sbnum)
{
  CA_FLOAT rc, rm;
  int j, jj, jc, jmax;
  CA_FLOAT sub_off, cell_offset, size_cell, *fg_r_pt;

  /* set pointers and variables for local usage */

  fg_r_pt = fg->r;
  sub_off = bp->sb[sbnum]->orig_sb[1];  /* lower left corner of the subblock */
  size_cell = bp->size_c[1];    /* cell size in z dir, it equals to that in r dir. */
  cell_offset = sub_off - 0.5 * size_cell;      /* center of the previous cell. */

  jmax = bp->nc[1];
  rc = cell_offset;             /* Initial value  */

  /* calculate positions of r arrays */
  for (j = 0; j < jmax; j++) {
    rc += size_cell;
    /* calculate r arrays of cells in a CFD domain */
    rm = rc;

         /*******************************************************/
    /*  check if the rc of cell is out of the CFD domain,  */
    /*  and given an error message if it is.               */
         /*******************************************************/
    if (rm < 0.0 || rm > fg_r_pt[fg->nr - 1]) {
      fprintf (stderr, "ERROR rm out of domain size [%f]", rm);
    } else {

      jc = search_r (fg, rm);

      node_ptr->nl[j] = jc;     /* nearest node on the left of the cell. */
       /***************************************************************/
      /* factor that the cell is away from the left/right nodes,     */
      /* when it lies on the left node, wl[i] = 1.                   */
       /***************************************************************/

      node_ptr->wl[j] = (fg_r_pt[jc + 1] - rm) / (fg_r_pt[jc + 1] - fg_r_pt[jc]);
      node_ptr->wr[j] = 1 - node_ptr->wl[j];
    }
  }
  /* end of wfact_r_calc     */
}

/* subroutine of bisection search */

int search_r (FGrid_str * fg, CA_FLOAT r)
{
  int i, im, iu, il, JP;
  CA_FLOAT *fg_r_pt;

  fg_r_pt = fg->r;
  iu = fg->nr - 1;
  il = 0;
  while (iu - il > 1) {
    im = (iu + il) / 2;
    if ((r > fg_r_pt[il]) && (r > fg_r_pt[im])) {
      il = im;
    } else {
      iu = im;
    }
  }
  JP = il;
  return (JP);
}

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_wfact_r_calc_c ()
{
  static char const rcsid[] = "$Id: wfact_r_calc.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_fidap_interp_calc_c subroutine */

/*RCS Id:$Id: wfact_r_calc.c 887 2006-03-01 18:21:01Z rcatwood $*/
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:48  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.12.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:40  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.6.1  2003/01/22 16:53:47  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:04  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:10:16  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.2  2000/03/02 13:09:59  rcatwood
RCS Log:Fixed copy-mat bug.
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 16:06:53  rcatwood
RCS Log:merge VAR and Multiblock
RCS Log:
RCS Log:Revision 5.0  2000/03/01 16:00:54  rcatwood
RCS Log:Merge robert and xxu versions
RCS Log:
RCS Log:Revision 1.6  2000/02/28 15:32:42  xxu
RCS Log:set local pointers for efficience
RCS Log:
RCS Log:Revision 1.5  2000/02/23 18:43:33  xxu
RCS Log:added prototyeps
RCS Log:
RCS Log:Revision 1.4  2000/02/23 16:37:12  rcatwood
RCS Log:Improved interpolation version
RCS Log:
RCS Log:Revision 1.3  2000/02/23 15:54:07  xxu
RCS Log:weighting fact in r direction calculation.
RCS Log:
RCS Log:Revision 1.2  2000/02/22 16:11:12  xxu
RCS Log:incremental improvement
RCS Log:
RCS Log:Revision 1.1  2000/02/22 14:20:26  xxu
RCS Log:Initial revision
RCS Log:
*/
