/**\file ca_solid.c */
/*

   Copyright (c) 1998 Dept. of Materials, ICSTM               
   
   All Rights Reserved                                        

   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM}

   The copyright notice above does not evidence any           
   actual or intended publication of such source code,        
   and is an unpublished work by Dept. of Materials, ICSTM.   
   This material contains CONFIDENTIAL INFORMATION that       
   is the property of Imperial College. Any use,              
   duplication or disclosure not specifically authorized      
   by Imperial College is strictly prohibited.                

 This code is part of the umats routines developed at in the  
 Materials Processing Group, Dept. of Materials, ICSTM.       

      email p.d.lee or r.atwood @@ic.ac.uk for details         

|ca_solid.c:|
 Main subroutine for a portable Cellular Automata code.	
 It uses a multi-block approach to solve large problems, and 
 can either be run by itself (using CA_WRAPPER),              
 as a postprocessor to fem programmes, or                     
 as a user subroutine within a fem programme (e.g. with ProCast)  
 The superblock is called: bigblock, and is divided into      
 subblocks.                                                   
 The current version simulates grain nucleation and growth    
 for eutectic equiaxed growth.                               
 Written by Peter D. Lee, Imperial College    		
 Wed Jul  1 18:38:31 bst 1998                 		
 See RCS log at end of file (CVS used)
*/

/*RCS ID:$Id: ca_solid.c 1491 2010-07-30 00:32:44Z  $*/
/* Includes and defines.*/
/* include system headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <signal.h>

/* include header files requred by subroutines */
#include "machine.h"
#include "read_ctrl.h"
#include "blocks.h"
#include "readfiles.h"
#include "writeblocks.h"
#include "interp.h"
#include "find_max.h"
/* headers needed for the package for reading finite-element grid data */
/* eg from Sandia code */
#include "read_list/read_fg_list.h"
#include "read_list/readwrite_fg.h"

/* different temperature calculation option funcitons */
#include "temp_calc.h"

/*CA-PHYSICA*/
#include "ca_phy_calc.h"

/*Fluidflow and heattransfer*/
#include "ff_mpflow.h"
#include "datatrans.h"


int init_output_img (RGB_struct * rgbp);

extern int pore_setup (BB_struct * bp, int sbnum);

/* Function Prototypes.*/
extern int init_output_img (RGB_struct * rgbp);
extern int pore_setup (BB_struct * bp, int sbnum);
extern void free_fg (FGrid_str * fg);

#ifdef TEST_CLOSE
extern int close_sb (BB_struct * bp, int sbnum);
#endif /*TEST_CLOSE */
extern void write_bin_blocks (BB_struct * bp);
extern void read_bin_blocks (BB_struct * bp, const char *fname);

extern int free_bb (BB_struct * bp);
extern int sb_boundary (BB_struct * bp, int sbnum);

/* functions used from bigblock.c */
extern int init_bb (BB_struct * bp, Ctrl_str * cp);

#ifdef VERBOSE_PROCAST
/* access to Procast variables, only for debugging */
/* in normal mode no access is needed in this routine */
#include "common.h"
extern float *t0;
extern float *t1;
#endif /* VERBOSE_PROCAST */

/* functions used from subblock.c */
extern int init_sb (BB_struct *, int);
extern int calc_sb (BB_struct *, int);
extern int open_sb (BB_struct *, int);

/* functions used from sb\_nuc.c */
extern int sb_nuc_area (BB_struct * bp, int sbnum, NucArea_struct * nap);

/* functions used from init\_sb\_neigh.c */
extern int init_sb_neigh (BB_struct *, int);

/* from sb_temp_calc.c */
extern void cell_temp_calc_procast (BB_struct * bp, int index_ca);
extern void find_cell_pressure (BB_struct * bp, int index_ca);
extern float min_temp_calc_procast (BB_struct * bp);

/* from step_output.c */
extern void step_screen_out (BB_struct * bp, int step);
extern void step_file_out (BB_struct * bp, int step);
extern void step_blk_out (BB_struct * bp, int step);

/* from ca_wrapper.c -- to test the signal */
extern int jflg;
extern int signal_change_freq;

/* from fg_read.c */
extern int fg_read (const char *listfilename, FGrid_str * fg, int fg_flag);

/* to set up the chosen mould boundary condition functions */
extern void setup_mould_src_function (Ctrl_str * cp, BB_struct * bp, Solute_props * sp);
extern void setup_temp_function (Ctrl_str * cp, BB_struct * bp);

extern void phy_temp_calc(BB_struct * bp);
extern void plt_op_fsv (BB_struct * bp, int sbnum);
extern void plt_op_pointinfo (BB_struct * bp, int sbnum, int i, int j, int k);
/* Routines to do any housekeeping necessary to allow control options to be */
/* changed upon restarting */

void restart_change_options (Ctrl_str * cp, BB_struct * bp)
{
  int sbnum = 0, i;
  int errors = 0;
  SB_struct *sp;
  CA_FLOAT k_gas, c_gas_liq;

  k_gas = bp->mprops.gasprops.part_coef[0];

  for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {
    sp = bp->sb[sbnum];
    if (sp->open != SB_OPEN)
      continue;
         /**********************************************/
         /**    Turn gas diffusion on when restarting **/
         /**********************************************/
    /* if we are turning on the gas calcualtion after a restart */
    /* we need to allocate and set the arrays */
    if (cp->restart_gas_on) {

      fprintf (stderr, "RESTARTING: turning on the gas diffusion.\n");
      sp->Svals[GAS].Cinit = bp->mprops.gasprops.Cinit; /*all the same for now! */
      /* lever rule partitioning */
      c_gas_liq = bp->mprops.gasprops.Cinit / (1 - (1 - k_gas) * sp->Tvals.fsavg);

      if (!(sp->c_sol = (CA_FLOAT *) calloc (bp->ncsb, sizeof (CA_FLOAT)))) {
        fprintf (stderr, "ERROR:restart_change_options: SB cell sol array malloc failed\n");
        exit (423);
      }
      /* set concentraion to initial value */
      /* according to the fraction solid in the cell */
      fprintf (stderr, "RESTARTING: setting initial gas concetnration to equilibrium partitioning.\n");
      for (i = 0; i < bp->ncsb; i++)
        *(sp->c_sol + i) = (1 - (1 - k_gas) * *(sp->c_fs + i)) * c_gas_liq;
      /* set value pointer array */
      bp->c_sol_values->block_array[sbnum] = sp->c_sol;
    }

    /* end restart gas */
 /************************************/
    /* Turn porosity on when restarting */
 /************************************/
    /* if we are turning on the pore calcualtion after a restart */
    /* we need to allocate and set the arrays,initialize threshold */
    if (cp->restart_pore_on) {
      errors += pore_setup (bp, sbnum);
      fprintf (stderr, "RESTARTING: turning on the pore calculation.\n");
      if (errors != 0) {
        fprintf (stderr, "ERROR:restart_change_options: pore setup failed\n");
        exit (errors);
      }
    }                           /* end restart pore */
  }                             /* end loop through subblocks */

    /*******************/
  /* reset the flags */
  /* if it is stopped and then restarted again, we do not want */
  /* to reallocate the arrays! */
    /*****************************/
  cp->restart_gas_on = 0;
  cp->restart_pore_on = 0;

}

/* decide how often to reload a finite-element grid */
void fg_find_steps (BB_struct * bp, FGrid_str * fg) {
  CA_FLOAT timedif;
  int stepdif;

  timedif = fg->tnext - fg->tstart;
  stepdif = (int) (FLOOR (timedif / bp->delt));
  fg->s_next = fg->s_start + stepdif;
  fg->stepdif = stepdif;
  fg->wf_add = (double) 1 / ((double) (stepdif));
  if (ABS (fg->wf_add) <= MINVAL) {
    fprintf (stderr, "ERROR: fg_find_steps: Incorrect wf_add or not enough precision %.5g\n", fg->wf_add);
    exit (0);
  }
}

/****************************************************************/
/* Beginning of the ca_solid program!				*/
/****************************************************************/
/* Main part.*/
/* Main part*/
/* Main part.*/

/**
 * ca_solid: function to
 * run a full micro step consisting of several micro timesteps 
 * or an entire stand-alone simulation.
 * @callgraph
 * @callergraph
*/
int ca_solid (int stat_flag, CA_FLOAT i_time, CA_FLOAT delt, Ctrl_str * cp, BB_struct * bp) {
  /* declare counters and output variables for main */
  int i, j, k, ii, sbnum;
  int fsprintflg = 0, tcprintflg = 0, tsprintflg = 0, tbprintflg = 0;   /* flags for testing output condition */
  int new_fg = FALSE;
  int h;
  static int fspchk[MAX_CTRL] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
  int err = 0;
  CA_FLOAT min_temp, start_temp;
  int step, pr_step, olddmp, oldscrdmp;
  int done = FALSE;
  CA_FLOAT ptime, del_ptime = 1.0;
  int IMAX, JMAX, KMAX;
  CA_FLOAT timenow; // added for external T input

  IMAX = bp->tnc[0];
  JMAX = bp->tnc[1];
  KMAX = bp->tnc[2];
  SB_struct *sp;

  /* declare the main CA structure that holds all the rest */
  time_t thetime;
  sigset_t sigmask, newsig;

#ifdef  VERBOSE_PROCAST
  /* only used for exhaustive debugging of procast */
  /* by dumping out all node numbers etc */
  int node_number;
  FILE *test6, *test7;
#endif /*VERBOSE_PROCAST */
#ifdef _DEBUG_MALLOC_INC
  unsigned long oldsize;
  unsigned long size1, size2;
  unsigned long hist0, hist1, hist2;

  oldsize = malloc_inuse (&hist0);
#endif /*_DEBUG_MALLOC_INC*/

  bp->n_a_sb = 0;

#ifdef  VERBOSE_PROCAST
  test6 = fopen ("procast.node", "w+");
  test7 = fopen ("procast.step", "w+");
#endif /* VERBOSE_PROCAST */

/*   test11=fopen("procast.current","w+"); */

/****************************************************************/
/* First check the stat\_flag to determine the operation mode.   */
/****************************************************************/
  switch (stat_flag) {
      /*******************************************/
    /* Restart: read in the bin block file     */
      /*******************************************/
  case RESTART_BB:
    {
      /* NOTE at this point cp and bp->ctrl are NOT the same */
      /* cp is the dummy ctrl structure for restart */
      /* whereas bp->ctrl is the structure that was saved in */
      /* the big block file */

      read_bin_blocks (bp, cp->fn_block_restart);
      bp->ctrl->restart = 1;    /* pass a flag for restart options */

      /* read any values that need to be changed */
      read_ctrl_vals (cp->fn_ctrl, bp->ctrl);
      read_matprop_vals (bp->ctrl, bp);
      read_geoplus_vals (bp->ctrl, bp);
      setup_temp_function (bp->ctrl, bp);
      setup_mould_src_function (bp->ctrl, bp, &(bp->mprops.gasprops));

      for(i=0;i<bp->mprops.ele_1;i++){
         setup_mould_src_function (bp->ctrl, bp, &(bp->mprops.alloyprops[i]));
      }

      /** \todo  get rgbp out of control structure  -- general */
      /*should be ONLY used in output image routines */
      init_output_img (bp->ctrl->rgbp);

      /* apply any re-initialization in order to change options */
      /* NOT ALL OPTIONS MAY BE MEANINGFULLY CHANGED until      */
      /* a routine is written to set up the changes needed.     */
      restart_change_options (bp->ctrl, bp);
      /* reset the real time counter */
      bp->starttime = time (0);
      bp->realtime = 0;

    }

#ifndef MIN_OUT
    if (bp->ctrl->excel)
      write_bb_excel (RESTART_CALL, bp);
#endif /* MIN_OUT */
    break;
/*******************************************/
/* end of RESTART_BB                       */
/*******************************************/

/****************************************************************/
/* Initialise all: move most of this into subroutines later...  */
/****************************************************************/
  case INIT_BB:                /* first call to ca_solid, do all initilisation */
    /* not restarting, the initial control structure may be used */
    /* as the real control structure */
    bp->ctrl = cp;
    if (init_bb (bp, cp) != 0) {
      fprintf (stderr, "exiting due to init_bb failure");
      exit (2);
    }
    
    /* fallthrough on this case, PROCAST initialized the big block */
    /* when generating the cells */
  case PROCAST_INIT_BB:
    bp->starttime = time (0);
    bp->realtime = 0;
    fprintf (stderr, "Starting to create and initialise subblocks \n");
    fprintf (stderr, "ntsb %d nsb: %d, %d, %d\n\n", bp->ntsb, bp->nsb[0], bp->nsb[1], bp->nsb[2]);
    /* loop through all required subblock for now creating them */
    fprintf (stderr, "ca_solid: B...\n");
    setup_temp_function (cp, bp);

    for(i=0;i<bp->mprops.ele_1;i++){
       setup_mould_src_function (bp->ctrl, bp, &(bp->mprops.alloyprops[i]));
    }

    setup_mould_src_function (cp, bp, &(bp->mprops.gasprops));

    /* macromodel from file --set up procedure once per run */
    if (cp->fgrid_input == TRUE) {
      /*******************************************/
      /* If the FE grid info is used, then       */
      /* Initialise the R direction interpolation */
      /* factors -- only once per run            */
      /* assuming that the R grid does not change */
      /*******************************************/
      fprintf (stderr, "Reading in fgrid structure data from %s ... \n", cp->fn_fgrid);
      /* read the first and second finit grids to set up the interpolation */
      fg_read (cp->fn_fgrid, bp->fg, FG_FIRST_READ);
      bp->fg->state = cp->fg_tr;
      fg_read (cp->fn_fgrid, bp->fg_next, FG_TRANS_READ);
      bp->fg_next->state = cp->fg_tr;
      /* set up the step to read the next file */
      bp->fg->s_start = bp->step;
      fg_find_steps (bp, bp->fg);
      bp->fg_next->s_start = bp->fg->s_next;
      fg_find_steps (bp, bp->fg_next);

      bp->fg->wfact = 0;

    }

    /* create the offset for the subblock neighbours */
    init_sb_neigh (bp, WRAP);

         /*****perform the temp and pres***/
         /*****allocation for each cell*************************/
    if (cp->procast) {
      for (h = 0; h < bp->total_cell_number; h++) {
        cell_temp_calc_procast (bp, h);
        if (bp->ctrl->flow_on == TRUE) {
          find_cell_pressure (bp, h);
        }
      }
      bp->first_micro = 0;
    }
         /**********************************************/

    if (cp->physica){
	if(bp->phy_tran==0) phy_temp_calc(bp);
    }
/*
    if (cp->ffheattran){
	calc_temp(bp);
    }
*/
    start_temp = bp->mprops.Tstart;
    if (FALSE) {                /** \todo  test subblocks for start condition -- multiblock */
      for (i = 0; i < bp->ntsb; i++) {
        min_temp = bp->sb[i]->Tvals.Tmin;
        if ((min_temp <= start_temp) && (bp->sb[i]->open == SB_NEW)) {
          if (open_sb (bp, i) != 0) {
            fprintf (stderr, "exiting due to open_sb failure");
            exit (2);
          }
          /* Check if there are nucleation areas, and set them! */
          for (j = 0; j < bp->nprops.nareanuc; j++) {
            if (bp->nprops.nap[j].sbnum == i)
              sb_nuc_area (bp, bp->nprops.nap[j].sbnum, &(bp->nprops.nap[j]));
          }
        }                       /*end start_temp comparison */
      }                         /*end loop through subblocks */

    } else {                    /* Non-concast: just open them all for now */
      for (i = 0; i < bp->ntsb; i++) {
        if (open_sb (bp, i) != 0) {
          fprintf (stderr, "exiting due to open_sb failure");
          exit (2);
        }
      }

      /* Check if there are nucleation areas, and set them! */
      for (i = 0; i < bp->nprops.nareanuc; i++) {
        sb_nuc_area (bp, bp->nprops.nap[i].sbnum, &(bp->nprops.nap[i]));
      }
    }                           /* end of open initial subblocks */

#ifndef MIN_OUT
    if (cp->excel)
      write_bb_excel (FIRST_CALL, bp);
#endif /* MIN_OUT */

    fprintf (stderr, "CA_SOLID: Finished creating&initialising subblocks. \n");
    break;

/*******************************************/
/* End of INIT_BB                          */
/*******************************************/

/****************************************************************/
/* CALC_BB: Perform one timestep calculation.                   */
/****************************************************************/
  case CALC_BB:                /* perform a series of sub-timesteps */
    cp = bp->ctrl;              /**AARGH  \todo  straighten this out! -- general */
       /************************************************/
    /* CASE SOLO mode:                              */
    /*    If in SOLO mode, loop through the entire  */
    /*    calc. without returning to ca\_wrapper.    */
    /* CASE CAP mode:                               */
    /*    If in CAP mode, loop through sub-steps    */
    /*    and then return for another cap timestep  */
       /************************************************/
    /* loop through time calling the ca routine...  */
       /************************************************/

    fprintf (stderr, "CA_SOLID: starting subblock calculation \n");
    fprintf (stderr, "CA_SOLID: sim,fin,delt,nsteps,bstep: %f, %f, %f, %i, %i\n", bp->sim_time, bp->finish_time, bp->delt, bp->nsteps,  bp->step);

    if(cp->physica) {
	if(bp->phy_tran==0){    
            float x[3];
	    x[0]=bp->orig_bb[0];
	    x[1]=bp->orig_bb[1]-bp->phy_growvelocity*bp->sim_time;
	    x[2]=bp->orig_bb[2];
	    fprintf(stderr, "the origin for T interpolate is %f   %f  %f \n", x[0], x[1], x[2]);
	}
    }

       /********at the begining of each macro time step reset the counter**/

    bp->first_micro = 0;
    step = 0;
    start_temp = bp->mprops.Tstart;

    /* loop through all required subblocks performing required steps */
    while (!done) {
         /**********************************************************************/
      /* mask the USR1 and USR2 signal, so that requested out put will be consistant */
      /* note you can still force ending with TERM signal ...               */
         /**********************************************************************/
      /* the ALL_SIGS preprocesser definition will prevent the signal masking */
      /* this is useful for debugging */

#ifndef ALL_SIGS
      sigaddset (&newsig, SIGUSR1);
      sigaddset (&newsig, SIGUSR2);
      sigprocmask (SIG_BLOCK, &newsig, &sigmask);
#endif
         /**********************************************************************/

             /*****perform for the current step the temp and pres***/
             /*****allocation for each cell*************************/
      if (cp->procast) {
        for (h = 0; h < bp->total_cell_number; h++) {
          cell_temp_calc_procast (bp, h);
          if (bp->ctrl->flow_on == TRUE) {
            find_cell_pressure (bp, h);
          }
        }
      }

     
      
      
      
      /**********************************************/

      bp->sim_time += bp->delt;
      thetime = time (0);
      bp->realtime = (int) (thetime) - (int) (bp->starttime);
      fsprintflg = 0;
      new_fg = FALSE;
      step++;
      bp->micro_step = step;
      /**********************************************/
      /* macromodel from file: set up per macro timestep */
      /* -- first test if this microstep is a new macrostep */
      /* Read in the temperature interpolation file */
      /* once per timestep unless steady state      */
       /**********************************************/
      if (cp->fgrid_input == TRUE) {
        if (bp->fg->state == TRANSIENT) {
          /* flip the next fg into the current one */
          /* get rid of the old fg data */
          if (bp->step == bp->fg->s_next) {
            new_fg = TRUE;
            free_fg (bp->fg);
            memcpy (bp->fg, bp->fg_next, sizeof (FGrid_str));
            fg_read (cp->fn_fgrid, bp->fg_next, FG_TRANS_READ);
            bp->fg_next->s_start = bp->fg->s_next;
            bp->fg_next->state = bp->fg->state;
            fg_find_steps (bp, bp->fg_next);
            bp->fg->wfact = 0;
          } else {
            /* update the linear weight factor for time interpolation */
            bp->fg->wfact += bp->fg->wf_add;
          }
        }
      }
      bp->n_a_sb = 0;
      bp->fs_active = 0;
      for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {
        /* loop trhough sb's checking if open */
        if (bp->sb[sbnum]->open == FALSE) {
          /* check if sb should be opened */
          /** \todo  figure out how to set this for the finite grid method -- general*/
          min_temp = bp->sb[sbnum]->Tvals.Tmin;
          if (min_temp <= start_temp) {
            if (open_sb (bp, sbnum) != 0) {
              fprintf (stderr, "CA_SOLID: Exiting due to open_sb %i failure\n", sbnum);
              exit (2);
            }
            /* open sb */
            /* Check if there are nucleation areas, and set them! */
            for (j = 0; j < bp->nprops.nareanuc; j++) {
              if (bp->nprops.nap[j].sbnum == i)
                sb_nuc_area (bp, bp->nprops.nap[j].sbnum, &(bp->nprops.nap[j]));
            }
          }                     /*end start_temp comparison */
        }

        /* end if sb not open */
 /************************/
        /*  Call the |calc_sb|  */
 /************************/
        else if (bp->sb[sbnum]->open == TRUE) { /* if open, calc it */
          /* if a new fg has been read in, update the base temeprature arrays */
          /* don't need to test fgrid_input cause new_fg is always false otherwise */
          if (new_fg) {
            fg_temp_upd (bp, sbnum);
          }

	  if (cp->physica){
	   if(bp->phy_tran==0) phy_temp_calc(bp);
	   else if(bp->phy_tran==1) {
	     bp->orig_bb[1]-=bp->sim_time*bp->phy_growvelocity;
	     ca_cord_gen(bp,bp->orig_bb);

	     if(bp->sim_time > (bp->fcount-1)*bp->macro_dt){
	       datarestore(bp);
	       bp->fcount++;
	       bp->fcountin++;
	       phy_readin(bp, bp->fcountin);
//	       fprintf(stderr, "readin number: %d \n", bp->fcountin);
	     }

	     phy_interp_slide(bp,bp->orig_bb,bp->phy_Told,bp->ca_Told);
	     phy_interp_slide(bp,bp->orig_bb,bp->phy_T,bp->ca_T);
	     bp->dtimenow = bp->sim_time-(bp->fcount-2)* bp->macro_dt;
	     phy_interp_Tnow(bp, bp->dtimenow, bp->macro_dt);

	     /*----transfer T[][][] to T[] for temperature reading---*/
	     vectorout(bp->ca_Tnow, bp->ca_cell_Tar, 1, IMAX, 1, JMAX, 1, KMAX );
	   }
	  }

	  if(cp->extinput == 1){

            timenow = bp->ext_times[0]+bp->sim_time;
	    if(timenow >= bp->ext_times[bp->ext_count]){
              ext_dataexchange(bp);
	      bp->ext_count ++;
	      ext_readin(cp, bp, bp->ext_count);
	    }
	    ext_temp_interp(bp, timenow, bp->ext_count);
	  }
          /* temperature calculation for each time step*/
	  /* heattran can be selcected only!!*/
	  
	  /* Initialize fluid flow conditions if flow starts delayed*/
	  
	  if(cp->ffheattran || cp->fluidflow && (cp->fstarttime < bp->sim_time)){
            
            if( bp->ffstartflag == 0) {
 	      fprintf(stderr, "Initialize fluid flow and heat transfer calculation... \n");
	      initvelocity(bp);
              /* time step check for fluid flow, ...
              if ( ff_timestep(bp) ) {
                fprintf(stderr, "ERROR: Time step is too big, exiting....\n");
                return 1;
              } */

              if (bp->tonecyc <= 0)  boundary(bp);
              else {
              boundary_cyc(bp);
              fprintf(stderr, " cycling Velociyt BCs applied, Cycle Time : %.2f s \n", bp->tonecyc);
              }

              fprintf(stderr, " Finished Velocity&Temperature initialisaztion !\n");

	      bp->ffstartflag = 1;
	    }
	  
	    /* Copy the [] to [][][] for fluid calculation*/
	    
	    
	    vectorin(bp->ca_cell_ftry, bp->sb[sbnum]->c_fs, 1, IMAX, 1, JMAX, 1, KMAX );   //fs
            vectorin(bp->ca_cell_ce, bp->ftmp_five, 0, IMAX+1, 0, JMAX+1, 0, KMAX+1 );  //ce

	    
            for (i=1; i<=IMAX; i++){
	      for (j=1; j<=JMAX; j++){
	        for (k=1; k<=KMAX; k++){
		  bp->ca_cell_ftry[i][j][k] = 1- bp->ca_cell_ftry[i][j][k];
		}
	      }
	    }
	    
	    if(cp->ffheattran)   {
		    calc_temp(bp);
//		    calc_temp_enthalpy(bp);
		    if(!cp->fluidflow) boundary(bp);
	    }
	    
//
	    if(cp->fluidflow) {
	     if((bp->micro_step-1)% cp->ffstepgap == 0 ){
	
//	       calc_fgh_imptest(bp);
   	       calc_fgh(bp);
	       poisson_rhs(bp);
	       bp->inneriter = ppe_iccgtest(bp);
	       newvelocityp2(bp);

	//       bp->inneriter  = ppe_sor(bp);
	 //      newvelocity(bp);
	     }
	     if (bp->tonecyc <= 0)  boundary(bp);
	     else boundary_cyc(bp);
	    
	    }
	  }
	  
          calc_sb (bp, sbnum);

	  /*calculate max undercooling map*/
	  if(cp->maxucmap ==1) sb_max_uc(bp, sbnum);
	  
#ifdef TEST_PHY	
	  fprintf(stderr, "cell temperature compare: %f  %f \n", bp->ca_cell_Tar[0],bp->sb[sbnum]->c_temp[0]);
#endif
          /* track the fraction solid in active sb's */
          bp->n_a_sb++;
          bp->fs_active += bp->sb[sbnum]->Tvals.fsavg;

        }
             /************************/
        /* end calc sb          */
             /************************/
      }                         /* end loop through subblocks */

      if (bp->n_a_sb > 0)
        bp->fs_active /= bp->n_a_sb;
      else
        bp->fs_active = 0;

      /* if no sub block is open return to the higher level routine */
      if (bp->n_a_sb == 0) {

               /************************************************/
        /* close all open input/output files and exit   */
               /************************************************/
#ifdef  VERBOSE_PROCAST
        fclose (test6);
        fclose (test7);
#endif /*VERBOSE_PROCAST */
        return (1);
      }
         /**********************************************************************/
      /* unmask the USR1,USR2 signal                                             */
         /**********************************************************************/
#ifndef ALL_SIGS
      sigprocmask (SIG_UNBLOCK, &newsig, &sigmask);
#endif
         /**********************************************************************/
      /* if the USR1 signal is received, it waits until here to process it. */
         /**********************************************************************/

          /*******************************************/
      /* Check for end conditions                */
          /*******************************************/
      if (cp->solo && bp->auto_fin) {
        if (bp->sim_time > bp->autofin_time) {  /* failsafe stop of programme... */
          done = TRUE;
          fprintf (stderr, "CA_SOLID: Reached autofin_time [%g].\n", bp->autofin_time);
        }
      } else if (bp->sim_time > bp->finish_time) {      /* failsafe stop of programme... */
        done = TRUE;
        fprintf (stderr, "CA_SOLID: Reached finish_time.\n");
      }
      /*stop at specified fraction solid */
      /*needs update for MULTIBLOCK */
      if (bp->sb[0]->Tvals.fsavg >= cp->fs_finish) {
        done = TRUE;
        if ((cp->procast) && (bp->last_micro == 0)) {
          bp->last_micro = 1;
        }
        fprintf (stderr, "CA_SOLID: Reached fs_finish fraction solid %g.\n", bp->sb[0]->Tvals.fsavg);
      }
#ifdef REALTIME_LIMIT
      /* stop at limit of real time */
      if (bp->realtime > bp->realtime_limit) {
        done = TRUE;
        fprintf (stderr, "CA_SOLID: Reached elapsed realtime limit %i.\n", bp->realtime);
      }
#endif

          /*******************************************************/
      /* check if fraction solid has reached a print stage   */
          /*******************************************************/
      for (j = 0; j < cp->nfsprint; j++) {
        if ((bp->sb[0]->Tvals.fsavg >= cp->fsprint[j]) && (fspchk[j] == 0)) {
          fspchk[j] = 1;
          fsprintflg = 1;
          fprintf (stderr, "CA_SOLID: Reached fsprint fraction solid %g.\n", bp->sb[0]->Tvals.fsavg);
        }
      }                         /* end check fsprint */
      if (cp->time_dump) {
        tcprintflg = 0;
        tsprintflg = 0;
        tbprintflg = 0;
        if (bp->sim_time >= ((bp->scr_dump_num + 1) * (cp->time_unit * cp->scr_dmp_freq))) {
          bp->scr_dump_num++;
          tcprintflg = 1;
        }
        if (bp->sim_time >= ((bp->slice_dump_num + 1) * (cp->time_unit * cp->slice_dmp_freq))) {
          bp->slice_dump_num++;
          tsprintflg = 1;
        }
        if (bp->sim_time >= ((bp->blk_dump_num + 1) * (cp->time_unit * cp->blk_dmp_freq))) {
          bp->blk_dump_num++;
          tbprintflg = 1;
        }
      }

          /******************************************/
      /* check if signal USR2 has been received */
          /******************************************/
      if (jflg == JFLG_WRITEBIN) {
          /************************************************/
        /* write the entire bb to a file              */
          /************************************************/
        fprintf (stderr, "writing binary bigblock, bstep %i\n", bp->step);
        write_bin_blocks (bp);
        jflg = 0;
      }

          /*******************************************/
      /* perform screen print for current step   */
          /*******************************************/
      /* protect from % operation if zero */
      if (cp->scr_dmp_freq != 0) {
        if ((bp->step % cp->scr_dmp_freq == 0 && cp->time_dump == 0) || (fsprintflg == 1) || (tcprintflg == 1)) {
          step_screen_out (bp, step);
        }                       /* end of screen dump printout */
      }

      /* protect from % operation if zero */
      if (cp->blk_dmp_freq != 0) {
        if ((bp->step % cp->blk_dmp_freq == 0 && cp->time_dump == 0) || (fsprintflg == 1) || (tbprintflg == 1)) {
          step_blk_out (bp, step);
        }                       /* end of block dump */
      }
      
      /* Tecpolt output by LY*/
       if (cp->plt_dmp_freq != 0) {
         if ((bp->step % cp->plt_dmp_freq == 0 && cp->time_dump == 0)) {
          plt_op_fsv (bp, sbnum);
	 }
       }
 
      /* Paraview output controller */
       if (cp->vtk_dmp_freq != 0) {
         if ((bp->step % cp->vtk_dmp_freq == 0 && cp->time_dump == 0)) 
           vtk_para_op (bp, sbnum);
       }
			  
	 
      /* for testing the single point output*/	 
	  
       if (cp->point_dmp_freq != 0) {
         if ((bp->step % cp->point_dmp_freq == 0 && cp->time_dump == 0)) {
            for (ii=0; ii<cp->nptrack; ii++){
	       plt_op_pointinfo (bp, sbnum, cp->ptrack[ii][0],cp->ptrack[ii][1], cp->ptrack[ii][2]);
	    }		 
          }
         }                       /* end of Tecplot dump printout */
      
       if (cp->line_dmp_freq != 0) {
         if ((bp->step % cp->line_dmp_freq == 0 && cp->time_dump == 0)) {
            for (ii=0; ii < cp->nlntrack; ii++){
	       plt_op_lineinfo (bp, sbnum, cp->lntrack, ii);
	    }		 
          }
         }                       /* end of Tecplot dump printout */
       

      /* not minimum output mode */
#ifndef MIN_OUT
          /************************************************/
      /* perform any file output for the current step */
          /************************************************/
      /* protect from % operation if zero */
      if (cp->slice_dmp_freq != 0) {
        if (((cp->nsbslice > 0)
             || (cp->nbbslice > 0))
            && bp->step != 0 && (((bp->step % cp->slice_dmp_freq == 0) && (cp->time_dump == 0))
                                 || (fsprintflg == 1) || (tsprintflg == 1))) {
#ifdef IMAGEOUT
          step_file_out (bp, step);
#else
          fprintf (stderr, "CA_SOLID: image output deactivated!\n");
#endif
        }                       /* end of current step file output */
      }
#endif /*MIN_OUT */

#ifdef VERBOSE_PROCAST
/* procast feature, not sure why needed? */
      for (j = 0; j < bp->total_cell_number; j++) {
        if (bp->cell_element_array[j] >= 0) {
          node_number = bp->cell_node_array[j];
          fprintf (test6, "%d \t %d \t %.12f \t %.12f \t %f \t %f \n", j, node_number, bp->cell_temp_procast[j] - 273.16,
                   bp->cell_temp_change_procast[j], t0[node_number] - 273.16, t1[node_number] - 273.16);
          break;
        }
      }
#endif /* VERBOSE_PROCAST */
#ifndef MIN_OUT
      if ((cp->excel) && ((step % cp->scr_dmp_freq == 0) || (fsprintflg == 1)))
        write_bb_excel (STEP_CALL, bp);
#endif /*MIN_OUT */

#ifdef MANYDUMPS
          /********************************************/
      /* Debug mode to                            */
      /* zero-in on interesting behaviour         */
      /* start producing output at defined higher */
      /* frequency                                */
          /********************************************/
      if (bp->step == MANY_DUMP_START) {
        olddmp = cp->slice_dmp_freq;
        oldscrdmp = cp->scr_dmp_freq;
        cp->slice_dmp_freq = MANY_FREQ;
        cp->scr_dmp_freq = MANY_FREQ;
      }
      if (bp->step == MANY_DUMP_STOP) {
        cp->slice_dmp_freq = olddmp;
        cp->scr_dmp_freq = oldscrdmp;
      }
#endif /*MANYDUMPS*/
  /*********************************************************************/
      /* apply any requested real-time change */
      if (signal_change_freq != 0 ){
          printf("%s: Output step change requested.\n",__func__);
          printf("New step: singal_change_freq= %i\n",signal_change_freq);
          cp->blk_dmp_freq = signal_change_freq;
          signal_change_freq=0;
      }
        bp->first_micro += 1;
        bp->step += 1;

#ifdef  VERBOSE_PROCAST
  /*********output time step information******/
      fprintf (test7, "step=%d \t bp->step=%d \t bp->first_micro=%d \t bp->sim_time=%f \t bp->finish_time=%f \t bp->nsteps=%d \n",
               step, bp->step, bp->first_micro, bp->sim_time, bp->finish_time, bp->nsteps);
  /***********************************************/
#endif /*VERBOSE_PROCAST */
    }
    break;

/*******************************************/
/* End of CALC_BB                          */
/*******************************************/

/****************************************************************/
/* FINISH_BB: Print out final results.                          */
/****************************************************************/
  case FINISH_BB:              /* finished all calc, preform final output and free */

#ifdef MIN_OUT
#else /*not MIN_OUT */
          /************************************************/
    /* write the entire bb to a file              */
          /************************************************/
#ifdef  VERBOSE_PROCAST
    fclose (test6);
    fclose (test7);
#endif /*VERBOSE_PROCAST */
    /* return(1); *//*??? not sure why this is here */
    fprintf (stderr, "writing binary bigblock, bstep %i\n", bp->step);
    write_bin_blocks (bp);
    
    for (sbnum = 1; sbnum < (bp->ntsb+1); sbnum++) {
      vtk_para_op (bp, sbnum);
    }
    step_file_out (bp, step);

    if (cp->excel) {
      write_bb_excel (LAST_CALL, bp);
      write_bb_excel_sum (LAST_CALL, bp);
    }
#endif /*not MIN_OUT */
    fprintf (stderr, "DFS Warnings: %i DFS Capped: %i\n", bp->dfs_warn, bp->dfs_cap);
/**********************/
/* free data pointers */
/**********************/
#ifdef TEST_CLOSE
    close_sb (bp, 0);
#endif /*TEST_CLOSE */
#ifdef _DEBUG_MALLOC_INC
    size1 = malloc_inuse (&hist1);
    fprintf (stderr, "ca_solid list before free_bb %i %i\n", hist0, hist1);
    malloc_list (2, bp->ctrl->hist[0], hist1);
#endif       /*_DEBUG_MALLOC_INC*/

    fprintf (stderr, "Freeing the Big Block\n");

    if (cp->fgrid_input) {
      fg_read (cp->fn_fgrid, bp->fg, FG_CLEANUP);
    }

    free_bb (bp);
#ifdef _DEBUG_MALLOC_INC
    size1 = malloc_inuse (&hist1);
    fprintf (stderr, "ca_solid list after free_bb %i %i\n", hist0, hist1);
    malloc_list (2, bp->ctrl->hist[0], hist1);
#endif       /*_DEBUG_MALLOC_INC*/
    break;

  default:
    fprintf (stderr, "ERROR:ca_solid: stat_flag value [%d] undefined.\n", stat_flag);
    break;
  }                             /* end of stat_flag switch */

/************************************************/
/* close all open input/output files and exit   */
/************************************************/
#ifdef _DEBUG_MALLOC_INC
  fprintf (stderr, "ca_solid list %i %i\n", hist0, hist1);
  malloc_list (2, hist0, hist1);
#endif /*_DEBUG_MALLOC_INC*/
  return (0);
}                               /* end of ca_solid */

/* rcs id subroutine.*/
/*Little subroutine to include the |rcs Id| in the object code.*/
/*This can also be called to print or include the Id in a file.*/

char const *rcs_id_ca_solid_c ()
{
  static char const rcsid[] = "$Id: ca_solid.c 1491 2010-07-30 00:32:44Z  $";

  return (rcsid);
}

/*
RCS Log:$Log$
RCS Log:Revision 11.3  2006/11/09 20:23:51  rcatwood
RCS Log:Tried to fix some things for doxygen
RCS Log:
RCS Log:Revision 11.2  2006/11/09 13:39:40  rcatwood
RCS Log:Merged the update for ca_procast version for procast 2006.0_beta
RCS Log:
RCS Log:Revision 11.1.12.1  2006/11/06 20:18:17  rcatwood
RCS Log:Improvements for coupled version
RCS Log:Signal flag to reread output frequency from ca_newstep.in
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:39  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5.2.1  2006/01/11 12:12:19  rcatwood
RCS Log:Altered to use the new solute properties structure arrays
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:52  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:46  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.7.8.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.8.2.3  2004/06/18 18:26:36  rcatwood
RCS Log:Added option for output freqency at uniform sim-time instead of steps
RCS Log:
RCS Log:Revision 9.8.2.2  2004/03/04 11:29:24  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.8.2.1  2004/02/05 12:52:24  rcatwood
RCS Log:Fixed a restart problem with reinitializing the control structure
RCS Log:
RCS Log:Revision 9.8  2003/12/11 15:56:03  rcatwood
RCS Log:Changed block output : uses .blk file and separate frequecy step
RCS Log:
RCS Log:Revision 9.7  2003/11/05 17:00:38  rcatwood
RCS Log:Combined the ca_solid and ca_solid_procast into a single version (ca_solid.c)
RCS Log:this required some changes to wrapper and other files
RCS Log:
RCS Log:Revision 9.4  2003/10/27 20:01:11  rcatwood
RCS Log:Harmonized header file cpp protection
RCS Log:Fixed filename bug for restart
RCS Log:
RCS Log:Revision 9.3  2003/10/23 16:32:39  rcatwood
RCS Log:chagned signal result to allow writing of the final files on USR1
RCS Log:
RCS Log:Revision 9.2  2003/09/16 11:59:14  rcatwood
RCS Log:Improved micro/macro interpolation
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:34  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 1.1.6.6  2003/08/14 14:17:57  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 1.1.6.5  2003/03/21 18:16:41  rcatwood
RCS Log:added flat (xy-plane) mulitblock geometry
RCS Log:Tried to improve Makefile for ca_read
RCS Log:
RCS Log:Revision 1.1.6.4  2003/02/11 10:56:43  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 1.1.6.3  2003/01/22 16:53:42  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 1.1.6.2  2003/01/17 16:09:29  rcatwood
RCS Log:Before changing all CA_FLOAT to CA_CA_FLOAT
RCS Log:
RCS Log:Revision 1.1.6.1  2003/01/15 19:16:22  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 1.1.4.3  2003/01/14 16:22:25  rcatwood
RCS Log:Removed many lint warnings from sb_decentered_step
RCS Log:Added signal function to ca_procast
RCS Log:Removed some unused files
RCS Log:
RCS Log:Revision 1.1.4.2  2003/01/14 12:59:39  rcatwood
RCS Log:Merged procast version which compiles successfullly
RCS Log:Not checked for running yet
RCS Log:Including WEI decentered square but not temperature routine links
RCS Log:
RCS Log:Revision 1.1.4.1  2002/11/06 17:27:45  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 8.5  2002/11/04 14:34:43  rcatwood
RCS Log:Removed some lint
RCS Log:
RCS Log:Revision 8.4  2002/11/04 13:35:41  rcatwood
RCS Log:Finished restart re-read reorgainization
RCS Log:
RCS Log:Revision 8.3  2002/10/25 12:17:00  rcatwood
RCS Log:Merged cygwin portable version for Windows machines
RCS Log:
RCS Log:Revision 8.2  2002/10/21 12:04:26  rcatwood
RCS Log:Fixed bug in section for opening sb based on minimum temp
RCS Log:due to X/Y switch
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:01  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.13  2002/10/17 16:52:37  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.12.8.10  2002/10/17 16:03:44  rcatwood
RCS Log:Fixed a subtle problem concerning the amount of gas in the pores
RCS Log:This required adding a lot of debugging #ifdef CHECK_GAS lines
RCS Log:
RCS Log:Revision 7.12.8.9  2002/10/04 16:21:27  rcatwood
RCS Log:adjusted the reporting of fraction solid, to allow for multi block
RCS Log:
RCS Log:Revision 7.12.8.8  2002/10/03 15:54:51  rcatwood
RCS Log:Quick fix: restart to read ctrl file and replace values
RCS Log:Fixed file name 'ca_restart.in'
RCS Log:
RCS Log:Revision 7.12.8.7  2002/10/02 15:20:06  rcatwood
RCS Log:added mask of USR2 signal
RCS Log:
RCS Log:Revision 7.12.8.5  2002/09/19 18:34:57  rcatwood
RCS Log:First working restart version using bin-block file
RCS Log:
RCS Log:Revision 7.12.8.4  2002/09/19 11:25:42  rcatwood
RCS Log:added control to print out at a specified fraction-solid
RCS Log:
RCS Log:Revision 7.12.8.3  2002/09/18 12:36:15  rcatwood
RCS Log:Split step output function into a subroutine (called from ca_solid)
RCS Log:Removed some lint errors.
RCS Log:
RCS Log:Revision 7.12.8.2  2002/09/17 20:51:07  rcatwood
RCS Log:Rearranged so that read/write leave the memory clean
RCS Log:Added signal 16 (SIGUSR1) catching for trapping after finishing the current timestep.
RCS Log:
RCS Log:Revision 1.1.4.1  2002/09/03 13:31:59  rcatwood
RCS Log:Merged with reorganized allocation routines, and adjusted nucleation to compartmentalize
RCS Log:the grain information assignment.
RCS Log:
RCS Log:Revision 1.1.2.1  2002/07/08 17:26:12  rcatwood
RCS Log:added test routine for write blocks to cvs repository
RCS Log:
RCS Log:Revision 7.11  2002/05/23 17:28:01  rcatwood
RCS Log:Used dbmalloc to eliminate all memory leaks!
RCS Log:Started close-sb routine to improve multiblock
RCS Log:
RCS Log:Revision 7.10  2002/02/14 13:15:01  rcatwood
RCS Log:Added write_block option instead of cpp definintion.
RCS Log:
RCS Log:Revision 7.9  2002/01/28 14:00:01  rcatwood
RCS Log:After thesis
RCS Log:
RCS Log:Revision 7.8  2001/07/20 13:30:53  rcatwood
RCS Log:added realtime screen output, improved grain nuc histogram
RCS Log:
RCS Log:Revision 7.7  2001/06/26 15:55:08  rcatwood
RCS Log:fix xegfault in ca_solid
RCS Log:
RCS Log:Revision 7.6  2001/06/19 13:33:57  rcatwood
RCS Log:added pore_write block choice
RCS Log:
RCS Log:Revision 7.5  2001/06/12 14:30:01  rcatwood
RCS Log:more no-pore errors
RCS Log:
RCS Log:Revision 7.4  2001/06/12 14:19:26  rcatwood
RCS Log:fixed bug if no pore and write_blocks = seg fault.
RCS Log:
RCS Log:Revision 7.3  2001/01/30 12:49:08  rcatwood
RCS Log:Version after NewYears, after NLM and before moving to CVS source ctrl
RCS Log:
RCS Log:Revision 7.2  2000/12/14 16:08:03  rcatwood
RCS Log:fixed up heatfolw, tctrace
RCS Log:
RCS Log:Revision 7.1  2000/11/18 15:40:01  rcatwood
RCS Log:fixed outpput
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.2  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.1  2000/10/16 10:37:22  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:34  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.4  2000/07/11 16:32:13  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.3  2000/06/27 11:41:19  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.2  2000/06/06 18:43:04  rcatwood
RCS Log:Changed r start and growth of pores -- pore radius equilibrium
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.6  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.5  2000/05/02 14:21:20  rcatwood
RCS Log:Migrated back from cweb to plain c
RCS Log:
RCS Log:Revision 5.4  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.4  2000/03/27 17:06:27  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.3  2000/03/27 11:21:22  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.3  2000/03/27 11:21:19  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.2  2000/03/24 19:33:30  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.2  2000/03/24 19:19:49  rcatwood
RCS Log:Particle dependant growth fixed
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.2  2000/03/02 13:09:59  rcatwood
RCS Log:Fixed copy-mat bug.
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.9  2000/02/15 15:29:11  rcatwood
RCS Log:Version after McWasp submitted
RCS Log:
RCS Log:Revision 4.7  2000/01/20 17:41:17  rcatwood
RCS Log:Pore bug fix. Bindump of fs, ss, diff coeff
RCS Log:
RCS Log:Revision 1.1.1.2  2000/02/22 18:21:33  rcatwood
RCS Log:Robert modified to use better interpolation
RCS Log:
RCS Log:Revision 1.1.1.1  2000/02/22 14:54:31  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 1.1  2000/02/22 14:52:19  rcatwood
RCS Log:Initial revision
RCS Log:
RCS Log:Revision 4.6  2000/01/13 15:32:29  rcatwood
RCS Log:Binary dump of grain and temperature,Quick temp. image.
RCS Log:
RCS Log:Revision 4.5  2000/01/05 12:47:29  rcatwood
RCS Log:Files as of New year 2000
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.3  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.2  1999/12/16 19:25:02  rcatwood
RCS Log:Alloy and gas working.
RCS Log:Pores working also.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.3  1999/12/16 13:01:54  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:54:33  rcatwood
RCS Log:Added rcs id and log info.
RCS Log:
*/
