#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "machine.h"
#include "blocks.h"
#include "ca_matrix.h"
#include "props.h"
#include "curvature.h"
#include "ff_memory.h"
#include "datatrans.h"

/***************************************************/
/* Curvature calculation in 2D        
 * using Voller's method                           */
/* by LYUAN on 06/2008                             */
/***************************************************/

int curv_alloc(BB_struct *bp){

  int IMAX, JMAX, KMAX;

  IMAX = bp->tnc[0];
  JMAX = bp->tnc[1];
  KMAX = bp->tnc[2];

  bp->theta = realmemalloc(0, IMAX+1, 0, JMAX+1, 0, KMAX+1);
  bp->curvature = realmemalloc(0, IMAX+1, 0, JMAX+1, 0, KMAX+1);
  bp->Tcurv = realmemalloc(0, IMAX+1, 0, JMAX+1, 0, KMAX+1);
  bp->fl_temp = realmemalloc(0, IMAX+1, 0, JMAX+1, 0, KMAX+1);

  return 0;
}

void curv_init(BB_struct *bp){

  int i,j,k;
  int IMAX, JMAX, KMAX;

  IMAX = bp->tnc[0];
  JMAX = bp->tnc[1];
  KMAX = bp->tnc[2];
  
  
  for (i=0; i<=IMAX+1; i++){
    for (j=0; j<=JMAX+1; j++){
      for (k=0; k<=KMAX+1; k++){
      bp->theta[i][j][k] = 0;
      bp->curvature[i][j][k] = 0;
      bp->fl_temp[i][j][k] = 1;
      }
    }
  }
									   
}


int curvature_2d(BB_struct *bp, int sbnum){

  int i, j, k;
  int IMAX, JMAX, KMAX;
  CA_FLOAT sizex,sizey,sizez;
  CA_FLOAT ***curv, ***theta, ***Tcurv;
  CA_FLOAT ***fl;

  SB_struct *sp;
  CA_FLOAT dfdy, dfdx, dfdxx, dfdyy, dfdxy;
  CA_FLOAT fxu, fxc, fxd, fyl, fym, fyr;
  CA_FLOAT fxxu, fxxc, fxxd, fyyl, fyym, fyyr;
  CA_FLOAT anis, therm_anis, gibbs;
  
  curv = bp->curvature;
  theta = bp->theta;
  Tcurv = bp->Tcurv;
//  fl = bp->fl_temp;
  therm_anis = bp->mprops.therm_anis;
  gibbs = bp->mprops.gibbs_thomson;
  
  sp = bp->sb[sbnum];
  IMAX = bp->tnc[0];
  JMAX = bp->tnc[1];
  KMAX = bp->tnc[2];
  sizex = bp->size_c[0];
  sizey = bp->size_c[1];
  sizez = bp->size_c[2];

  vectorin(bp->fl_temp, bp->sb[sbnum]->c_fs, 1, IMAX, 1, JMAX, 1, KMAX );

  
  for (i=1; i<=IMAX; i++){
    for (j=1; j<=JMAX; j++){
      for (k=1; k<=KMAX; k++){
           bp->fl_temp[i][j][k] = 1.0- bp->fl_temp[i][j][k];
      }
    }
  }
	   
  fl = bp->fl_temp;
  
/* main loop for curvature, only for 0<fl<1 */

  for (i=1; i<=IMAX; i++){
    for (j=1; j<=JMAX; j++){
      for (k=1; k<=KMAX; k++){
        
	if(fl[i][j][k] >= 1 || fl[i][j][k] <=0) {
	  curv[i][j][k] = 0;
	  theta[i][j][k] = 0;
	  Tcurv[i][j][k] = 0;
	  continue;
	}
        
	fxu = (fl[i+1][j+1][k]- fl[i-1][j+1][k])/(2*sizex);
	fxc = (fl[i+1][j][k]- fl[i-1][j][k])/(2*sizex);
	fxd = (fl[i+1][j-1][k]- fl[i-1][j-1][k])/(2*sizex);
        dfdx = (CUR_ALPHA*fxu + fxc + CUR_ALPHA*fxd)/(1+2*CUR_ALPHA);

	fyr = (fl[i+1][j+1][k]- fl[i+1][j-1][k])/(2*sizey);
	fym = (fl[i][j+1][k]- fl[i][j-1][k])/(2*sizey);
	fyl = (fl[i-1][j+1][k]- fl[i-1][j-1][k])/(2*sizey);
	dfdy = (CUR_ALPHA*fyr + fym + CUR_ALPHA*fyl)/(1+2*CUR_ALPHA);

	theta[i][j][k] = atan2(dfdy, dfdx);

	fxxu = (fl[i+1][j+1][k] - 2*fl[i][j+1][k] + fl[i-1][j+1][k])/(sizex*sizex);
	fxxc = (fl[i+1][j][k] - 2*fl[i][j][k] + fl[i-1][j][k])/(sizex*sizex);
	fxxd = (fl[i+1][j-1][k] - 2*fl[i][j-1][k] + fl[i-1][j-1][k])/(sizex*sizex);
	dfdxx = (CUR_ALPHA*fxxu + fxxc + CUR_ALPHA*fxxd)/(1+2*CUR_ALPHA);

	fyyl = (fl[i-1][j+1][k] - 2*fl[i-1][j][k] + fl[i-1][j-1][k])/(sizey*sizey);
	fyym = (fl[i][j+1][k] - 2*fl[i][j][k] + fl[i][j-1][k])/(sizey*sizey);
	fyyr = (fl[i+1][j+1][k] - 2*fl[i+1][j][k] + fl[i+1][j-1][k])/(sizey*sizey);
	dfdyy = (CUR_ALPHA*fyyl + fyym + CUR_ALPHA*fyyr)/(1+2*CUR_ALPHA);

	dfdxy = (fl[i+1][j+1][k]-fl[i-1][j+1][k]+fl[i+1][j-1][k]-fl[i-1][j-1][k])/(4*sizex*sizex);
	
	if( (dfdx*dfdx > 0) || (dfdy*dfdy > 0) )
	  curv[i][j][k] = (dfdy*dfdy*dfdxx- 2*dfdx*dfdy*dfdxy + dfdx*dfdx*dfdyy)/pow((dfdx*dfdx+dfdy*dfdy),1.5) ;     
	anis = gibbs*(1-therm_anis*(cos(4*theta[i][j][k])));   /* crystaline orientation alines with axis*/
	Tcurv[i][j][k] =  curv[i][j][k]*anis;
      }
    }
  }
	 
  /*transfer 3d array to 1d array*/
  vectorout(Tcurv, sp->Tunder_curv, 1, IMAX, 1, JMAX, 1, KMAX);

  return 1;
}

int curv_free(BB_struct *bp){

  int IMAX, JMAX, KMAX;

  IMAX = bp->tnc[0];
  JMAX = bp->tnc[1];
  KMAX = bp->tnc[2];

  freerealmem(bp->theta, 0, IMAX+1, 0, JMAX+1, 0, KMAX+1);
  freerealmem(bp->curvature, 0, IMAX+1, 0, JMAX+1, 0, KMAX+1);
  freerealmem(bp->Tcurv, 0, IMAX+1, 0, JMAX+1, 0, KMAX+1);
  freerealmem(bp->fl_temp, 0, IMAX+1, 0, JMAX+1, 0, KMAX+1);

  return 0;
}

/*-------------------------------------------------------------------*/
/*               2D curvature ended ....   lyuan                     */
/*------------------------------------------------------------------ */


int interface_normal (BB_struct * bp, int sbnum)
{
  SB_struct *sp;
  int i, j, k;                  /* tmp counter */
  int *nid;
  int nx, ny, nz, skip;
  CA_FLOAT *ofs, *ofs_e, *ofs_w, *ofs_n, *ofs_s, *ofs_ne, *ofs_se, *ofs_nw, *ofs_sw;
  CA_FLOAT fs_e, fs_w, fs_s, fs_n;
  CA_FLOAT *normal_x, *normal_y;

  /*float size_cell_x, size_cell_y; */

  sp = bp->sb[sbnum];

  nx = bp->nc[0];
  ny = bp->nc[1];
  nz = bp->nc[2];
  skip = 2 * (nx + 2);

  /* size_cell_x = bp->size_c[0];
     size_cell_y = bp->size_c[1]; */

  /*set local pointer */
  normal_x = sp->norm_x;
  normal_y = sp->norm_y;
  ofs = bp->ftmp_one;
  nid = sp->index;

  /*move the ptr from outside corner to inside corner */
  ofs += bp->cubeptr.flist[0][START];

  ofs_e = ofs + 1;
  ofs_w = ofs - 1;
  ofs_n = ofs + nx + 2;
  ofs_s = ofs - nx - 2;
  ofs_ne = ofs_n + 1;
  ofs_nw = ofs_n - 1;
  ofs_se = ofs_s + 1;
  ofs_sw = ofs_s - 1;

   /***********************************************/
  /* Beginning of main loop(s)                   */
   /***********************************************/

  for (k = 0; k < nz; k++) {
    for (j = 0; j < ny; j++) {
      for (i = 0; i < nx; i++) {

        fs_e = 0.25 * (*ofs_se + *ofs_ne + 2 * (*ofs_e));
        fs_w = 0.25 * (*ofs_sw + *ofs_nw + 2 * (*ofs_w));
        fs_s = 0.25 * (*ofs_sw + *ofs_se + 2 * (*ofs_s));
        fs_n = 0.25 * (*ofs_ne + *ofs_nw + 2 * (*ofs_n));

        if (*nid == 2) {
          *normal_x = 0.5 * (fs_w - fs_e);
          *normal_y = 0.5 * (fs_s - fs_n);
        } else {
          *normal_x = 0.;
          *normal_y = 0.;
        }
        nid++;
        normal_x++;
        normal_y++;
        ofs++;
        ofs_e++;
        ofs_w++;
        ofs_n++;
        ofs_s++;
        ofs_ne++;
        ofs_nw++;
        ofs_se++;
        ofs_sw++;
      }                         /* end of I loop */
      ofs += 2;
      ofs_e += 2;
      ofs_w += 2;
      ofs_n += 2;
      ofs_s += 2;
      ofs_ne += 2;
      ofs_nw += 2;
      ofs_se += 2;
      ofs_sw += 2;

    }                           /* end of J loop */
    ofs += skip;
    ofs_e += skip;
    ofs_w += skip;
    ofs_n += skip;
    ofs_s += skip;
    ofs_ne += skip;
    ofs_nw += skip;
    ofs_se += skip;
    ofs_sw += skip;

  }                             /* end of K loop */
  return (1);
}

int interface_curvature (BB_struct * bp, int sbnum)
{
  SB_struct *sp;
  int i, j, k;                  /* tmp counter */
  int nx, ny, nz, skip;
  int *nid;
  CA_FLOAT *normal_x, *onx, *onx_e, *onx_w, *onx_n, *onx_s, *onx_ne, *onx_nw, *onx_se, *onx_sw;
  CA_FLOAT *normal_y, *ony, *ony_e, *ony_w, *ony_n, *ony_s, *ony_ne, *ony_nw, *ony_se, *ony_sw;
  CA_FLOAT normal, normal_e, normal_w, normal_s, normal_n, normal_ne, normal_nw, normal_se, normal_sw;
  CA_FLOAT *curv;
  CA_FLOAT k_x, k_y;

  /*float size_cell_x, size_cell_y; */

  sp = bp->sb[sbnum];
  bp->cubeptr.curr = sbnum;

  nx = bp->nc[0];
  ny = bp->nc[1];
  nz = bp->nc[2];
  skip = 2 * (nx + 2);

  normal_x = sp->norm_x;
  normal_y = sp->norm_y;

  onx = bp->ftmp_nx;
  ony = bp->ftmp_ny;

  fcopy_matrix (PAD, onx, normal_x, bp, NULL, sbnum);
  fcopy_matrix (PAD, ony, normal_y, bp, NULL, sbnum);

  onx += bp->cubeptr.flist[0][START];
  ony += bp->cubeptr.flist[0][START];

  /* size_cell_x = bp->size_c[0];
     size_cell_y = bp->size_c[1]; */

  /*set local pointer */
  onx_e = onx + 1;
  onx_w = onx - 1;
  onx_n = onx + nx + 2;
  onx_s = onx - nx - 2;
  onx_ne = onx_n + 1;
  onx_nw = onx_n - 1;
  onx_se = onx_s + 1;
  onx_sw = onx_s - 1;
  ony_e = ony + 1;
  ony_w = ony - 1;
  ony_n = ony + nx + 2;
  ony_s = ony - nx - 2;
  ony_ne = ony_n + 1;
  ony_nw = ony_n - 1;
  ony_se = ony_s + 1;
  ony_sw = ony_s - 1;

  nid = sp->index;
  curv = sp->curv;

   /***********************************************/
  /* Beginning of main loop(s)                   */
   /***********************************************/

  for (k = 0; k < nz; k++) {
    for (j = 0; j < ny; j++) {
      for (i = 0; i < nx; i++) {
        if (*nid == 2) {
          normal = sqrt (*onx * (*onx) + *ony * (*ony));
          normal_e = sqrt (*onx_e * (*onx_e) + *ony_e * (*ony_e));
          normal_w = sqrt (*onx_w * (*onx_w) + *ony_w * (*ony_w));
          normal_s = sqrt (*onx_s * (*onx_s) + *ony_s * (*ony_s));
          normal_n = sqrt (*onx_n * (*onx_n) + *ony_n * (*ony_n));
          normal_ne = sqrt (*onx_ne * (*onx_ne) + *ony_ne * (*ony_ne));
          normal_nw = sqrt (*onx_nw * (*onx_nw) + *ony_nw * (*ony_nw));
          normal_se = sqrt (*onx_se * (*onx_se) + *ony_se * (*ony_se));
          normal_sw = sqrt (*onx_sw * (*onx_sw) + *ony_sw * (*ony_sw));

          *onx = (normal > 0) ? *onx / normal : 0;
          *ony = (normal > 0) ? *ony / normal : 0;

          *onx_e = (normal_e > 0) ? *onx_e / (normal_e) : 0;
          *onx_w = (normal_w > 0) ? *onx_w / (normal_w) : 0;
          *onx_s = (normal_s > 0) ? *onx_s / (normal_s) : 0;
          *onx_n = (normal_n > 0) ? *onx_n / (normal_n) : 0;
          *onx_ne = (normal_ne > 0) ? *onx_ne / (normal_ne) : 0;
          *onx_nw = (normal_nw > 0) ? *onx_nw / (normal_nw) : 0;
          *onx_se = (normal_se > 0) ? *onx_se / (normal_se) : 0;
          *onx_sw = (normal_sw > 0) ? *onx_sw / (normal_sw) : 0;

          *ony_e = (normal_e > 0) ? *ony_e / (normal_e) : 0;
          *ony_w = (normal_w > 0) ? *ony_w / (normal_w) : 0;
          *ony_s = (normal_s > 0) ? *ony_s / (normal_s) : 0;
          *ony_n = (normal_n > 0) ? *ony_n / (normal_n) : 0;
          *ony_ne = (normal_ne > 0) ? *ony_ne / (normal_ne) : 0;
          *ony_nw = (normal_nw > 0) ? *ony_nw / (normal_nw) : 0;
          *ony_se = (normal_se > 0) ? *ony_se / (normal_se) : 0;
          *ony_sw = (normal_sw > 0) ? *ony_sw / (normal_sw) : 0;

          *onx_e = 0.25 * (*onx_se + *onx_ne + 2 * (*onx_e));
          *onx_w = 0.25 * (*onx_sw + *onx_nw + 2 * (*onx_w));
          *ony_s = 0.25 * (*ony_se + *ony_sw + 2 * (*ony_s));
          *ony_n = 0.25 * (*ony_nw + *ony_ne + 2 * (*ony_n));

          k_x = 0.5 * (*onx_e - *onx_w);
          k_y = 0.5 * (*ony_n - *ony_s);

          *curv = 1 * (k_x + k_y);
        } else {
          *curv = 0;
        }

        nid++;
        curv++;
        onx++;
        ony++;
        onx_e++;
        onx_w++;
        onx_n++;
        onx_s++;
        onx_ne++;
        onx_nw++;
        onx_se++;
        onx_sw++;
        ony_e++;
        ony_w++;
        ony_n++;
        ony_s++;
        ony_ne++;
        ony_nw++;
        ony_se++;
        ony_sw++;

      }                         /*end of I loop */
      onx += 2;
      ony += 2;
      onx_e += 2;
      onx_w += 2;
      onx_n += 2;
      onx_s += 2;
      onx_ne += 2;
      onx_nw += 2;
      onx_se += 2;
      onx_sw += 2;
      ony_e += 2;
      ony_w += 2;
      ony_n += 2;
      ony_s += 2;
      ony_ne += 2;
      ony_nw += 2;
      ony_se += 2;
      ony_sw += 2;
    }                           /*end of J loop */
    onx += skip;
    ony += skip;
    onx_e += skip;
    onx_w += skip;
    onx_n += skip;
    onx_s += skip;
    onx_ne += skip;
    onx_nw += skip;
    onx_se += skip;
    onx_sw += skip;
    ony_e += skip;
    ony_w += skip;
    ony_n += skip;
    ony_s += skip;
    ony_ne += skip;
    ony_nw += skip;
    ony_se += skip;
    ony_sw += skip;
  }                             /*end of K loop */
  return (1);
}

/***************************************************/
/* Surface Tension                                 */
/* by Wei Wang on 06-03-01                         */
/***************************************************/
/* using NASTAC 'template' rule (1999) */
int surface_curvature (BB_struct * bp, int sbnum)
{
  SB_struct *sp;
  int i, j, k;                  /* tmp counter */
  int nx, ny, nz, skip;
  int *nid;
  CA_FLOAT *ofs, *ofs_e, *ofs_w, *ofs_n, *ofs_s, *ofs_ne, *ofs_nw, *ofs_se, *ofs_sw;
  CA_FLOAT *ncv;
  CA_FLOAT fs_aver;

  sp = bp->sb[sbnum];

  nx = bp->nc[0];
  ny = bp->nc[1];
  nz = bp->nc[2];
  skip = 2 * (nx + 2);

  /* Set up local pointers */

  ofs = bp->ftmp_one;
  ncv = sp->curv;               /* array of surface curvature, size nc[i]^3 */
  nid = sp->index;

  /*move the ptr to outside corner to inside corner */
  ofs += bp->cubeptr.flist[0][START];

  ofs_e = ofs + 1;
  ofs_w = ofs - 1;
  ofs_n = ofs + nx + 2;
  ofs_s = ofs - nx - 2;
  ofs_ne = ofs_n + 1;
  ofs_nw = ofs_n - 1;
  ofs_se = ofs_s + 1;
  ofs_sw = ofs_s - 1;

   /***********************************************/
  /* Beginning of main loop(s)                   */
   /***********************************************/

  for (k = 0; k < nz; k++) {
    for (j = 0; j < ny; j++) {
      for (i = 0; i < nx; i++) {

        if (*nid != 0) {

          fs_aver = *ofs + *ofs_e + *ofs_w + *ofs_n + *ofs_s + *ofs_ne + *ofs_nw + *ofs_se + *ofs_sw;

          *ncv = 0.0 * (1.0 - 0.22222222 * fs_aver);    /* NASTAC 1999 */
        }
        nid++;
        ncv++;
        ofs++;
        ofs_e++;
        ofs_w++;
        ofs_n++;
        ofs_s++;
        ofs_ne++;
        ofs_nw++;
        ofs_se++;
        ofs_sw++;
      }                         /*end of I loop */
      ofs += 2;
      ofs_e += 2;
      ofs_w += 2;
      ofs_n += 2;
      ofs_s += 2;
      ofs_ne += 2;
      ofs_nw += 2;
      ofs_se += 2;
      ofs_sw += 2;
    }                           /* end of J loop */
    ofs += skip;
    ofs_e += skip;
    ofs_w += skip;
    ofs_n += skip;
    ofs_s += skip;
    ofs_ne += skip;
    ofs_nw += skip;
    ofs_se += skip;
    ofs_sw += skip;
  }                             /* end of K loop */

  return (1);
}

char const *rcs_id_curvature_c ()
{
  static char const rcsid[] = "$Id: curvature.c 1329 2008-07-07 17:09:11Z  $";

  return (rcsid);
}
