
#include <stdio.h>
#include <math.h>
#include "blocks.h"
#include "machine.h"

#define MAXROOT 0.1
#define MINROOT -0.1
#define MAX_ITERATIONS 100
#define CONVERGE_LIMIT 1e-10

/* find the radius using Newton's method , adapted from Numerical Recipes in C by Press and Flannery */

CA_FLOAT find_new_radius (BB_struct * bp, CA_FLOAT n, CA_FLOAT p, CA_FLOAT tempK, CA_FLOAT r)
{
  CA_FLOAT f, df, dx, sigma, rold;
  int i;

  sigma = bp->mprops.surf_tens;

  for (i = 0; i < MAX_ITERATIONS; i++) {
    f = FOURPI_BY_THREE * (p * r * r * r + 2 * sigma * r * r) - n * GAS_CONST_SI * tempK;
    df = FOURPI_BY_THREE * (3 * p * r * r + 4 * sigma * r);
    dx = f / df;
    rold = r;
    r -= dx;
#ifdef TEST_FINDROOT
    printf ("r: %.10g, r-rold %.10g\n", r, r - rold);
#endif

    if ((MAXROOT - r) * (r - MINROOT) < 0.0) {
#ifdef ERROR_EXIT
      fprintf (stderr, "ERROR: find_new_radius: exceeded bounds on radius.\n");
      exit (10);
#endif
      return (r);
    }
    if (ABS (r - rold) < CONVERGE_LIMIT) {
#ifdef TEST_FINDROOT
      printf ("Niter: %i\n", i);
#endif
      return (r);
    }
  }
#ifdef ERROR_EXIT
  fprintf (stderr, "ERROR: find_new_radius: exceeded maximum iterations.\n");
  exit (10);
#endif
  return (r);
}

#ifdef TEST_FINDROOT
void main ()
{
  BB_struct bb;
  CA_FLOAT p, tempK, r, n, rNew;

  bb.mprops.surf_tens = 0.856;
  p = 101300.0;
  r = 1e-6;
  tempK = 600;

  for (;;) {
    printf ("Input nmols:\n");
    scanf ("%f", &n);
    rNew = find_new_radius (&bb, n, p, tempK, r);
    printf ("Rnew = %.5g\n", rNew);
    r = rNew;
  }

}
#endif

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file */
char const *rcs_id_findroot_c ()
{
  static char const rcsid[] = "$Id: findroot.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_subroutine */
