#include <math.h>
#include <stdio.h>
#include "machine.h"

/*
   BOX-MULLER routine for returning a random number normally distributed
   about a mean of zero, with a standard deviation of 1. Adapted from Numerical Recipes in C by Press and Flannery.
*/

CA_FLOAT box_muller ()
{
  static int flag = 0;
  static CA_FLOAT gset = 0;
  CA_FLOAT v1, v2, fac, gdev, r;

  if (!flag) {
    do {
      v1 = 2 * drand48 () - 1;
      v2 = 2 * drand48 () - 1;
      r = v1 * v1 + v2 * v2;
    } while (r >= 1 || r == 0);
    fac = sqrt ((-2 * LOG (r) / r));
    gset = v1 * fac;
    gdev = v2 * fac;
    flag = 1;
  } else {
    gdev = gset;
    flag = 0;
  }
  return (gdev);
}                               /* end of box_muller */

/* return gaussian devates for a given mean and standard deviation */
/* by calling the box_muller routine. */

CA_FLOAT gaussdev (CA_FLOAT * params)
{
  CA_FLOAT stdev;
  CA_FLOAT mean;

  if (params == NULL || (params + 1) == NULL) {
    fprintf (stderr, "ERROR: gaussdev: You messed up, NULL pointer here.\n");
    exit (1);
  }

  mean = params[0];
  stdev = params[1];
  return ((stdev * box_muller ()) + mean);
}

#ifdef TEST_BOX_MULLER
void main ()
{
  int i;
  CA_FLOAT stdev = 2;
  CA_FLOAT mean = 10;
  CA_FLOAT params[2];

  params[0] = mean;
  params[1] = stdev;
  for (i = 0; i < 10000; i++) {
    printf ("%.5g\n", gaussdev (params));
  }

}
#endif

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file */
char const *rcs_id_gaussdev_c ()
{
  static char const rcsid[] = "$Id: gaussdev.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_subroutine */
