/****************************************************************/
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      All Rights Reserved                                     */
/*      THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM    */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      This material contains CONFIDENTIAL INFORMATION that    */
/*      is the property of Imperial College. Any use,           */
/*      duplication or disclosure not specifically authorized   */
/*      by Imperial College is strictly prohibited.             */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/*  output_excel.c:						*/
/*  All subroutines related to output in EXCEL format.          */
/*  including:							*/
/*   sb_dump2screen      dump a matrix to the screen            */
/*   write_excel_header  write a header describing run to file  */
/*   write_bb_excel      write a single step's info to file     */
/*   write_bb_excel_sum: write a final summary info to file     */
/*   init_stat_val:      init. an array for calc. stats         */
/*   add_stat_val:       add a single value to stats array      */
/*   calc_stat_val:      calc. stats from stats array           */
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Wed Jul  1 18:38:31 bst 1998                 		*/
/****************************************************************/
/* 	MODIFIED by:						*/
/*  PDL: Aug 16, 1998						*/
/*  PDL: Sept 5, 1998						*/
/****************************************************************/
/****** To Do List **********************************************/
/*General:							*/
/* 1)                 						*/
/*   print_bb:           print out the bigblock in cap form	*/
/****************************************************************/
/*RCS Id:$Id: output_ex.c 887 2006-03-01 18:21:01Z rcatwood $*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>

#include "machine.h"
#include "blocks.h"
#include "find_max.h"
#include "props.h"
#include "interp.h"
#include "castats.h"
#include "checks.h"
#include "pore.h"
#include "write_values.h"

extern void write_grain_histo (BB_struct * bp, int stat_flag);
extern int sb_line_int (CA_FLOAT * line_res, BB_struct * bp, int sbnum);
extern void find_minmax (p_c_list * list, int cellminmax[]);

extern void write_header_ctrl (FILE * ffp, BB_struct * bp);

/* functions defined later in file  */
void write_conc_prof (int stat_flag, BB_struct * bp, Value_struct * vp);
void write_conc_prof_multi (int stat_flag, BB_struct * bp, Value_struct * vp, int i);
void write_temp_prof (int stat_flag, BB_struct * bp);
int write_excel_header (FILE * ffp, BB_struct * bp);

/* from sb_temp_calc.c */

/********************************************************/
/********************************************************/
/* sb_dump2screen                               	*/
/*      Subroutine to write a subblock to screen.       */
/********************************************************/
/********************************************************/
int sb_dump2screen (CA_FLOAT * a, int nx, int ny, int nz)
{
  int i, j, k, index;           /* tmp counters */

/************************************************/
/* Dump it out to screen for now...             */
/************************************************/
  index = 0;
  fprintf (stderr, "\nDump of CA_FLOAT[%d,%d]\n", nx, ny);
  for (k = 0; k < nz; k++) {    /* loop sb's in z direction */
    fprintf (stderr, "\nLayer[%d]\n", k);
    for (j = 0; j < ny; j++) {  /* loop sb's in z direction */
      for (i = 0; i < nx; i++) {        /* loop cells's in z direction */
        fprintf (stderr, "%2d ", (int) (a[index++] * 10));
      }
      fprintf (stderr, "\n");
    }
  }

  return (0);
}                               /* end of sb_dump2screen subroutine */

/********************************************************/
/********************************************************/
/* write_excel_header                                  	*/
/*   Subroutine to write HEADER SUMMARY in excel format.*/
/********************************************************/
/********************************************************/
int write_excel_header (FILE * ffp, BB_struct * bp)
{
  int i;                        /* tmp counters */
  int errors = 0;               /* error flag on return */
  int ele_num, ele_1;
  MultiS_struct *ms;
  Mat_str *mp;
  Ctrl_str *cp;
  Nuc_str *np;
  CA_FLOAT part, slop, liqdiff, soldiff, cinit;

  cp = (bp->ctrl);
  ele_num = cp->NUM_COMP;
  ele_1 = ele_num - 1;
  ms = &(bp->MultiSvals);
  mp = &(bp->mprops);
  np = &(bp->nprops);

  /*fprintf(stderr,"Hello world\n"); */
  fprintf (ffp, "Hello world\n");
  /* write out base filename used and date */
  fprintf (ffp, "Excel Data for run,%s\n", bp->ctrl->fn_base);
  fprintf (ffp, "Mat file name,%s\n", bp->ctrl->fn_mat);
  fprintf (ffp, "Geo file name,%s\n", bp->ctrl->fn_geo);
  fprintf (ffp, "Porosity,%i\nSchiel:,%i\n", bp->ctrl->pore, bp->ctrl->scheil);
  fprintf (ffp, "Number of components: %d \n", bp->ctrl->NUM_COMP);
  fprintf (ffp, "\n");

  fprintf (ffp, "alloy exponent,%f", ALLOY_EXPONENT);
  /* the diffusion info */
  if (bp->ctrl->diffuse == TRUE) {
#ifdef CONST_DIFF_COEFF
    fprintf (ffp, " CONST_DIFF_COEFF used, ");
#endif
#ifdef COND_DIFF_COEFF
    fprintf (ffp, " COND_DIFF_COEFF used, ");
#endif
#ifdef AV_DIFF_COEFF
    fprintf (ffp, " AV_DIFF_COEFF used, ");
#endif
#ifdef CELL_DIFF_COEFF
    fprintf (ffp, " CELL_DIFF_COEFF used, ");
#endif
#ifdef C_LIQ_OUTPUT
    fprintf (ffp, " C_LIQ_OUTPUT used, ");
#endif
    fprintf (ffp, "\n");
  }
  if (bp->ctrl->diffuse_alloy_multi == TRUE) {
    fprintf (ffp, "material property for multi component \n");
    for (i = 0; i < ele_1; i++) {
      liqdiff = ms->LDiff_multi[i];
      soldiff = ms->SDiff_multi[i];
      cinit = ms->Cinit_multi[i];
      part = ms->part_coef_multi[i];
      slop = ms->slope_multi[i];
      fprintf (ffp, "DliqAlloy for Component %d: %e \n", i + 1, liqdiff);
      fprintf (ffp, "DsolAlloy for component %d: %e \n", i + 1, soldiff);
      fprintf (ffp, "Initial Conc.Alloy for component %d: %f \n", i + 1, cinit);
      fprintf (ffp, "partcoefAlloy for component %d: %f \n", i + 1, part);
      fprintf (ffp, "slopeAlloy for component %d: %f \n", i + 1, slop);

    }
  }
#ifdef MULTICOMP
  fprintf (ffp, " T_EUT_1,%f, T_EUT_2, %f \n", T_EUT_1, T_EUT_2);
  fprintf (ffp, " TP_1,%f, TP_2, %f, \n", TP_1, TP_2);
  fprintf (ffp, " MAX_B_CONC_1,%f , MAX_B_CONC_2, %f \n ", MAX_B_CONC_1, MAX_B_CONC_2);
#endif

  fprintf (ffp, " DAS_PRE,%f\n", DAS_PRE);
  fprintf (ffp, " DAS_EXP,%f\n", DAS_EXP);
  fprintf (ffp, " DAS_COS_THETA,%f\n", DAS_COS_THETA);
  fprintf (ffp, " ALLOY_EXPONENT,%f\n", ALLOY_EXPONENT);
  fprintf (ffp, "\n");
  fprintf (ffp, " FLAT_GROWTH,%f\n", FLAT_GROWTH);
  fprintf (ffp, " P_NUC_MIN_SS,%f\n", P_NUC_MIN_SS);
  fprintf (ffp, " P_NUC_SIG_MULT,%f\n", P_NUC_SIG_MULT);
  fprintf (ffp, "\n");
  fprintf (ffp, "Nucleation Model,%i\n", np->nmodel);

  switch (np->nmodel) {
  case N_HETERO:               /* N_HETERO model */
    fprintf (ffp, "\nHeterogenious On [Species Nmax A rad B]: ");
    for (i = 0; i < np->nhet; i++)
      fprintf (ffp, "%d, %g, %g, %g, %g; ", i, np->nuc[i].Nmax, np->nuc[i].CA_A, np->nuc[i].rad, np->nuc[i].B);
    break;
  case N_DIST:
    fprintf (ffp, "nuc_dist_func_type,%i\n", np->nucdist.nuc_dist_func_type);
    fprintf (ffp, "Function Id string,%s\n", np->nucdist.Nuc_func_id[np->nucdist.nuc_dist_func_type]);
    fprintf (ffp, "n_U_bins,%i,Unuc_incr,%.5g,n_Tund_bins,%i,Tund_incr,%.5f\n",
             np->nucdist.n_U_bins, np->nucdist.Unuc_incr, np->nucdist.n_Tund_bins, np->nucdist.Tund_incr);
    fprintf (ffp, "\nNucleation Parameters");
    fprintf (ffp, "\ngd_max, %g; ", np->gd_max);
    fprintf (ffp, "Tn, %g; ", np->NucParams[0]);
    fprintf (ffp, "Tsigma, %g; ", np->NucParams[1]);
    break;
  case N_RAPPAZ:               /*fallthru */
  case N_RAP_DEL:              /*fallthru */
  case N_OLDF_DEL:

    fprintf (ffp, "\nNucleation Parameters");
    fprintf (ffp, "\ngd_max, %g; ", np->gd_max);
    fprintf (ffp, "Tn, %g; ", np->NucParams[0]);
    fprintf (ffp, "Tsigma, %g; ", np->NucParams[1]);
    break;
  case N_BLOCK:
    fprintf (ffp, "random function type,%i", np->nucdist.nuc_dist_func_type);
    fprintf (ffp, "\nNucleation Parameters");
    fprintf (ffp, "\ngd_max, %g; ", np->gd_max);
    fprintf (ffp, "Tn, %g; ", np->NucParams[0]);
    fprintf (ffp, "Tsigma, %g; ", np->NucParams[1]);
    fprintf (ffp, "GNParam2, %g; ", np->NucParams[2]);
    fprintf (ffp, "GNParam3, %g; ", np->NucParams[3]);
    break;

  default:
    fprintf (ffp, "\nERROR:write_excel_header: ********* Nucleation Model Undefined!!!!!!!!!!!![ %i ]\n", np->nmodel);
  }

/*******************************************/
/* Control structure values                */
/*******************************************/
  write_header_ctrl (ffp, bp);
  if (bp->ctrl->pore == TRUE) {
    fprintf (ffp, "\nPorosity ON\n");
    write_pprop_values (ffp, &(bp->pprops));
  }
  fprintf (ffp, "\n\n");

  fprintf (ffp, "\n*******Header Finished*******\n");

  return (errors);
}                               /* end of write_excel_header subroutine */

/********************************************************/
/********************************************************/
/* bb_excel_step                                	*/
/*      Subroutine to write data to file to be read by  */
/* excel.                                               */
/********************************************************/
/********************************************************/
int write_bb_excel (int stat_flag, BB_struct * bp)
{
  int i, j, k, index, err;      /* tmp counters */
  int poreminmax[6];
  Ctrl_str *cp;
  MultiS_struct *ms;
  int rtn_flag = 0, NTpores = 0;        /* error flag on return, total pores */
  int sbnum;                    /*subblock number */
  int ele_1, ele_num;
  Value_struct *local_v_struct;
  Value_struct **multi_conc_ptr;
  int t_index;

  char command[MAX_STRING_LEN];
  CA_FLOAT Nstat[MAX_NSP][4];
  CA_FLOAT *t_list;

  CA_FLOAT Tstat[4], fs_stat[4];
  CA_FLOAT statlist[N_T_LISTS][PORE_NTRAD][4], val;
  CA_FLOAT pctporlist[PORE_NTRAD];
  int count[N_T_LISTS][PORE_NTRAD];
  FILE *fd_ex, *fd_ex_pore, *fd_ex_pore_extent, *fd_ex_pore_stats;
  PORE_str *pl;
  T_struct *Tp;
  time_t realtime;
  char open_action[5];

  ms = &(bp->MultiSvals);
  cp = bp->ctrl;
  ele_num = cp->NUM_COMP;
  ele_1 = ele_num - 1;

  strcpy (open_action, "w");
/********************************************************/
/* if Flag = FIRST_CALL open file and write headers     */
/********************************************************/
  switch (stat_flag) {

  case RESTART_CALL:
    strcpy (open_action, "a");
  case FIRST_CALL:             /* first call to write_bb_excel, do all initilisation */
    /* open output file */
    bp->scr_dump_num = 0;
    sprintf (command, "%s.csv", bp->ctrl->fn_base);
    if ((bp->ctrl->fd_ex = fopen (command, open_action)) == NULL) {
      fprintf (stderr, "Error: can't open output file [%s]\n", command);
      rtn_flag = 1;
    } else {
      fd_ex = bp->ctrl->fd_ex;
      write_excel_header (fd_ex, bp);
      fprintf (fd_ex, "Realtime,");
      fprintf (fd_ex, "Step,Time");

      /* header for the nucleation model values */
      fprintf (fd_ex, ",GrainNcell,GrainNgrow,BB ngrains");
      if (bp->nprops.nmodel == N_HETERO) {
        for (i = 0; i < bp->nprops.nhet; i++)
          fprintf (fd_ex, ",Ntot[%d]", i);
      }

      /* header for the values per sb [T, ngr, fs] */
      for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {      /* write header for each sb */
        if (bp->sb[sbnum]->open) {
          fprintf (fd_ex, ",Temp[%d],ngrains[%d],fs[%d],fscheck,ngrowchk", sbnum, sbnum, sbnum);
          if (bp->ctrl->diffuse == TRUE) {
            fprintf (fd_ex, ",Ctot[%d],Cmax[%d],Cmin[%d]", sbnum, sbnum, sbnum);
            fprintf (fd_ex, ",SATmax[%d],SSmax[%d]", sbnum, sbnum);
            fprintf (fd_ex, ",CLmax[%d],CLmin[%d]", sbnum, sbnum);
          }
          if ((bp->ctrl->diffuse_alloy == TRUE) && (bp->ctrl->diffuse_alloy_multi == FALSE)) {
            fprintf (fd_ex, ",Atot[%d],Amax[%d],Amin[%d]", sbnum, sbnum, sbnum);
            fprintf (fd_ex, ",ALmax[%d],ALmin[%d]", sbnum, sbnum);
          }
          if (bp->ctrl->mould_src != 0) {
            fprintf (fd_ex, ",alloyaddsol[%d],alloyt_addsol[%d]", sbnum, sbnum);
          }
          if (bp->ctrl->mould_src != 0) {
            fprintf (fd_ex, ",gasaddsol[%d],gast_addsol[%d]", sbnum, sbnum);
          }
          if (bp->ctrl->pore == TRUE)
            fprintf (fd_ex, ",Npores[%d]/[%d],pore_nmols[%d]", sbnum, bp->sb[sbnum]->Npores, sbnum);
        }
      }
      fprintf (fd_ex, "\n");
    }

    /* output the frac solid info */
    write_conc_prof (stat_flag, bp, bp->c_fs_values);
    write_temp_prof (stat_flag, bp);

    /* output the gas info */
    if (bp->ctrl->diffuse == TRUE) {
      write_conc_prof (stat_flag, bp, bp->c_sol_values);
    }

    /* output the alloy info */
    if (bp->ctrl->diffuse_alloy == TRUE && bp->ctrl->diffuse_alloy_multi == FALSE) {
      write_conc_prof (stat_flag, bp, bp->c_sol_alloy_values);
    }

    #ifdef CHIRAZI_MULTI
    /* output the multi component alloy info */
    if (bp->ctrl->diffuse_alloy_multi == TRUE) {
      for (i = 0; i < ele_1; i++) {
#ifdef C_LIQ_OUTPUT
        local_v_struct = &(bp->multi_conc[i][0]);
        sprintf (local_v_struct->id_string, "A_ELE_%d", i + 1);
        sprintf (command, "%s%s_multi.txt", local_v_struct->id_string, bp->ctrl->fn_base);
        local_v_struct->fd_exel = fopen (command, "w");
        fd_ex = local_v_struct->fd_exel;
        fprintf (fd_ex, "concentration for component %d is written \n ", i + 1);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);
#else
        local_v_struct = &(bp->multi_conc_tot[i][0]);
        sprintf (local_v_struct->id_string, "A_TOT_ELE_%d", i + 1);
        sprintf (command, "%s%s_multi.txt", local_v_struct->id_string, bp->ctrl->fn_base);
        local_v_struct->fd_exel = fopen (command, "w");
        fd_ex = local_v_struct->fd_exel;
        fprintf (fd_ex, "concentration for component %d is written \n ", i + 1);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);

        local_v_struct = &(bp->multi_conc_n_eut[i][0]);
        sprintf (local_v_struct->id_string, "A_Solid_ELE_%d", i + 1);
        sprintf (command, "%s%s_multi.txt", local_v_struct->id_string, bp->ctrl->fn_base);
        local_v_struct->fd_exel = fopen (command, "w");
        fd_ex = local_v_struct->fd_exel;
        fprintf (fd_ex, "concentration for component %d is written \n ", i + 1);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);
#endif
      }
    }
    #endif /* CHIRAZI_MULTI */

    /* output the pore info to seperate file */
    if (bp->ctrl->pore == TRUE) {
      /*open the main pore info file */
      sprintf (command, "P%s.csv", bp->ctrl->fn_base);
      if ((bp->ctrl->fd_ex_pore = fopen (command, open_action)) == NULL) {
        fprintf (stderr, "Error: can't open pore output file [%s]\n", command);
        rtn_flag = 1;
      } else {
        fd_ex_pore = bp->ctrl->fd_ex_pore;
        write_excel_header (fd_ex_pore, bp);
        fprintf (stderr, "Writing pore header\n");
        fprintf (fd_ex_pore, "Subblock:");

        for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {    /* write header for each sb */
          if (bp->sb[sbnum]->open) {

            NTpores += bp->sb[sbnum]->Npores;
            fprintf (fd_ex_pore, ",%i", sbnum);
          }
        }
        fprintf (fd_ex_pore, ",TotalPores,%i\n", NTpores);
        fprintf (fd_ex_pore, "MaxNpores: %i", sbnum);
        for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {    /* write header for each sb */
          if (bp->sb[sbnum]->open) {
            NTpores += bp->sb[sbnum]->Npores;
            fprintf (fd_ex_pore, ",%i", bp->sb[sbnum]->Npores);
          }
        }
        fprintf (fd_ex_pore, "\n");
        fprintf (fd_ex_pore, "\n");
      }
      /*open the pore min/max extent file */
      sprintf (command, "PE%s.csv", bp->ctrl->fn_base);
      if ((bp->ctrl->fd_ex_pore_extent = fopen (command, open_action)) == NULL) {
        fprintf (stderr, "Error: can't open pore E output file [%s]\n", command);
        rtn_flag = 1;
      } else {
        fd_ex_pore_extent = bp->ctrl->fd_ex_pore_extent;
        write_excel_header (fd_ex_pore_extent, bp);
        fprintf (stderr, "Writing pore header\n");
        fprintf (fd_ex_pore_extent, "Subblock:");

        for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {    /* write header for each sb */
          if (bp->sb[sbnum]->open) {
            NTpores += bp->sb[sbnum]->Npores;
            fprintf (fd_ex_pore, ",%i", sbnum);
          }
        }
        fprintf (fd_ex_pore_extent, ",TotalPores,%i\n", NTpores);
        fprintf (fd_ex_pore_extent, "MaxNpores: %i", sbnum);
        for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {    /* write header for each sb */
          if (bp->sb[sbnum]->open) {
            NTpores += bp->sb[sbnum]->Npores;
            fprintf (fd_ex_pore_extent, ",%i", bp->sb[sbnum]->Npores);
          }
        }
        fprintf (fd_ex_pore_extent, "\n");
        fprintf (fd_ex_pore_extent, "\n");
        fprintf (fd_ex_pore_extent, "Pore Extent:\n");
        fprintf (fd_ex_pore_extent, "PoreNum,MinX,MinY,MinZ,MaxX,MaxY,MaxZ\n");
      }
      /*open the pore stat only file */
      sprintf (command, "PS%s.csv", bp->ctrl->fn_base);
      if ((bp->ctrl->fd_ex_pore_stats = fopen (command, open_action)) == NULL) {
        fprintf (stderr, "Error: can't open pore S output file [%s]\n", command);
        rtn_flag = 1;
      } else {
        fd_ex_pore_stats = bp->ctrl->fd_ex_pore_stats;
        write_excel_header (fd_ex_pore_stats, bp);
        fprintf (stderr, "Writing pore header\n");
        fprintf (fd_ex_pore_stats, "Pore statistics\n");
      }

    }                           /* end of write pore info */
    write_grain_histo (bp, stat_flag);
    break;                      /* end of FIRST_CALL */

/********************************************************/
/* if Flag = STEP_CALL write out one timestep's info    */
/********************************************************/
  case STEP_CALL:
    fd_ex = bp->ctrl->fd_ex;
    realtime = time (0);
    /* loop through all sb's generating average values... */
    if (bp->ntsb == 1) {        /* if only one, use sb[0] values */
      Tstat[0] = Tstat[1] = Tstat[2] = bp->sb[0]->Tvals.Tavg;
      Tstat[3] = 0.0;
      fs_stat[0] = fs_stat[1] = fs_stat[2] = bp->sb[0]->Tvals.fsavg;
      fs_stat[3] = 0.0;
    } else {                    /* more than one, so loop it!    */
      init_stat_val (Tstat);
      init_stat_val (fs_stat);
      for (i = 0; i < bp->ntsb; i++) {  /* calc stats for each sb      */
        if (bp->sb[i]->open) {
          add_stat_val (Tstat, bp->sb[i]->Tvals.Tavg);
          add_stat_val (fs_stat, bp->sb[i]->Tvals.fsavg);
        }
      }
      calc_stat_val (Tstat, (CA_FLOAT) bp->ntsb);
      calc_stat_val (fs_stat, (CA_FLOAT) bp->ntsb);
    }                           /* end of stat generating loop */

      /******************************************************/
    /* Write out all values on one line for this timestep */
      /******************************************************/
    /* write out the BB stat values */
    fprintf (fd_ex, "%.10g,", (double) realtime);
    fprintf (fd_ex, "%d,%.10g,%.10g,%.10g", bp->step, bp->sim_time, grain_ncells (bp), grain_ngrow (bp));

    /* write out the nucleation model values */
    fprintf (fd_ex, ",%d", bp->nprops.ngr);
    if (bp->nprops.nmodel == N_HETERO) {
      for (i = 0; i < bp->nprops.nhet; i++)
        fprintf (fd_ex, ",%g", bp->nprops.nuc[i].Ntot);
    }

    /*write out frac_solid profile to seperate excel-type file */
    write_conc_prof (stat_flag, bp, bp->c_fs_values);

    /*write out concentration profile to seperate excel-type file */
    if (bp->ctrl->diffuse == TRUE) {
      write_conc_prof (stat_flag, bp, bp->c_sol_values);
    }

    /*write out concentration profile to seperate excel-type file */
    if ((bp->ctrl->diffuse_alloy == TRUE) && (bp->ctrl->diffuse_alloy_multi == FALSE)) {
      write_conc_prof (stat_flag, bp, bp->c_sol_alloy_values);
    }

    #ifdef CHIRAZI_MULTI
    /*write out multi concentration profile to seperate excel-type file */
    if (bp->ctrl->diffuse_alloy_multi == TRUE) {
      for (i = 0; i < ele_1; i++) {
#ifdef C_LIQ_OUTPUT
        local_v_struct = &(bp->multi_conc[i][0]);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);
#else
        local_v_struct = &(bp->multi_conc_tot[i][0]);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);

        local_v_struct = &(bp->multi_conc_n_eut[i][0]);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);
#endif

      }
    }
    #endif /* CHIRAZI_MULTI */

    if (bp->ctrl->pore == TRUE) {
      /* collate the pore radius values */
      sbnum = bp->ctrl->pore_dump_sb /*=0*/ ;
      k = bp->scr_dump_num;
      fprintf (stderr, "  screendump # %i\n", k);
      if (bp->sb[sbnum]->open) {
        pl = bp->sb[sbnum]->porelist;
        if (k > bp->ctrl->nscrdumps) {
          fprintf (stderr, "ERROR: output_ex: Too many pore dumps! [%i]\n", k);
        } else {
          for (i = 0; i < bp->sb[sbnum]->Npores; i++) {
            if (pl[i].State != PORE_NONE) {
#ifdef MORE_PORE
              pl[i].radlist[k] = pl[i].Radius;
              pl[i].nmollist[k] = pl[i].Nmols;
              pl[i].templist[k] = pl[i].Temp;
#endif
            }                   /* end of if pore > 0 */
          }                     /* end of looping through pores */
        }
      }
    }

    /* end of if pore = true */
    /* append the histogram of grain sizes */
    write_grain_histo (bp, stat_flag);

    /* write out the values per sb [T, ngr, fs,npores] */
    for (sbnum = 0; sbnum < bp->ntsb; sbnum++) {        /* write values for each sb */
      if (bp->sb[sbnum]->open) {

        /* print T, ngr, fs */
        Tp = &(bp->sb[sbnum]->Tvals);
        fprintf (fd_ex, ",%.10g,%d,%.10g,%.10g,%i", Tp->Tavg, bp->sb[sbnum]->ngr, Tp->fsavg, find_fs_av (bp, sbnum),
                 find_ngrow (bp, sbnum));
        if (bp->ctrl->diffuse == TRUE) {
          /*find the maximum and the total in the conc. array */
          err = find_max_conc (bp, bp->c_sol_values, sbnum);
          if (err != 0)
            fprintf (stderr, "ERROR: output_ex: step %i: %d negative gas concentration values!", bp->step, err);
          fprintf (fd_ex, ",%g,%g,%g", ((bp->vol_c) * (bp->c_sol_values->Ctot)), bp->c_sol_values->Cmax, bp->c_sol_values->Cmin);
          fprintf (fd_ex, ",%g,%g", bp->c_sol_values->SATmax, bp->c_sol_values->SSmax);
          fprintf (fd_ex, ",%g,%g", bp->c_sol_values->CLmax, bp->c_sol_values->CLmin);
        }                       /*end diffuse (gas) */
        if ((bp->ctrl->diffuse_alloy == TRUE) && (bp->ctrl->diffuse_alloy_multi == FALSE)) {
          /*find the maximum and the total in the conc. array */
          err = find_max_conc (bp, bp->c_sol_alloy_values, sbnum);
          if (err != 0)
            fprintf (stderr, "ERROR: output_ex: step %i:  %d negative alloy concentration values!", bp->step, err);
          fprintf (fd_ex, ",%g,%g,%g", bp->c_sol_alloy_values->Ctot, bp->c_sol_alloy_values->Cmax, bp->c_sol_alloy_values->Cmin);
          fprintf (fd_ex, ",%g,%g", bp->c_sol_alloy_values->CLmax, bp->c_sol_alloy_values->CLmin);
        }                       /*end diffuse alloy */
        if (bp->ctrl->mould_src != 0) {
          fprintf (fd_ex, ",%g,%g", bp->sb[sbnum]->Svals[ALLOY].addsol, bp->sb[sbnum]->Svals[ALLOY].t_addsol);
          fprintf (fd_ex, ",%g,%g", bp->sb[sbnum]->Svals[GAS], bp->sb[sbnum]->Svals[GAS].t_addsol);
        }

        /* and the amount of H2 in all pores */
        if (bp->ctrl->pore == TRUE) {
          CA_FLOAT pore_nmols = 0;

          pl = bp->sb[sbnum]->porelist;
          fprintf (fd_ex, ",%d", bp->sb[sbnum]->sb_npores);
          /* add up the n-mols in the pores */
          for (i = 0; i < bp->sb[sbnum]->Npores; i++) {
            switch (pl[i].State) {
            case PORE_OFF:     /*fallthrough */
            case PORE_NONE:    /*fallthrough */
            case PORE_LATENT:
            case NOT_CASTING:
              /* do nothing */
              break;

            case PORE_SPHERE:  /*fallthrough */
            case PORE_TUBE:
            case PORE_FROZEN:
            case PORE_MULTI:
              pore_nmols += pl[i].NmolsH;
              break;

            default:
              fprintf (stderr, "Output_ex: Pore state in hyperspace %i\n", pl[i].State);
              break;
            }                   /* end of pore switch */
          }                     /* end of looping through pores */
          fprintf (fd_ex, ",%.10g", pore_nmols);
#ifdef ONEPORE
          fprintf (fd_ex, ",%.10g", pl[0].EqRad);
#endif
        }
      }                         /*end check if block is open */
    }                           /*end loop trhough subblocksl */

    fprintf (fd_ex, "\n");
    fflush (NULL);
    bp->scr_dump_num++;
    break;

/********************************************************/
/* if Flag = LAST_CALL write out one timestep's info    */
/********************************************************/
  case LAST_CALL:

    /*write out frac solid profile to seperate excel-type file */
    write_conc_prof (stat_flag, bp, bp->c_fs_values);

    /*write out concentration profile to seperate excel-type file */
    if (bp->ctrl->diffuse == TRUE) {
      write_conc_prof (stat_flag, bp, bp->c_sol_values);
    }

    /*write out concentration profile to seperate excel-type file */
    if ((bp->ctrl->diffuse_alloy == TRUE) && (bp->ctrl->diffuse_alloy_multi == FALSE)) {
      write_conc_prof (stat_flag, bp, bp->c_sol_alloy_values);
    }

    #ifdef CHIRAZI_MULTI
    /*write out multi concentration profile to seperate excel-type file */
    if (bp->ctrl->diffuse_alloy_multi == TRUE) {
      for (i = 0; i < ele_1; i++) {
#ifdef C_LIQ_OUTPUT
        local_v_struct = &(bp->multi_conc[i][0]);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);
#else
        local_v_struct = &(bp->multi_conc_tot[i][0]);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);

        local_v_struct = &(bp->multi_conc_n_eut[i][0]);
        write_conc_prof_multi (stat_flag, bp, local_v_struct, i);
#endif
      }
    }
    #endif /* CHIRAZI_MULTI */

    /* write out the pore list */
    if (bp->ctrl->pore == TRUE) {
      fd_ex_pore_extent = bp->ctrl->fd_ex_pore_extent;
      sbnum = bp->ctrl->pore_dump_sb;
      if (bp->sb[sbnum]->open) {
        k = bp->scr_dump_num;
        pl = bp->sb[sbnum]->porelist;

       /*******************************************/
        /* the min,max extent of the pore to a seperate file */
       /*******************************************/
        for (i = 0; i < bp->sb[sbnum]->Npores; i++) {
          int ii;

          switch (pl[i].State) {
          case PORE_NONE:      /*fallthrough */
          case PORE_LATENT:
          case NOT_CASTING:
            /* do nothing */
            break;

          case PORE_OFF:       /*fallthrough */
          case PORE_SPHERE:    /*fallthrough */
          case PORE_TUBE:
          case PORE_MULTI:
          case PORE_FROZEN:
            fprintf (fd_ex_pore_extent, "%i", i);
            find_minmax (pl[i].boundary, poreminmax);
            for (ii = 0; ii < 6; ii++) {
              fprintf (fd_ex_pore_extent, ",%i", poreminmax[ii]);
            }
            fprintf (fd_ex_pore_extent, "\n");
            break;

          default:
            fprintf (stderr, "Output_ex: Pore state in hyperspace %i\n", pl[i].State);
            break;

          }                     /* end existing pore test */
        }                       /* end loop to find extent of each pore */

        /* produce trad list */
        fprintf (bp->ctrl->fd_ex_pore, "\n");
        fprintf (bp->ctrl->fd_ex_pore, "Pore Data lists:\n");
        /*******************************************/
        /* loop to print out all defined t_lists   */
        /*******************************************/
        for (t_index = 0; t_index < N_T_LISTS; t_index++) {     /*see pore.h */

          /*write the temperature line */
          fprintf (bp->ctrl->fd_ex_pore, "Temp:,x,x,x,x,x,x,x,x,x,x,x,x,x,x,x");
          for (j = 0; j < PORE_NTRAD; j++) {
            fprintf (bp->ctrl->fd_ex_pore, ",%.10e", bp->pprops.P_trad_max - j * bp->pprops.P_trad_step);
          }
          fprintf (bp->ctrl->fd_ex_pore, "\n");
          /* initialize the stat values list for this t_list */
          for (j = 0; j < PORE_NTRAD; j++) {
            init_stat_val (statlist[t_index][j]);
            count[t_index][j] = 0;
          }
          /* process this data for all pores and output */
          for (i = 0; i < bp->sb[sbnum]->Npores; i++) {
            /* process existing pores */
            switch (pl[i].State) {
            case PORE_NONE:    /*fallthrough */
            case PORE_LATENT:
            case NOT_CASTING:
              /* do nothing */
              break;

            case PORE_OFF:     /*fallthrough */
            case PORE_SPHERE:  /*fallthrough */
            case PORE_TUBE:
            case PORE_MULTI:
            case PORE_FROZEN:
              t_list = pl[i].t_lists[t_index];  /* local version of current data t_list */

              /* row headers */
#                     ifndef STAT_ONLY  /* flag to limit amount of data to stat values only - for big runs */
              fprintf (bp->ctrl->fd_ex_pore, "%s:,%i", t_listnames[t_index], i);
              fprintf (bp->ctrl->fd_ex_pore, ",State:,%i", pl[i].State);
              fprintf (bp->ctrl->fd_ex_pore, ",Thr:,%.10g", pl[i].Thresh);
              fprintf (bp->ctrl->fd_ex_pore, ",NucSS,%.10g", pl[i].NucSS);
              fprintf (bp->ctrl->fd_ex_pore, ",NucTemp,%.10g", pl[i].NucTemp);
              fprintf (bp->ctrl->fd_ex_pore, ",PoreNcells:,%i", pl[i].ncells);
              fprintf (bp->ctrl->fd_ex_pore, ",PoreXYZ:,%i,%i,%i", pl[i].Cell[0], pl[i].Cell[1], pl[i].Cell[2]);
#                     endif /*STAT_ONLY */

              /* use the stat_values routines and print values */
              for (j = 0; j < PORE_NTRAD; j++) {
                val = t_list[j];
                if (val != 0.0) {
                  add_stat_val (statlist[t_index][j], val);
                  count[t_index][j]++;
                }
#                        ifndef STAT_ONLY       /* flag to limit amount of data to stat values only - for big runs */
                fprintf (bp->ctrl->fd_ex_pore, ",%.10e", val);
#                        endif /*STAT_ONLY */
              }                 /* end process current list (index j loop) */
#                     ifndef STAT_ONLY  /* flag to limit amount of data to stat values only - for big runs */
              fprintf (bp->ctrl->fd_ex_pore, "\n");
#                     endif /*STAT_ONLY */
              break;

            default:
              fprintf (stderr, "Output_ex: Pore state in hyperspace %i\n", pl[i].State);
              break;

            }                   /* end existing pore test */
          }                     /*end list out list for current pore (index i loop) */
#          ifndef STAT_ONLY     /* flag to limit amount of data to stat values only - for big runs */
          fprintf (bp->ctrl->fd_ex_pore, "\n");
#          endif /*STAT_ONLY */
        }                       /* end of t_list output loop (t_index loop) */

        fprintf (bp->ctrl->fd_ex_pore, "\n");

        /* caculate percent porosity from vol sums */
        /* Have to do this before calc-stat-val    */
        /* while statlist has the sum instead of the mean */
        for (j = 0; j < PORE_NTRAD; j++) {
          pctporlist[j] = statlist[T_VOL_LIST][j][0] / bp->vol_sb;
        }
        /* list outthe statistical values */
        for (t_index = 0; t_index < N_T_LISTS; t_index++) {

          fprintf (bp->ctrl->fd_ex_pore, "\n");
          fprintf (bp->ctrl->fd_ex_pore, "%s stat values:", t_listnames[t_index]);
          fprintf (bp->ctrl->fd_ex_pore_stats, "\n");
          fprintf (bp->ctrl->fd_ex_pore_stats, "%s stat values:", t_listnames[t_index]);
          for (j = 0; j < PORE_NTRAD; j++) {
            calc_stat_val (statlist[t_index][j], (CA_FLOAT) count[t_index][j]);
          }

          fprintf (bp->ctrl->fd_ex_pore, "\n");
          fprintf (bp->ctrl->fd_ex_pore, "Temp:,x,x,x,x,x,x,x,x,");
          fprintf (bp->ctrl->fd_ex_pore_stats, "\n");
          fprintf (bp->ctrl->fd_ex_pore_stats, "Temp:,x,x,x,x,x,x,x,x,");
          for (j = 0; j < PORE_NTRAD; j++) {
            fprintf (bp->ctrl->fd_ex_pore, ",%.10e", bp->pprops.P_trad_max - j * bp->pprops.P_trad_step);
            fprintf (bp->ctrl->fd_ex_pore_stats, ",%.10e", bp->pprops.P_trad_max - j * bp->pprops.P_trad_step);
          }
          /* loop through stat value array for each value */
          for (i = 0; i < 4; i++) {
            fprintf (bp->ctrl->fd_ex_pore, "\n");
            fprintf (bp->ctrl->fd_ex_pore, "statval %i:,x,x,x,x,x,x,x,x,", i);
            fprintf (bp->ctrl->fd_ex_pore_stats, "\n");
            fprintf (bp->ctrl->fd_ex_pore_stats, "statval %i:,x,x,x,x,x,x,x,x,", i);
            for (j = 0; j < PORE_NTRAD; j++) {
              fprintf (bp->ctrl->fd_ex_pore, ",%.10e", statlist[t_index][j][i]);
              fprintf (bp->ctrl->fd_ex_pore_stats, ",%.10e", statlist[t_index][j][i]);
            }
          }
          fprintf (bp->ctrl->fd_ex_pore, "\n");
          fprintf (bp->ctrl->fd_ex_pore, "count:,x,x,x,x,x,x,x,x,");
          fprintf (bp->ctrl->fd_ex_pore_stats, "\n");
          fprintf (bp->ctrl->fd_ex_pore_stats, "count:,x,x,x,x,x,x,x,x,");
          for (j = 0; j < PORE_NTRAD; j++) {
            fprintf (bp->ctrl->fd_ex_pore, ",%i", count[t_index][j]);
            fprintf (bp->ctrl->fd_ex_pore_stats, ",%i", count[t_index][j]);
          }
          fprintf (bp->ctrl->fd_ex_pore, "\n");
          fprintf (bp->ctrl->fd_ex_pore_stats, "\n");
        }                       /* end list out all statistical values */
        /* list out the percent porostiy */
        fprintf (bp->ctrl->fd_ex_pore, "pct por:,x,x,x,x,x,x,x,x,");
        fprintf (bp->ctrl->fd_ex_pore_stats, "pct por:,x,x,x,x,x,x,x,x,");
        for (j = 0; j < PORE_NTRAD; j++) {
          fprintf (bp->ctrl->fd_ex_pore, ",%.10e", pctporlist[j]);
          fprintf (bp->ctrl->fd_ex_pore_stats, ",%.10e", pctporlist[j]);
        }
        fprintf (bp->ctrl->fd_ex_pore, "\n");
        fprintf (bp->ctrl->fd_ex_pore_stats, "\n");

        /*add a recognizable end string to aid with post=processing */
        fprintf (bp->ctrl->fd_ex_pore, "EnDeNd EnDeNd\n");
        fprintf (bp->ctrl->fd_ex_pore_stats, "EnDeNd EnDeNd\n");
        fprintf (bp->ctrl->fd_ex_pore_extent, "EnDeNd EnDeNd\n");

        fclose (bp->ctrl->fd_ex_pore);
        fclose (bp->ctrl->fd_ex_pore_stats);
        fclose (bp->ctrl->fd_ex_pore_extent);
      }                         /* end check if sb is open */
    }
    /* end if  pore mode on */
    fclose (bp->ctrl->fd_ex);
    break;

  default:
    fprintf (stderr, "ERROR:output_ex: stat_flag value [%d] undefined.\n", stat_flag);
    break;
  }                             /* end of Flag switch */

  return (rtn_flag);
}                               /* end of write_bb_excel subroutine */

/********************************************************/
/********************************************************/
/* write_bb_excel_sum                                  	*/
/*    Subroutine to write SUMMARY table in excel format.*/
/********************************************************/
/********************************************************/
int write_bb_excel_sum (int stat_flag, BB_struct * bp)
{
  int i, j, k, index;           /* tmp counters */
  int rtn_flag = 0;             /* error flag on return */
  char command[MAX_STRING_LEN];
  int *c_per_gr, *gr, *zerograin;
  CA_FLOAT tmp, gr_stat[4], gr_Rstat[4];
  FILE *fd;
  Ind_grain *c_gr;

/********************************************************/
/* open the output file                                 */
/********************************************************/
  /* open output file */
  sprintf (command, "%s_sum.csv", bp->ctrl->fn_base);
  if ((fd = fopen (command, "w")) == NULL) {
    fprintf (stderr, "Error: can't open output file [%s]\n", command);
    rtn_flag = 1;
  } else {
    fprintf (fd, "test\n");

/********************************************************/
/* malloc array to hold grain # of cells and set = 0    */
/********************************************************/
    if (!(c_per_gr = (int *) calloc (1 + bp->nprops.ngr, sizeof (int)))) {
      fprintf (stderr, "ERROR: gr size counting array malloc failed\n");
      return (1);
    }
    if (!(zerograin = (int *) calloc (bp->ncsb, sizeof (int)))) {
      fprintf (stderr, "ERROR: zerogr array malloc failed\n");
      return (1);
    }

/********************************************************/
/* for each sb loop through and calc average gs and     */
/* write it out to the file ...                         */
/********************************************************/
    for (i = 0; i < bp->ntsb; i++) {    /*loop through all sb */
      if (bp->sb[i]->open == TRUE) {
        gr = bp->sb[i]->gr;
      } else {
        gr = zerograin;
      }
      for (j = 0; j < bp->ncsb; j++) {  /*loop thru all cells in sb */
        if (*gr >= 0) {
          c_per_gr[*(gr++)]++;
        }
      }
    }

/********************************************************/
/* calculate stats on grain sizes                       */
/********************************************************/
    init_stat_val (gr_stat);
    for (i = 0; i < bp->nprops.ngr; i++) {      /*write header for each sb */
      add_stat_val (gr_stat, (CA_FLOAT) c_per_gr[i]);
    }
    calc_stat_val (gr_stat, (CA_FLOAT) bp->nprops.ngr);
    for (i = 0; i < 4; i++) {   /* convert to mm^3 and microns */
      gr_stat[i] = bp->vol_c * gr_stat[i];
      tmp = gr_stat[i] * THREE_BY_4PI;
      gr_Rstat[i] = 1000.0 * POW (tmp, (CA_FLOAT) (1.0 / 3.0));
    }

/********************************************************/
/* write out the file                                   */
/********************************************************/
    fprintf (fd, "test\n");
    write_excel_header (fd, bp);
    if (bp->ctrl->diffuse_alloy_multi == TRUE) {
      fprintf (fd, "number of cells with eutectic phase is: %d  \n", bp->nbeut);
      fprintf (fd, "number of cells with ternary phase is: %d  \n", bp->nteut);
    }
    fprintf (fd, "Volume in m^3, Radius in Millimeters\n");
    fprintf (fd, "Ngr,Avg V,Min V,Max V,Sigma V");
    fprintf (fd, ",Avg R,Min R,Max R,Sigma R");
    fprintf (fd, ",LineIntGrSz\n");
    fprintf (fd, "%d", bp->nprops.ngr);
    for (i = 0; i < 4; i++)
      fprintf (fd, ",%g", gr_stat[i]);
    for (i = 0; i < 4; i++)
      fprintf (fd, ",%g", gr_Rstat[i]);
    for (i = 0; i < bp->ntsb; i++) {
      CA_FLOAT lineres;

      if (sb_line_int (&lineres, bp, i) == 0) {
        fprintf (fd, ",%.5g", lineres);
      } else {
        fprintf (fd, ",ERROR");
      }
    }
    fprintf (fd, "\n");
    fprintf (fd, "\nGrain,Ncells,NucTh,TNuc,TunderNuc,CellConcNuc,");
    fprintf (fd, "Xmax,Ymax,Zmax,Xmin,Ymin,Zmin\n");
    for (i = 1; i < bp->nprops.ngr; i++) {      /*write out the # cells/grain */
      c_gr = *(bp->gr + i);
      fprintf (fd, "%d,%d,%.10g,%.10g,%.10g,%.10g,", i, c_per_gr[i], c_gr->NucTh, c_gr->TNuc, c_gr->TunderNuc, c_gr->CellConcNuc);
      for (j = 0; j < 3; j++) {
        fprintf (fd, "%d,", c_gr->max[j]);
      }
      for (j = 0; j < 3; j++) {
        fprintf (fd, "%d,", c_gr->min[j]);
      }

      fprintf (fd, "\n");
    }

    if (bp->ctrl->pore == TRUE) {
      for (j = 0; j < bp->ntsb; j++) {
        fprintf (fd, "Pores:Subblock %i\n", j);
        fprintf (fd, "Cellnum,Radius,Dwelltime\n");
        for (i = 0; i < bp->sb[j]->Npores; i++) {
          if (!(bp->sb[j]->porelist[i].State == PORE_NONE || bp->sb[j]->porelist[i].State == PORE_LATENT)) {
            fprintf (fd, "%d,%g,%g\n", bp->sb[j]->porelist[i].Cellnum, bp->sb[j]->porelist[i].Radius, bp->sb[j]->porelist[i].Time);
          }
        }
      }
    }
    fclose (fd);
    free (c_per_gr);
    free (zerograin);
  }
  return (rtn_flag);
}                               /* end of write_bb_excel_sum subroutine */

void write_temp_prof (int stat_flag, BB_struct * bp)
{
}

/********************************************************/
/********************************************************/
/* write_conc_prof                                     	*/
/*    write a concentration profile to excel file        */
/********************************************************/
/********************************************************/
void write_conc_prof (int stat_flag, BB_struct * bp, Value_struct * output_v)
{
  if (!(bp->ctrl->do_conc_prof)) {
    return;                     /* return immediately if the flag for concentration profiles is not set */
  }

  {                             /* otherwise do the subroutine */
    int start, i, *conc_prof, sbnum;
    char command[MAX_STRING_LEN];

#ifdef PRINT_AV_PROFILE
    char avprofname[MAX_STRING_LEN];
    FILE *avfp;                 /*av profile file pointer */
#endif /*PRINT_AV_PROFILE */
    static int msg[3] = { 0, 0, 0 };
    CA_FLOAT *fs, *fsp, *sol, *sp, partcoef;
    FILE *fp;                   /*file pointer */
    char open_action[5];

    strcpy (open_action, "w");
/********************************************************/
/* if Flag = FIRST_CALL open file and write headers     */
/********************************************************/
    switch (stat_flag) {

    case RESTART_CALL:
      strcpy (open_action, "a");
      /* fall through */
    case FIRST_CALL:
      sprintf (command, "%s%s.txt", output_v->id_string, bp->ctrl->fn_base);
      if ((output_v->fd_exel = fopen (command, open_action)) == NULL) {
        fprintf (stderr, "Error: can't open %s. output file [%s]\n", output_v->id_string, command);
        return;
      } else {
        fp = output_v->fd_exel;
        fprintf (fp, "%s CONCENTRATION OUTPUT FOR MULTICOMP \n", output_v->id_string);
        write_excel_header (fp, bp);
        fprintf (fp, "Step,Time,");
        for (i = 0; i < bp->nc[0]; i++) {
          fprintf (fp, "%f,", i * bp->size_c[0]);
        }
        fprintf (fp, "\n");
      }
#ifdef PRINT_AV_PROFILE
      sprintf (avprofname, "%sAV%s.txt", output_v->id_string, bp->ctrl->fn_base);
      if ((avfp = fopen (avprofname, open_action)) == NULL) {
        fprintf (stderr, "Error: can't open %s. output file [%s]\n", output_v->id_string, command);
        return;
      } else {
        fprintf (avfp, "%s CONCENTRATION OUTPUT\n", output_v->id_string);
        write_excel_header (avfp, bp);
        fprintf (avfp, "Step,Time,");
        for (i = 0; i < bp->nc[0]; i++) {
          fprintf (avfp, "%f,", i * bp->size_c[0]);
        }
        fprintf (avfp, "\n");
      }
      fclose (avfp);
      if (strcmp ("G_", output_v->id_string) == 0) {
        /* do a supersat. profile also */
        sprintf (avprofname, "SS_%sAV%s.txt", output_v->id_string, bp->ctrl->fn_base);
        if ((avfp = fopen (avprofname, open_action)) == NULL) {
          fprintf (stderr, "Error: can't open %s. output file [%s]\n", output_v->id_string, command);
          return;
        } else {
          fprintf (avfp, "%s SUPERSATURATION OUTPUT\n", output_v->id_string);
          write_excel_header (avfp, bp);
          fprintf (avfp, "Step,Time,");
          for (i = 0; i < bp->nc[0]; i++) {
            fprintf (avfp, "%f,", i * bp->size_c[0]);
          }
          fprintf (avfp, "\n");
        }
        fclose (avfp);

      }                         /* end supersat profile bit */
#endif /*PRINT_AV_PROFILE */
      break;
    case STEP_CALL:
      conc_prof = bp->ctrl->conc_prof;
      sbnum = conc_prof[0];
      if (bp->sb[sbnum]->open == FALSE) /*sb not active yet */
        return;

      partcoef = output_v->part_coef;
      fp = output_v->fd_exel;
      /*fprintf(fp,"step,%i,time,%.5g",bp->step,bp->sim_time); */
      if (sbnum > bp->ntsb && !msg[0]) {
        fprintf (stderr, "ERROR: ConcProfile sb %i out of range\n", sbnum);
        fprintf (fp, "\nERROR: ConcProfile sb %i out of range\n", sbnum);
        msg[0] = TRUE;
        return;
      }
      if (conc_prof[1] >= bp->nc[2] && !msg[2]) {
        fprintf (stderr, "ERROR: ConcProfile slice %i out of range\n", conc_prof[1]);
        fprintf (fp, "\nERROR: ConcProfile slice %i out of range\n", conc_prof[1]);
        msg[2] = TRUE;
        return;
      }
      if (conc_prof[2] > bp->nc[1] && !msg[1]) {
        fprintf (stderr, "ERROR: ConcProfile row %i out of range\n", conc_prof[2]);
        fprintf (fp, "\nERROR: ConcProfile row %i out of range\n", conc_prof[2]);
        msg[1] = TRUE;
        return;
      }
      if (msg[0] || msg[1] || msg[2])
        return;
      sol = output_v->block_array[sbnum];
      if (bp->ctrl->scheil == TRUE) {
        fs = bp->sb[sbnum]->sch_fs;
      } else {
        fs = bp->sb[sbnum]->c_fs;
      }
      start = bp->nc[0] * bp->nc[1] * conc_prof[1] + bp->nc[0] * conc_prof[2];
      sp = sol + start;
      fsp = fs + start;
      fprintf (fp, "%i,%f,", bp->step, bp->sim_time);
      for (i = 0; i < bp->nc[0]; i++) {
        if (*fsp == NOT_CASTING) {
          fprintf (fp, "-1 ,");
        } else {
          fprintf (fp, "%.10f,", *sp);
        }
        sp++;
        fsp++;
      }
      fprintf (fp, "\n");
#ifdef PRINT_AV_PROFILE
      {
        int jj, kk;
        CA_FLOAT av = 0, ss = 0, gas = 0, alloy = 0, fs = 0, cell_temp = 0;
        int num = 0;
        int index_ca;

        sprintf (avprofname, "%sAV%s.txt", output_v->id_string, bp->ctrl->fn_base);
        if ((avfp = fopen (avprofname, "a")) == NULL) {
          fprintf (stderr, "Error: can't open %s. output file [%s]\n", output_v->id_string, command);
          return;
        }

        fprintf (avfp, "%i,%f,", bp->step, bp->sim_time);
        for (i = 0; i < bp->nc[0]; i++) {
          av = 0;
          num = 0;
          for (jj = 0; jj < bp->nc[1]; jj++) {
            for (kk = 0; kk < bp->nc[2]; kk++) {
              av += *(sol + i + jj * bp->nc[0] + kk * bp->nc[0] * bp->nc[1]);
              num++;
            }
          }
          av /= (CA_FLOAT) num;
          fprintf (avfp, "%.10f,", av);

        }
        fprintf (avfp, "\n");

        fclose (avfp);
        if (strcmp ("G_", output_v->id_string) == 0) {
          /* do a supersat. profile also */
          sprintf (avprofname, "SS_%sAV%s.txt", output_v->id_string, bp->ctrl->fn_base);
          if ((avfp = fopen (avprofname, "a")) == NULL) {
            fprintf (stderr, "Error: can't open %s. output file [%s]\n", output_v->id_string, command);
            return;
          }
          fprintf (avfp, "%i,%f,", bp->step, bp->sim_time);

          /* average over each i=const slice */
          for (i = 0; i < bp->nc[0]; i++) {
            av = 0;
            num = 0;
            for (jj = 0; jj < bp->nc[1]; jj++) {
              for (kk = 0; kk < bp->nc[2]; kk++) {
                /* the array index of the current cell */
                index_ca = i + jj * bp->nc[0] + kk * bp->nc[0] * bp->nc[1];
                cell_temp = bp->sb[sbnum]->c_temp[index_ca];
                gas = *(sol + index_ca);
                /*if (bp->ctrl->diffuse_alloy_multi == FALSE) {*/
                if((bp->ctrl->diffuse_alloy_multi == FALSE)&&(bp->ctrl->diffuse_alloy_poly == FALSE)){
                  alloy = *(bp->sb[sbnum]->c_sol_alloy + index_ca);
                  fs = *(bp->sb[sbnum]->c_fs + index_ca);
                  ss = gas / (find_sat (&(bp->mprops), cell_temp + STD_TEMP, alloy, fs));
                  av += ss;
                }
                num++;
              }
            }
            av /= (CA_FLOAT) num;
            fprintf (avfp, "%.10f,", av);

          }
          fprintf (avfp, "\n");
          fclose (avfp);

        }
        /* end supersat profile bit */
      }
#endif /*PRINT_AV_PROFILE */

      break;
    case LAST_CALL:
#ifdef PRINT_AV_PROFILE
#endif /*PRINT_AV_PROFILE */
      fclose (output_v->fd_exel);
      break;
    default:
      fprintf (stderr, "ERROR: write_conc_profile: it should be impossible to reach this line.\n");
      fprintf (stderr, "Something is seriously wrong.\n");
      exit (2);
      break;
    }                           /*end of switch stat flag */
  }
}                               /*end of write execl conc. */

/********************************************************/
/* write_conc_prof_multi                                      */
/*    write a concentration profile to excel file        */
/********************************************************/
/********************************************************/
void write_conc_prof_multi (int stat_flag, BB_struct * bp, Value_struct * output_v, int j)
{
  int start, i, *conc_prof, sbnum;
  char command[MAX_STRING_LEN];
  static int msg[3] = { 0, 0, 0 };
  CA_FLOAT *fs, *fsp, *sol, *sp, partcoef;
  FILE *fp;                     /*file pointer */
  MultiS_struct *ms;

  ms = &(bp->MultiSvals);

  switch (stat_flag) {

  case FIRST_CALL:
    sprintf (command, "%s%s_multi.txt", output_v->id_string, bp->ctrl->fn_base);
    if ((output_v->fd_exel = fopen (command, "w")) == NULL) {
      fprintf (stderr, "Error: can't open %s. output file [%s]\n", output_v->id_string, command);
      return;
    } else {
      fp = output_v->fd_exel;
      fprintf (fp, "%s CONCENTRATION OUTPUT for MULTICOMP\n", output_v->id_string);
      write_excel_header (fp, bp);
      fprintf (fp, "Step,Time,");
      for (i = 0; i < bp->nc[0]; i++) {
        fprintf (fp, "%f,", i * bp->size_c[0]);
      }
      fprintf (fp, "\n");
    }
    break;
  case STEP_CALL:
    conc_prof = bp->ctrl->conc_prof;
    sbnum = conc_prof[0];
    if (bp->sb[sbnum]->open == FALSE)   /*sb not active yet */
      return;

    fp = output_v->fd_exel;
    fprintf (fp, "Profile started at step,%i,time,%.5g \n", bp->step, bp->sim_time);
    if (sbnum > bp->ntsb && !msg[0]) {
      fprintf (stderr, "ERROR: ConcProfile sb %i out of range\n", sbnum);
      msg[0] = TRUE;
      return;
    }
    if (conc_prof[1] >= bp->nc[2] && !msg[2]) {
      fprintf (stderr, "ERROR: ConcProfile slice %i out of range\n", conc_prof[1]);
      fprintf (fp, "ERROR: ConcProfile slice %i out of range\n", conc_prof[1]);
      msg[2] = TRUE;
      return;
    }
    if (conc_prof[2] > bp->nc[1] && !msg[1]) {
      fprintf (stderr, "ERROR: ConcProfile row %i out of range\n", conc_prof[2]);
      fprintf (fp, "ERROR: ConcProfile row %i out of range\n", conc_prof[2]);
      msg[1] = TRUE;
      return;
    }
    if (msg[0] || msg[1] || msg[2])
      return;
    sol = output_v->block_array[sbnum];
    if (bp->ctrl->scheil == TRUE) {
      fs = bp->sb[sbnum]->sch_fs;
    } else {
      fs = bp->sb[sbnum]->c_fs;
    }
    start = bp->nc[0] * bp->nc[1] * conc_prof[1] + bp->nc[0] * conc_prof[2];
    sp = sol + start;
    fsp = fs + start;
    fprintf (fp, "%i,%f,", bp->step, bp->sim_time);
    for (i = 0; i < bp->nc[0]; i++) {
      fprintf (fp, "%.10f,", *sp);
      sp++;
    }
    fprintf (fp, "\n");
    break;
  case LAST_CALL:
    fclose (output_v->fd_exel);
    break;
  default:
    fprintf (stderr, "ERROR: write_conc_profile: it should be impossible to reach this line.\n");
    fprintf (stderr, "Something is seriously wrong.\n");
    exit (2);
    break;
  }                             /*end of switch stat flag */
}                               /*end of write execl conc multi. */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_output_ex_c ()
{
  static char const rcsid[] = "$Id: output_ex.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_output_ex_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3.2.2  2006/02/01 14:16:25  lthuinet
RCS Log:#update change of the calculation of supersaturation for gas in multicomponent
RCS Log:
RCS Log:Revision 10.3.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.3.4.2  2004/03/04 11:29:25  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.3.4.1  2003/12/11 16:04:03  rcatwood
RCS Log:Branch for developing the mould solute source function
RCS Log:
RCS Log:Revision 9.3  2003/10/16 11:29:24  rcatwood
RCS Log:Changed incorrect use of con_cast to use_cell_temp
RCS Log:Added icc support in Makefile
RCS Log:
RCS Log:Revision 9.2  2003/09/08 17:13:33  rcatwood
RCS Log:changed ca_feedback to use the average of the nearest cells for each node.
RCS Log:Fixed som warnings (implicit declarations)
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:38  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.9  2003/08/14 14:18:03  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.1.8.8  2003/06/26 12:55:36  rcatwood
RCS Log:Fixed a subtle allocatoin bug
RCS Log:
RCS Log:Revision 8.1.8.7  2003/05/06 15:48:07  rcatwood
RCS Log:Altered linear binary phase diagram usage to consistently use the values input from the control files rather than the header files.
RCS Log:
RCS Log:Revision 8.1.8.6  2003/03/14 16:41:47  hdong
RCS Log:Fixed temperature option selection for re-start.
RCS Log:Fixed output_excel for restart case
RCS Log:
RCS Log:Revision 8.1.8.5  2003/02/27 23:04:38  rcatwood
RCS Log:Removed use of old temperature routines , all temperatures shoudl
RCS Log:be determined by checking the array c_temp in teh subblock, if the
RCS Log:subblock is open
RCS Log:
RCS Log:Revision 8.1.8.4  2003/02/26 18:44:48  rcatwood
RCS Log:Modified (non-decentered) so that the temperature uses an array
RCS Log:calcuated before each step. Exception: Procast mode still overrides
RCS Log:and uses Ali's routine.
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:44  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:01  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.2  2003/01/14 16:22:25  rcatwood
RCS Log:Removed many lint warnings from sb_decentered_step
RCS Log:Added signal function to ca_procast
RCS Log:Removed some unused files
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:47  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.15.4.2  2002/11/04 11:22:05  rcatwood
RCS Log:Check-in after Ali has gone
RCS Log:
RCS Log:Revision 7.15.4.1  2002/08/27 14:18:17  chirazi
RCS Log:adding files for multi-component-Procast version of CA
RCS Log:
RCS Log:Revision 7.13  2001/08/28 19:08:51  rcatwood
RCS Log:thesis version
RCS Log:
RCS Log:Revision 7.12  2001/07/17 15:57:50  rcatwood
RCS Log:added supersat average contour
RCS Log:
RCS Log:Revision 7.11  2001/07/12 17:41:49  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 7.10  2001/07/06 15:18:04  rcatwood
RCS Log:fixed another bug which caused seg fault when no pore mode selected
RCS Log:
RCS Log:Revision 7.9  2001/05/31 16:24:45  rcatwood
RCS Log:changed M macro to PD_SLOPE (Easier to find!)
RCS Log:
RCS Log:Revision 7.8  2001/04/06 11:56:19  rcatwood
RCS Log:added header info for RANSLEY/IMAB
RCS Log:
RCS Log:Revision 7.7  2001/03/27 11:31:50  rcatwood
RCS Log:Protected error messages from excessive numbers
RCS Log:
RCS Log:Revision 7.6  2001/03/13 10:36:46  rcatwood
RCS Log:Ready to Merge with anchou version
RCS Log:Fixed not-casting problems,
RCS Log:seperate pore stats file.
RCS Log:
RCS Log:Revision 7.5  2001/02/19 21:35:25  rcatwood
RCS Log:fixed more histo,
RCS Log:also found memory allocation bug, not sur eif it is fixed.
RCS Log:
RCS Log:Revision 7.4  2001/02/19 19:28:47  rcatwood
RCS Log:fixed histo
RCS Log:for grains
RCS Log:
RCS Log:and also make TcTrace mode override const. cooling rate
RCS Log:
RCS Log:Revision 7.3  2001/02/16 20:30:49  rcatwood
RCS Log:Added some commenting for DAS and Limrad
RCS Log:Removed some JUNK
RCS Log:
RCS Log:Revision 7.2  2001/02/15 15:15:52  rcatwood
RCS Log:Improved STAT_ONLY mode. Added pore extent output.
RCS Log:
RCS Log:Revision 7.1  2000/12/06 21:10:40  rcatwood
RCS Log:fixed up heatfolw, tctrace
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.2  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.1  2000/10/09 16:57:04  rcatwood
RCS Log:Before changing nucleation
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:35  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.1  2000/09/25 17:34:29  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.6  2000/07/11 16:32:13  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.5  2000/06/29 14:31:02  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.4  2000/06/27 11:41:19  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.3  2000/06/12 16:23:30  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.2  2000/06/06 18:43:04  rcatwood
RCS Log:Changed r start and growth of pores -- pore radius equilibrium
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.3  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.2  2000/03/15 18:43:37  rcatwood
RCS Log:improved directional temp calc of pores
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.8  2000/02/16 10:51:47  rcatwood
RCS Log:Fixed grain structure
RCS Log:
RCS Log:Revision 4.7  2000/02/15 15:29:11  rcatwood
RCS Log:Version after McWasp submitted
RCS Log:
RCS Log:Revision 4.5  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.3  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.2  1999/12/16 19:15:49  rcatwood
RCS Log:Alloy and gas diffusion working, pores working. Changed file name for conc. output A for alloy C for gas conc.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
