/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/*RCS Id:$Id: read_sb.c 953 2006-11-09 13:39:43Z rcatwood $*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "machine.h"
#include "blocks.h"
extern int init_new_grain (BB_struct * bp, int igr, int sbnum, int xcell, int ycell, int zcell, int nc);

/* read in a binary block of memory into a subblock - 8 bit data of grain-number*/
int read_bin_sb (BB_struct * bp, int sbnum)
{
  int errors = 0, i;
  int num_data, nsolid = 0;
  unsigned char *data, *dataorig;
  CA_FLOAT *fsp;
  int *gp;
  FILE *fp;

  gp = bp->sb[sbnum]->gr;
  fsp = bp->sb[sbnum]->c_fs;
  num_data = bp->ncsb;

  fprintf (stderr, "Entering read_bin_sb, sb %i \n", sbnum);
  if ((fp = fopen (bp->ctrl->fn_inp, "r")) == NULL)
    fprintf (stderr, "ERROR:read_bin_sb: can't open input file [%s]\n", bp->ctrl->fn_inp);
  bp->ctrl->fd_inp = fp;

  if (!(data = (unsigned char *) calloc (num_data, sizeof (unsigned char))))
    fprintf (stderr, "ERROR:read_bin_sb: Couldn't calloc data space.\n");
  dataorig = data;

  if (fread (data, sizeof (unsigned char), num_data, fp) == NULL)
    fprintf (stderr, "ERROR:read_bin_sb: Couldn't read subblock binary data\n");

  for (i = 0; i < num_data; i++) {
    *gp = (int) *data;
    #ifndef RECRYSTALLIZE
       /* Leave the grain numbers as they are for RECRYSTALLIZE */
       /* otherwise assign 255 not NOT_CASTING, 0 to liquid */
       /* and set the grain number for the other grains*/
       if (*gp == 255)
         *fsp = NOT_CASTING;
       else if (*gp == 0)
         *fsp = LIQUID;
       else {
         *fsp = .9999;
         nsolid++;
       }
    #endif

    gp++;
    data++;
    fsp++;
  }

  bp->sb[sbnum]->Tvals.fsavg += (((CA_FLOAT) nsolid) / bp->ncsb);
  if (fclose (fp) != 0)
    fprintf (stderr, "ERROR:read_bin_sb:couldn't fclose\n");
  free (dataorig);
  return (errors);

}                               /*end of read_bin_sb */

/* just read in a single slice for pseudo 2D purposes */
int read_bin_sb_slice (BB_struct * bp, int sbnum)
{
  int errors = 0, i, slice = 0;
  int num_slices, num_data, nsolid = 0;
  unsigned char *data = NULL, *datap = NULL;
  CA_FLOAT *fsp, *fs;
  int *gp, *g;
  FILE *fp;

  fprintf (stderr, "Entering read_bin_sb_slice, sb %i slice%i\n", sbnum, slice);
  num_data = bp->nc[0] * bp->nc[1];
  num_slices = bp->nc[2];
  g = gp = ((bp->sb[sbnum]->gr) + (slice * num_data));
  fs = fsp = ((bp->sb[sbnum]->c_fs) + (slice * num_data));

  if ((fp = fopen (bp->ctrl->fn_inp, "r")) == NULL)
    fprintf (stderr, "ERROR:read_bin_sb_slice: can't open input file [%s]\n", bp->ctrl->fn_inp);
  bp->ctrl->fd_inp = fp;

  if (!(datap = data = (unsigned char *) malloc (num_data * sizeof (unsigned char))))
    fprintf (stderr, "ERROR:read_bin_sb_slice: Couldn't calloc data space.\n");

  if (fread (data, sizeof (unsigned char), num_data, fp) == NULL)
    fprintf (stderr, "ERROR:read_bin_sb_slice: Couldn't read subblock binary data\n");

  for (i = 0; i < num_data; i++) {
    *gp = (int) *data;

       /* assign 255 to NOT_CASTING, 0 to liquid */
       /* and set the grain number for the other grains*/
    if (*gp == 255)
      *fsp = NOT_CASTING;
    else if (*gp == 0)
      *fsp = LIQUID;
    else {
      *fsp = .9999;
      nsolid++;
    }

    gp++;
    data++;
    fsp++;

  }
  /* copy the input slice to all slices */
  for (i = 1; i < num_slices; i++) {
    memcpy ((g + i * num_data), g, num_data * sizeof (int));
    memcpy ((fs + i * num_data), fs, num_data * sizeof (CA_FLOAT));
  }
  nsolid *= num_slices;

  bp->sb[sbnum]->Tvals.fsavg += (((CA_FLOAT) nsolid) / bp->ncsb);
  fclose (fp);
  free (datap);

  return (errors);
}                               /*end of read_bin_sb_slice */

/* fix up the grain structures after reading in a subblock */
int fix_grains (BB_struct * bp, int sbnum)
{
  double dtmp;
  int errors = 0;
  int ngrains, nbound = 0;
  int i, j, k;
  int grainlist[256], ncells[256];
  int xcells[256], ycells[256], zcells[256];
  int num_data;
  int *gp;                      /*Grain Pointer */
  int *nucp;                    /*NUCleation list Pointer */
  int *elp;                     /*ELement Pointer used as recr. flag for g.b. */
  CA_FLOAT *fs, *fsp;

  for (i = 0; i < 256; i++)
    grainlist[i] = ncells[i] = xcells[i] = ycells[i] = zcells[i] = 0;

  ngrains = bp->nprops.ngr;
  gp = bp->sb[sbnum]->gr;       /*rewind */
  fsp = fs = bp->sb[sbnum]->c_fs;
  elp = bp->sb[sbnum]->c_elm;   /*rewind */
  num_data = bp->ncsb;

  for (k = 0; k < bp->nc[2]; k++) {
    for (j = 0; j < bp->nc[1]; j++) {
      for (i = 0; i < bp->nc[0]; i++) {
        {
          if (*gp <= 1) {
            *gp = 0;
          } else if (*gp == 255) {
            *gp = NOT_CASTING;
            *fsp = NOT_CASTING;
          } else if (grainlist[*gp] == 0) {
            ngrains++;
            grainlist[*gp] = ngrains;
            xcells[ngrains] = i;
            ycells[ngrains] = j;
            zcells[ngrains] = k;
          }
          fsp++;
          gp++;
          elp++;
        }
      }
    }
  }
  gp = bp->sb[sbnum]->gr;       /*rewind grain */
  elp = bp->sb[sbnum]->c_elm;   /*rewind element */
  for (i = 0; i < num_data; i++) {
    /* renumber all the grains avoiding any missing numbers */
    if (*gp != NOT_CASTING) {
      *gp = grainlist[*gp];
      ncells[*gp]++;
    }
    gp++;
    elp++;

  }
  for (i = bp->nprops.ngr; i <= ngrains; i++) {
    Ind_grain *g;
    CA_FLOAT a0, a1, a2;
    CA_FLOAT c0, s0, c1, s1, c2, s2;

    init_new_grain (bp, i, sbnum, xcells[i], ycells[i], zcells[i], ncells[i]);
    g = bp->gr[i];
    /* generate rotate matrix for grain */
    /* added by Wei WANG on 20-09-02 */
    /* copied here as quick way to choose angle for */
    /* grain which is read in from the image file */
    /**  \todo  improve the input for grain angle -- decentred - maybe obsolete */
    if (bp->ctrl->decentred_octahedron == TRUE) {
#ifdef XY_ANGLE
      a0 = XY_ANGLE;
#else
      a0 = dtmp;                /* need to be changed */
#endif /*XY_ANGLE */
      a1 = 0.0;
      a2 = 0.0;
      g->ang[0] = a0;
      g->ang[1] = a1;
      g->ang[2] = a2;

      c0 = g->cang[0] = cos (g->ang[0]);
      s0 = g->sang[0] = sin (g->ang[0]);
      c1 = g->cang[1] = cos (g->ang[1]);
      s1 = g->sang[1] = sin (g->ang[1]);
      c2 = g->cang[2] = cos (g->ang[2]);
      s2 = g->sang[2] = sin (g->ang[2]);

      /* rotation matrix */
      g->g[0][0] = c2 * c0 - s2 * s1 * s0;
      g->g[1][0] = c2 * s0 + s2 * c1 * c0;
      g->g[2][0] = s2 * s1;

      g->g[0][1] = -1. * s2 * c0 - c2 * c1 * s0;
      g->g[1][1] = -1. * s2 * s0 + c2 * c1 * c0;
      g->g[2][1] = c2 * s1;

      g->g[0][2] = s1 * s0;
      g->g[1][2] = -1. * s1 * c0;
      g->g[2][2] = c1;
    }

  }
  bp->nprops.ngr = ngrains;
  return (errors);
}                               /*endof fix_grains */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_read_sb_c ()
{
  static char const rcsid[] = "$Id: read_sb.c 953 2006-11-09 13:39:43Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_read_sb_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.2  2006/11/09 13:39:40  rcatwood
RCS Log:Merged the update for ca_procast version for procast 2006.0_beta
RCS Log:
RCS Log:Revision 11.1.12.1  2006/11/09 13:29:19  rcatwood
RCS Log:added a few comments
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 9.1.12.1  2004/08/19 14:38:14  xly
RCS Log:read in 3d mould image into sb.
RCS Log:Revision 8.6.8.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:39  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.6.2.4  2003/02/14 17:19:42  xly
RCS Log:Fixed bad free of data pointer
RCS Log:
RCS Log:Revision 8.6.2.3  2003/01/22 16:53:46  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.6.2.2  2003/01/15 19:02:01  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.2.4.1  2003/01/15 17:40:35  rcatwood
RCS Log:merged changes from trunk (8.6 version)
RCS Log:
RCS Log:Revision 8.6  2003/01/15 14:05:15  xly
RCS Log:Fixed grain reading routine to be compatible with Linux
RCS Log:Unsigned Char data type is needed
RCS Log:
RCS Log:Revision 8.5  2003/01/10 18:35:02  xly
RCS Log: Fixed - read_sb did not have math library header file
RCS Log: GD library location for sgi fixed in image output
RCS Log:
RCS Log:Revision 8.4  2003/01/10 12:22:44  rcatwood
RCS Log:dtmp was not declared! fixed the problem.
RCS Log:
RCS Log:Revision 8.3  2003/01/09 18:38:34  rcatwood
RCS Log:added quick angle choice (compile time) for grain read from image
RCS Log:using XY_ANGLE definition
RCS Log:
RCS Log:Revision 8.2  2002/10/24 18:10:55  rcatwood
RCS Log:Incorporated NOT_CASTING mode with decentered octahedron
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:03  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.1  2002/10/17 16:52:38  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.0.14.1  2002/09/03 13:31:59  rcatwood
RCS Log:Merged with reorganized allocation routines, and adjusted nucleation to compartmentalize
RCS Log:the grain information assignment.
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.1  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.2  2000/07/11 16:33:44  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.5  2000/02/15 15:29:11  rcatwood
RCS Log:Version after McWasp submitted
RCS Log:
RCS Log:Revision 4.4  2000/01/27 12:18:48  rcatwood
RCS Log:Overgrowth addressed. Bindump of t, fs
RCS Log:
RCS Log:Revision 4.3  2000/01/10 14:57:00  rcatwood
RCS Log:Quick SIGTERM handler in ca_wrapper
RCS Log:
RCS Log:Revision 4.2  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
