/** \file sb_ca_step.c */
/****************************************************************/
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      All Rights Reserved                                     */
/*      THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM    */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      This material contains CONFIDENTIAL INFORMATION that    */
/*      is the property of Imperial College. Any use,           */
/*      duplication or disclosure not specifically authorized   */
/*      by Imperial College is strictly prohibited.             */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/****************************************************************/
/*      sb_ca_step.c:                                           */
/*                                                              */
/* The main subroutine to calculate growth and extend into new  */
/* cells.:                                                      */
/*                                                              */
/*                                                              */
/*                                                              */
/****************************************************************/
/*RCS ID: $Id: sb_ca_step.c 957 2006-11-09 20:23:51Z rcatwood $*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "machine.h"
#include "blocks.h"
#include "ca_matrix.h"
#include "props.h"
/* prototypes for the nucleation function choices */
#include "nucfuncs.h"

/* Prototype defs */
extern void add_to_grain (Ind_grain * gp, int i, int j, int k);

/* from p_growth.c */
extern CA_FLOAT growth_primary (CA_FLOAT gg_const, CA_FLOAT gg_cub, CA_FLOAT dt, CA_FLOAT Tunder, int eut_flag, CA_FLOAT cell_fs_n_eut,
                                CA_FLOAT cell_fs_b_eut);
extern CA_FLOAT growth_eutectic (CA_FLOAT gg_const, CA_FLOAT gg_cub, CA_FLOAT dt, CA_FLOAT Tunder, int eut_flag,
                                 CA_FLOAT cell_fs_n_eut, CA_FLOAT cell_fs_b_eut);
CA_FLOAT particle_growth (CA_FLOAT pg_const_a, CA_FLOAT pg_const_b, CA_FLOAT dt, CA_FLOAT Tunner, CA_FLOAT p_conc);

/*from trans_interp_calc.c*/
extern CA_FLOAT trans_interp_calc (FGrid_str * fg, BB_struct * bp, int sbnum, int x, int y);

/* from sb_temp_calc.c */

#ifdef CHIRAZI_MULTI
extern CA_FLOAT cell_liq_calc_multi (CA_FLOAT ** conc_multi, int index_local, BB_struct * bp);
extern int interpolate (Ctrl_str * cp, BB_struct * bp, int cell_index, CA_FLOAT * cell_temp, int sbnum, int inter_flag,
                        CA_FLOAT cell_temperature);
#endif

extern int init_new_grain (BB_struct * bp, int igr, int sbnum, int xcell, int ycell, int zcell, int nc);

/*beginning of sb_ca_step*/

/** 
 * Do a normal ca step
 *
 * @callgraph
 * @callergraph
*/
int sb_ca_step (BB_struct * bp, int sbnum)
{
  int errors = 0, ngrowing = 0;

#ifdef OLD_TUNDER
  CA_FLOAT *otu, *otup;
#endif /*OLD_TUNDER */
  int i, j, k, n;
  int *oni, *nni;
  int *tmp_oni, *tmp_nni, tmp_n_neigh;
  int nx, ny, nz, skip;
  int inter_count;
  int inter_flag;
  SB_struct *sp = bp->sb[sbnum];
  Ind_grain *this_grain;

/******added for the multi component****/
  CA_FLOAT **conc_multi;
  MultiS_struct *ms;

/****************************************/
  CA_FLOAT Tliq, Tunder, prob, TliqFixed;
  CA_FLOAT *cell_conc;
  CA_FLOAT *c_temp_p, *ofs, *nfs, *op, *np, *sol, *solp, *sfs, *sfsp, *thr, *thrp;
  CA_FLOAT *local_dfs_primary, *local_dfs_primary_start;
  CA_FLOAT *local_dfs_eutectic, *local_dfs_eutectic_start;
  int *bin_flag, *bin_flag_start;
  CA_FLOAT sum_del_fs, dr, dfs, dfs_tmp, extra_fs;
  CA_FLOAT dr_primary, dr_eutectic, dfs_primary, dfs_eutectic;
  CA_FLOAT oldfs = 0;
  int *ngr, *ogr;
  int oriented_on, cell_nuc_on, cell_temp_on, n_neigh;

  int particle_on = bp->ctrl->particle;
  int phase_diag_on = bp->ctrl->phase_diag_on;
  int global_undercooling = bp->ctrl->use_global_undercooling;

  int eut_flag = 0;
  static int nucmsg = 0;
  CA_FLOAT sch_sum;
  int (*cell_nuc_func) (BB_struct * bp, CA_FLOAT Tunder, CA_FLOAT argthree);
  Ctrl_str *cp = bp->ctrl;

  int index_ca;

  dfs_tmp = 0;
  ms = &(bp->MultiSvals);
  sfs = sfsp = bp->ftmp_three;

  /* set up phase diagram mode */
  phase_diag_on = bp->ctrl->phase_diag_on;
  global_undercooling = cp->use_global_undercooling;
  particle_on = bp->ctrl->particle;

  if (cp->use_csol_alloy) {
    cell_conc = sp->c_sol_alloy;
    #ifdef CHIRAZI_MULTI
    conc_multi = sp->c_sol_alloy_multi; /*multi component conc array */
    #endif
  }

  /* Set up local pointers */
#ifdef OLD_TUNDER
  otup = otu = bp->old_Tunder;
#endif /*OLD_TUNDER */
  local_dfs_primary = local_dfs_primary_start = sp->cell_dfs_primary;
  local_dfs_eutectic = local_dfs_eutectic_start = sp->cell_dfs_eutectic;
  op = ofs = bp->ftmp_one;
  bin_flag = bin_flag_start = bp->bin_flag;
  /*pop = pofs = sp->fs_n_eut; */
  /* eop = eofs = sp->fs_b_eut; */
  np = nfs = sp->c_fs;
  solp = sol = sp->c_sol;
  ogr = bp->itmp_one;
  ngr = sp->gr;
  c_temp_p = sp->c_temp;        /* cell temperature array pointer */

  /* dummy ref not used for cell_nuc_func,            */
  /* should be old Tunder when needed or threshold,   */
  /* but has to be something if neither mode is on!   */
  thr = thrp = sp->c_sol;

  nx = bp->nc[0];
  ny = bp->nc[1];
  nz = bp->nc[2];

  /*set up local values */
  skip = 2 * (nx + 2);
  oriented_on = bp->nprops.oriented;
  TliqFixed = bp->mprops.Tliq;
  Tliq = TliqFixed;
  sum_del_fs = 0.0;

  /* Set up option flags */

  cell_nuc_on = cp->use_cell_nuc;

  if (cell_nuc_on) {            /*Select the cell nuc function -- should move to higher level! */
    switch (bp->nprops.nmodel) {
    case N_RAPPAZ:             /* use the PDL coded Rappaz based method, 'rolling dice' per cell... */
      cell_nuc_func = cell_nuc;
      break;
    case N_DIST:               /* use the DISTRIBUTION lookup-table method */
      cell_nuc_func = dist_cell_nuc;
      break;
    case N_HETERO:
      fprintf (stderr, "ERROR: sb_ca_step: cannot find cell_nuc function for N_HETERO model\n");
      exit (0);
      break;
    case N_RAP_DEL:
#           ifndef OLD_TUNDER
      fprintf (stderr, "ERROR:sb_ca_step: At present you have to define OLD_TUNDER to use N_RAP_DEL nucleation model.\n");
      exit (0);
#           endif /*OLD_TUNDER */
      cell_nuc_func = del_cell_nuc_norate;
      break;
    case N_OLDF_DEL:
      fprintf (stderr, "ERROR: sb_ca_step: cannot find cell_nuc function for N_OLDF_DEL model\n");
      exit (0);
      break;
    case N_BLOCK:              /* use method of setting a threshold for each cell */
      /* similar to the method used for pores */
      cell_nuc_func = block_cell_nuc;
      thr = thrp = sp->c_nuc_thresh;
      break;
    default:
      fprintf (stderr, "ERROR: sb_ca_step: cannot find cell_nuc function; Nucleation model undefined.(%i)\n", (bp->nprops.nmodel));
      exit (0);
      break;
    }
  }

  /*end of set up cell nuc function */
  /*   if (bp->nprops.nmodel == N_DIST) cell_nuc_func = dist_cell_nuc; */
  /*   else cell_nuc_func = cell_nuc; */
  /* set up temporary neighbourhood */
  n_neigh = tmp_n_neigh = bp->ctrl->n_neigh;
  switch (n_neigh) {
  case NEIGH_6:
    oni = bp->nbhd.onq;
    nni = bp->nbhd.nnq;
    break;
  case NEIGH_8:
    oni = bp->nbhd.onhl;
    nni = bp->nbhd.nnhl;
    break;
  case NEIGH_10:
    oni = bp->nbhd.ono;
    nni = bp->nbhd.nno;
    break;
  case NEIGH_26:
    oni = bp->nbhd.onc;
    nni = bp->nbhd.nnc;
    break;
  }
  tmp_oni = oni;
  tmp_nni = nni;
  /* end set up temporary neighbourhood */

  /*sciel fraction solid */
  sfsp = sfs + bp->cubeptr.flist[0][START];

  /*outside frac solid pointer */
  op = ofs + bp->cubeptr.flist[0][START];
  bin_flag = bin_flag_start + bp->cubeptr.flist[0][START];
  local_dfs_primary = local_dfs_primary_start;
  local_dfs_eutectic = local_dfs_eutectic_start;
  /*pop = pofs; */
  /* eop = eofs; */

  /*outside grain */
  ogr = ogr + bp->cubeptr.flist[0][START];

  /*nside pointer */
  np = nfs;

  /*keep track of scheil frac solid seperately */
  if (bp->ctrl->scheil == TRUE) {
    sch_sum = 0;
  }

   /***********************************************/
  /* Beginning of main loop(s)                   */
   /***********************************************/

  sp->ncsolid = 0;

  for (k = 0, index_ca = 0; (k < nz && index_ca < bp->tnc[0] * bp->tnc[1] * bp->tnc[2]); k++) {
    for (j = 0; j < ny; j++) {
      for (i = 0; i < nx; i++) {

        /*possibly unroll this section? */
        if (phase_diag_on) {
          eut_flag = FALSE;
          if ((cp->diffuse_alloy_multi == FALSE) && (cp->diffuse_alloy == TRUE)) {
            Tliq = cell_liq_calc (*cell_conc, &(bp->mprops));
          }
          /* Multi component */
          if (cp->diffuse_alloy_multi == TRUE) {

            /** \todo integrate Ludovic Thuinets' poly component into non-decentered version */
            #ifdef CHIRAZI_MULTI
            if (bp->ctrl->thermocalc == FALSE) {
              Tliq = cell_liq_calc_multi (conc_multi, index_ca, bp);
            } else {
              inter_flag = LIQUIDUS;
              inter_count = interpolate (cp, bp, index_ca, &Tliq, sbnum, inter_flag, *c_temp_p);
            }
            #endif /*CHIRAZI_MULTI*/
          }
          /* end Multi component */
          if (bp->ctrl->diffuse_alloy_multi == FALSE && cp->diffuse_alloy_poly == FALSE) {
            /** \todo  if the gas has an effect on the solution then this needs rethinking -- multicomponent */

            /* test if cell should solidify as eutectic */ 
            if (*c_temp_p <= bp->mprops.alloyprops[0].T_eut && *cell_conc >= bp->mprops.alloyprops[0].c_eut) {
              eut_flag = TRUE;
            }
          }
        }

        /*undercooling is always Tliq-Tcell */

        Tunder = Tliq - *c_temp_p;
#         ifdef OLD_TUNDER
        if (bp->step <= 1)
          *otup = Tunder;
#         endif /*OLD_TUNDER */

          /************************************/
        /*If the cell is GREATER THAN SOLID */
        /* this shouldnt be able to happen. */
          /************************************/
        if (*np > SOLID) {
          fprintf (stderr, "ERROR:sb_ca_step: cell > solid! sb %i, i %i, j %i, k %i, step %i\n", sbnum, i, j, k, bp->step);
#ifdef ERROR_EXIT
          fprintf (stderr, "EXITING!\n");
          exit (3);
#endif
        }
          /*********************************************/
        /* check the state of the cell and apply the */
        /* appropriate algorithm                     */
          /*********************************************/
        if (*np == NOT_CASTING) {
#ifdef EXTRA_TRAPS
          int dumb;

          /*******************************************/
          /*                                         */
          /*     if the cell is NOT IN THE CASTING   */
          /*                                         */
          /*******************************************/

                    /********************/
          /* do nothing       */
                    /********************/
          dumb = 0;             /* need to be able to set a trap here */
#endif /* EXTRA_TRAPS */
        } else if (*np == SOLID) {

          /*******************************************/
          /*                                         */
          /*     if the cell is SOLID                */
          /*                                         */
          /*******************************************/

          /*calculate schiel frac solid */
          if (bp->ctrl->scheil == TRUE) {
            *sfsp = schiel (*c_temp_p);
            sch_sum += *sfsp;
            if (*sfsp >= 1.0) {
              sp->ncsolid++;
            }                   /* end if schiel fully solid */
          } /*end if schiel */
          else {
            sp->ncsolid++;
          }

          /*******************************************/
          /*                                         */
          /*     if the cell is LIQUID               */
          /*                                         */
          /*******************************************/
        } else if (*np == LIQUID) {
          if (cell_nuc_on) {
#ifdef GLOBAL_UND_NUC
            /*for global undercooling nucleation model */
            if ((global_undercooling))
              Tunder = TliqFixed - *c_temp_p;
#endif

#              ifdef OLD_TUNDER /* only used when trying to calculate threshold by integrating */
            /* the undercooling from the previous step ..  */
            if (bp->ctrl->block_nuc) {
              fprintf (stderr, "You can't do that! You have to debug the code.\n");
              fprintf (stderr, "You can't use OLD_TUNDER and block_nuc at the same time\n.");
              exit (1);
            }
            /* then cell_nuc_func needs to have the old Tunder pointer otup defined */
            if (Tunder > 0 && (*cell_nuc_func) (bp, Tunder, *otup) == 1)
#              else /*OLD_TUNDER */
            {                   /*DEBUGGING section */
              static int nnucmsg = 0;

              if (nnucmsg < MAX_NUC_MSG && Tunder > 0) {
                if (bp->ctrl->diffuse_alloy_multi == FALSE) {
                  fprintf (stderr, "cell temp[%g] cell conc[%g] cell Tliq[%g]\n", *c_temp_p, phase_diag_on ? *cell_conc : 0, Tliq);
                }
                nnucmsg++;
              }
            }                   /*end DEBUGGING section */
            if (Tunder > 0 && (*cell_nuc_func) (bp, Tunder, *thrp) == 1)
#              endif /*OLD_TUNDER */
            {                   /* trheshold is exceeded so ... */

               /*********************/
              /* Nucleate a Grain  */
               /*********************/

              /* error checks */
              if (nucmsg < MAX_NUC_MSG) {
                if (bp->ctrl->diffuse_alloy_multi == FALSE) {
                  fprintf (stderr, "cell temp[%g] cell conc[%g] cell Tliq[%g]\n", *c_temp_p, phase_diag_on ? *cell_conc : 0, Tliq);
                }
                nucmsg++;
              }
              if (bp->nprops.ngr == bp->nprops.gd_max_total) {
                fprintf (stderr, "ERROR: sb_ca_step - Max # grains set by user [%d] exceeded. Increase option MaxTotGrains.\n",
                         bp->nprops.gd_max_total);
              } else {
                /* no errors */
                /* create new grain and set up */
                /*record nucleation values in structure */
                *op = EMBRYO;
                (bp->nprops.ngr)++;
                (bp->sb[sbnum]->ngr)++;
                *ogr = bp->nprops.ngr;
                /*
                   init_new_grain(bp, *ogr, sbnum, i + j * nx + k * nx * ny, 1);
                 */
                init_new_grain (bp, *ogr, sbnum, i, j, k, 1);
                this_grain = bp->gr[*ogr];
#ifdef DBM
                /* diagnostic output from dbmalloc (debug malloc) library */
                {
                  static int dbm_trapgrain = 0;

                  fprintf (stderr, "grain created,trap,num,cell,mem:,%i,%i,%i,%x\n", dbm_trapgrain, this_grain->num,
                           i + j * nx + k * nx * ny, this_grain);
                  dbm_trapgrain++;
                }
#endif           /*DBM*/
                  this_grain->TNuc = *c_temp_p;
                this_grain->TunderNuc = Tunder;
                this_grain->NucTh = *thrp;
                if (bp->ctrl->diffuse_alloy_multi == FALSE) {
                  this_grain->CellConcNuc = phase_diag_on ? *cell_conc : 0;
                }
                #ifdef CHIRAZI_MULTI
                if (bp->ctrl->diffuse_alloy_multi == TRUE) {
                  this_grain->CellConcNuc = phase_diag_on ? conc_multi[0][index_ca] : 0;
                }
                #endif

              }                 /* end create new grain and set up */
            }                   /* end nucleation threshold test */
          }                     /* end if cell nuc on */
          if (bp->ctrl->scheil == TRUE) {
            *sfsp = 0;
          }
#ifdef OLD_TUNDER
          *otup = Tunder;       /*save the old Tunder */
#endif /*OLD_TUNDER */

        } else {                /* end of LIQUID section */
          /*******************************************/
          /*                                         */
          /*     if the cell is GROWING              */
          /*                                         */
          /*******************************************/
          /*The cell must be growing */
          ngrowing++;
          if (phase_diag_on) {
             /*******************************************/
            /* Phase Diagram -- Must consider growth   */
            /* fixing later by diffuse subroutine.     */
            /* Also use growth pct.-- grow when not    */
            /* fully solid.                            */
             /*******************************************/
#ifdef GLOBAL_UND_GRO
            /*for global undercooling growth model */
            if ((global_undercooling))
              Tunder = TliqFixed - *c_temp_p;
#endif

            /*growth will be fixed in diffuse subroutine */
            if (Tunder > 0) {   /*calculate growth rate per cell - repeated below */
              if (bp->ctrl->particle == TRUE) { /*replace with function pointer later */
                dr = particle_growth (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder, *cell_conc);
                /* use fixed initial concentration of particles in this case */
              } else {
                if (bp->ctrl->diffuse_alloy_multi == 0) {
                  dr = growth (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder);
                  dfs = dr / bp->size_c[0];
                  dfs_tmp = oriented_on ? (dfs * (bp->gr[*ngr]->gro_fact)) : dfs;
                  sum_del_fs += dfs_tmp;
                  *op += dfs_tmp;
                } else {
                  if (*bin_flag == 0) {
                    dr_primary =
                      growth_primary (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder, *bin_flag, sp->fs_n_eut[index_ca],
                                      sp->fs_b_eut[index_ca]);
                    dfs_tmp = dfs_primary = dr_primary / bp->size_c[0];
                    *local_dfs_primary = dfs_tmp;
                    sum_del_fs += dfs_tmp;
                    /* *pop += dfs_primary; */
                    *op += dfs_tmp;
                    if (*op > SOLID) {
                      extra_fs = *op - SOLID;
                      sum_del_fs -= extra_fs;
                      /* *pop = SOLID; */
                      *op = SOLID;
                      sp->ncsolid++;
                      bp->gr[*ogr]->ngrow--;
                    }
                  } else {
                    dr_primary =
                      growth_primary (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder, *bin_flag, sp->fs_n_eut[index_ca],
                                      sp->fs_b_eut[index_ca]);
                    dr_eutectic =
                      growth_eutectic (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder, *bin_flag, sp->fs_n_eut[index_ca],
                                       sp->fs_b_eut[index_ca]);
                    dfs_primary = (dr_primary / (2.0 * bp->size_c[0]));
                    *local_dfs_primary = dfs_primary;
                    dfs_eutectic = (dr_eutectic / (2.0 * bp->size_c[0]));
                    *local_dfs_eutectic = dfs_eutectic;
                    dfs_tmp = dfs = dfs_primary + dfs_eutectic;
                    sum_del_fs += dfs_tmp;
                    /*      *pop += dfs_primary; */
                    /*      *eop += dfs_eutectic; */
                    *op += dfs_tmp;
                    if (*op > SOLID) {
                      extra_fs = *op - SOLID;
                      sum_del_fs -= extra_fs;
                      /*       *pop -=extra_fs/2.0; */
                      /*       *eop -=extra_fs; */
                      *op = SOLID;
                      sp->ncsolid++;
                      bp->gr[*ogr]->ngrow--;
                    }
                  }
                }
              }

              if (dfs > DFS_WARNING) {
                bp->dfs_warn++;
              }
              if (dfs > DFS_CAP) {
                dfs = DFS_CAP;
                bp->dfs_cap++;
              }
            } else {
              dr = 0.0;
              dfs = 0.0;
            }                   /*end caluclate growth rate per cell */

            if (*np >= bp->fsgrow) {
#include "oriented.inc"
              /*use EMBRYO if there is no extra fs */
              extra_fs = EMBRYO;
              /*special case if solid exceeded in this step */
              if (bp->ctrl->diffuse_alloy_multi == 0) {
                if (*op > SOLID) {
                  extra_fs = *op - SOLID;
                  sum_del_fs -= extra_fs;

                  if (extra_fs > CAP_GROWTH) {
                    extra_fs = CAP_GROWTH;
                    bp->extrafs_cap++;
                  }

                  *op = SOLID;
                  sp->ncsolid++;
                  bp->gr[*ogr]->ngrow--;
                }               /*end solid exceeded */
              }

              for (n = 0; n < tmp_n_neigh; n++) {       /* neighbour loop */
                if (*ogr == 0) {
                  continue;
                } else {
                  if (*(ogr + tmp_oni[n]) == LIQUID) {  /*activate neighbour */
                    *(op + tmp_oni[n]) = extra_fs;
                    *(ogr + tmp_oni[n]) = *ogr;
                    /* add the cell to the grain structure */
                    /* keeping track of its size and location */
                    add_to_grain (bp->gr[*ogr], i, j, k);
                    sum_del_fs += extra_fs;
                  }             /* end activate neigbour */
                }
              }                 /*end check neighbour loop */
            } else {            /* end if frac >= fsgrow */
              if (*op >= SOLID) {       /*annoying exception */
                /*solid has increased beyond 1.0 from below growth_pct */
                extra_fs = *op - 0.9999;        /*allow diffuse to correct */
                sum_del_fs -= extra_fs;
                *op = 0.9999;
                bp->gr[*ogr]->ngrow++;
              }                 /* end annoying exception */
            }                   /* end if frac ! >= fsgrow */
          } else {              /*phase diag not on */

             /*******************************************/
            /* Non Phase Diagram                       */
            /* Does not use growth pct. just now...    */
            /*                                         */
             /*******************************************/

            /*growth is finalised, eutectic doesn't exist, etc */
            if (cell_temp_on || bp->ctrl->procast == TRUE) {
              if (Tunder > 0) { /*calculate growth rate per cell - repeated above */
                if (bp->ctrl->particle == TRUE) {       /*replace with function pointer later */
                  dr = particle_growth (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder, *cell_conc);
                  /* use fixed initial concentration of particles in this case */
                } else {
                  dr = growth (bp->mprops.gg_const, bp->mprops.gg_cub, bp->delt, Tunder);
                }
                dfs = dr / bp->size_c[0];
                if (dfs > DFS_WARNING) {
                  bp->dfs_warn++;
                }
                if (dfs > DFS_CAP) {
                  dfs = DFS_CAP;
                  bp->dfs_cap++;
                }
              } else {
                dr = 0.0;
                dfs = 0.0;
              }                 /*end caluclate growth rate per cell */
            }                   /*end if cell temp on */
            dfs_tmp = oriented_on ? (dfs * (bp->gr[*ngr]->gro_fact)) : dfs;
            *op += dfs_tmp;
            sum_del_fs += dfs_tmp;
            if (*op >= SOLID) {
#include "oriented.inc"
              extra_fs = *op - 1.0;
              sum_del_fs -= extra_fs;
              *op = 1.0;
              bp->gr[*ogr]->ngrow--;
              if (bp->ctrl->scheil == FALSE) {
                sp->ncsolid++;
              }
              for (n = 0; n < tmp_n_neigh; n++) {       /* neighbour loop */
                if (*ogr == 0) {
                  continue;
                } else {
                  if (*(ogr + tmp_oni[n]) == LIQUID) {  /*activate neighbour */
                    *(op + tmp_oni[n]) = extra_fs;
                    *(ogr + tmp_oni[n]) = *ogr;
                    bp->gr[*ogr]->ncells++;
                    bp->gr[*ogr]->ngrow++;
                    sum_del_fs += extra_fs;
                    /*set min and max cells for grain */
                    bp->gr[*ogr]->max[0] = MAX (bp->gr[*ogr]->max[0], i);
                    bp->gr[*ogr]->max[1] = MAX (bp->gr[*ogr]->max[1], j);
                    bp->gr[*ogr]->max[2] = MAX (bp->gr[*ogr]->max[2], k);
                    bp->gr[*ogr]->min[0] = MIN (bp->gr[*ogr]->min[0], i);
                    bp->gr[*ogr]->min[1] = MIN (bp->gr[*ogr]->min[1], j);
                    bp->gr[*ogr]->min[2] = MIN (bp->gr[*ogr]->min[2], k);
                  }             /* end activate neigbour */
                }
              }                 /*end check neighbour loop */
            }
            /* end if frac >= SOLID */
            if (bp->ctrl->scheil == TRUE) {
              *sfsp = schiel (*c_temp_p);
              sch_sum += *sfsp;
              if (*sfsp >= 1.0) {
                sp->ncsolid++;
              }
            }                   /*end schiel fs calculation */
          }                     /*end phase diag not on */
        }                       /* end of GROWING section */

        if (*op > 1.0) {
          fprintf (stderr,
                   "WARNING:SB_CA_STEP: Something Weird is Happening!\n *op > 1.0\nsb %i, i %i, j %i, k %i, step %i, *op %.5g\n",
                   sbnum, i, j, k, bp->step, *op);
        }

        /* end of somethign wierd message */
        /* increment the pointers depending on   */
        /* whether they are padded or non-padded */
        /* arrays                                */
#ifdef OLD_TUNDER
        otup++;
#endif /*OLD_TUNDER */
        thrp++;                 /*nuc threshold */
        sfsp++;                 /*shciel fraction solid */
        np++;
        op++;
        bin_flag++;
        local_dfs_primary++;
        local_dfs_eutectic++;
        /* pop++; */
        /* eop++; */
        ngr++;
        index_ca++;
        solp++;                 /* gas concentration */
        ogr++;
        c_temp_p++;
        if (phase_diag_on || particle_on)
          cell_conc++;          /* alloy conc */
      }                         /*end of i loop -- x direction */
      sfsp += 2;
      op += 2;
      ogr += 2;
      bin_flag += 2;
    }                           /* end of j loop - y direction */
    sfsp += skip;
    op += skip;
    ogr += skip;
    bin_flag += skip;
  }                             /* end of k loop - z direction */

   /************************************************/
  /* end of horrible nested loops                 */
   /************************************************/

#ifdef PROBE_CELL
  {
    FILE *cell_file;
    char fname[64] = "";

    sprintf (fname, "cell%i.txt", PROBE_CELL);
    cell_file = fopen (fname, "a+");
    fprintf (cell_file, "%.5g,%.10g\n", bp->sim_time, sp->c_temp[PROBE_CELL]);
    fclose (cell_file);
  }
#endif /* PROBE_CELL */

  oldfs = sp->Tvals.fsavg;
  if (bp->ctrl->scheil == TRUE) {
    sp->Tvals.fsavg = sch_sum / bp->ncsb;
  } else {
    sp->Tvals.del_fs = sum_del_fs / (bp->ncsb - sp->nmould);
    sp->Tvals.fsavg += sp->Tvals.del_fs;
  }
  if (oldfs > sp->Tvals.fsavg) {
    static int fsmsg = 0;

    if (fsmsg < 5) {
      fprintf (stderr, "ERROR:sb_ca_step: fs decreasing!\n");
      fsmsg++;
    }
  }

  if (sp->ncsolid >= bp->ncsb) {
    fprintf (stderr, "SB_ca_step(): SB#%d completely solid.\n", sbnum);
    sp->done = TRUE;
  }
  return (errors);
}                               /* end of sb_ca_step */

/* subroutine to return rcs-id string */
char const *rcs_id_sb_ca_step_c ()
{
  static char const rcsid[] = "$Id: sb_ca_step.c 957 2006-11-09 20:23:51Z rcatwood $";

  return (rcsid);
}

/*
RCS Log:$Log$
RCS Log:Revision 11.3  2006/11/09 20:23:51  rcatwood
RCS Log:Tried to fix some things for doxygen
RCS Log:
RCS Log:Revision 11.2  2006/11/09 13:39:40  rcatwood
RCS Log:Merged the update for ca_procast version for procast 2006.0_beta
RCS Log:
RCS Log:Revision 11.1.12.1  2006/11/02 18:48:06  rcatwood
RCS Log:Fixed up old non-diffusion ca routine to work with coupled macro
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5.2.2  2006/01/18 16:10:27  rcatwood
RCS Log:Merged version that compiles
RCS Log:NEW_MOULD_SRC option removed (this is the correct version now)
RCS Log:
RCS Log:Revision 10.5.2.1  2006/01/11 12:12:19  rcatwood
RCS Log:Altered to use the new solute properties structure arrays
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.3.8.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.3.4.1  2004/03/09 11:30:53  rcatwood
RCS Log:Changed to read materials properties from separate files for alloy and gas
RCS Log:
RCS Log:Revision 9.3  2003/11/18 13:22:40  rcatwood
RCS Log:Added routines to find and store the interface cells between the casting and the mould.
RCS Log:Added the surface cell storage structure to the subblock.
RCS Log:Improved mould source and nucleation  routines to use the surface cell structure
RCS Log:
RCS Log:Revision 9.2  2003/09/16 11:59:15  rcatwood
RCS Log:Improved micro/macro interpolation
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:39  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.3.2.14  2003/08/14 14:18:04  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.3.2.13  2003/06/26 13:30:24  rcatwood
RCS Log:fixed not casting capture bug
RCS Log:
RCS Log:Revision 8.3.2.12  2003/05/06 15:48:08  rcatwood
RCS Log:Altered linear binary phase diagram usage to consistently use the values input from the control files rather than the header files.
RCS Log:
RCS Log:Revision 8.3.2.11  2003/05/02 13:14:37  maijer
RCS Log:Removed reference to T_EUT in sb_ca_step
RCS Log:
RCS Log:Revision 8.3.2.10  2003/04/08 11:44:35  rcatwood
RCS Log:after bug fix for directional gradient
RCS Log:
RCS Log:Revision 8.3.2.9  2003/03/14 14:45:35  rcatwood
RCS Log:Added grain extent tracking
RCS Log:Moved the grain update to a subroutine and call from sb_ca_step, capture_octahedron, and capture_octahedron_diffuse
RCS Log:
RCS Log:Revision 8.3.2.8  2003/02/27 23:04:39  rcatwood
RCS Log:Removed use of old temperature routines , all temperatures shoudl
RCS Log:be determined by checking the array c_temp in teh subblock, if the
RCS Log:subblock is open
RCS Log:
RCS Log:Revision 8.3.2.7  2003/02/26 18:44:49  rcatwood
RCS Log:Modified (non-decentered) so that the temperature uses an array
RCS Log:calcuated before each step. Exception: Procast mode still overrides
RCS Log:and uses Ali's routine.
RCS Log:
RCS Log:Revision 8.3.2.6  2003/02/05 13:15:21  rcatwood
RCS Log:Fixed some memory errors when list-file NOT used
RCS Log:Added ifdefs to get rid of sgi-specific image output routines
RCS Log:
RCS Log:Revision 8.3.2.5  2003/01/27 16:10:00  rcatwood
RCS Log:Version needing debugging of decentered square methond combined with fg input
RCS Log:
RCS Log:Revision 8.3.2.4  2003/01/22 16:53:46  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.3.2.3  2003/01/17 16:09:30  rcatwood
RCS Log:Before changing all CA_FLOAT to CA_CA_FLOAT
RCS Log:
RCS Log:Revision 8.3.2.2  2003/01/15 19:02:01  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.3  2003/01/14 16:22:26  rcatwood
RCS Log:Removed many lint warnings from sb_decentered_step
RCS Log:Added signal function to ca_procast
RCS Log:Removed some unused files
RCS Log:
RCS Log:Revision 8.1.6.2  2003/01/14 12:59:39  rcatwood
RCS Log:Merged procast version which compiles successfullly
RCS Log:Not checked for running yet
RCS Log:Including WEI decentered square but not temperature routine links
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.4.4.3  2002/11/04 11:22:06  rcatwood
RCS Log:Check-in after Ali has gone
RCS Log:
RCS Log:Revision 7.4.4.2  2002/08/27 15:01:20  chirazi
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 7.3  2001/06/27 11:42:13  rcatwood
RCS Log:added #IFDEF GLOBAL_UND_GRO or NUC cpp option.
RCS Log:
RCS Log:Revision 7.2  2001/04/03 11:57:34  rcatwood
RCS Log:included grad h in pore selection.
RCS Log:
RCS Log:Revision 7.1  2000/12/06 21:10:40  rcatwood
RCS Log:fixed up heatfolw, tctrace
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.2  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.1.1.1  2000/10/16 17:27:51  rcatwood
RCS Log:started making not-casting cells
RCS Log:
RCS Log:Revision 6.1  2000/10/16 10:37:22  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:35  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.7  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.6  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.6  2000/03/27 17:06:26  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.5  2000/03/27 11:21:22  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.5  2000/03/27 11:21:16  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.4  2000/03/24 19:33:30  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.4  2000/03/24 19:19:46  rcatwood
RCS Log:Particle dependant growth fixed.
RCS Log:
RCS Log:Revision 5.3  2000/03/23 19:00:16  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.3  2000/03/23 18:57:35  rcatwood
RCS Log:Particle dependant growth added.
RCS Log:
RCS Log:Revision 5.2  2000/03/15 16:25:32  rcatwood
RCS Log:backup checkin
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.3  2000/03/02 13:09:59  rcatwood
RCS Log:Fixed copy-mat bug.
RCS Log:
RCS Log:Revision 5.0.2.2  2000/03/01 16:23:02  rcatwood
RCS Log:not tested yet
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:09:31  rcatwood
RCS Log:Improve multiblock by adding open when activated routine.
RCS Log:
RCS Log:Revision 4.11.1.1  2000/02/17 13:32:57  rcatwood
RCS Log:version with growth bug -- growth in partly full cells allowed.
RCS Log:
RCS Log:Revision 4.11  2000/02/16 10:51:47  rcatwood
RCS Log:Fixed grain structure
RCS Log:
RCS Log:Revision 4.10  2000/02/14 19:34:29  rcatwood
RCS Log:Put back to c-only and removed pull and recrystallise options.
RCS Log:
RCS Log:Revision 4.7  2000/02/14 19:22:34  rcatwood
RCS Log:last w version, putting it back to c only
RCS Log:
RCS Log:Revision 4.5.1.2  2000/01/28 15:51:27  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 4.5.1.1  2000/01/27 11:37:57  rcatwood
RCS Log:Version with growth proceeding with a partly full cell.
RCS Log:
RCS Log:Revision 4.5  2000/01/05 12:47:29  rcatwood
RCS Log:Files as of New year 2000
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.2  1999/12/16 19:15:49  rcatwood
RCS Log:Alloy and gas diffusion working, pores working. 
RCS Log:Changed file name for conc. output A for alloy C for gas conc.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 13:01:54  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
