%%% Reaching towards a line target with multiplicate and additive noise %%%
% This code was used to produce Figure 9.6 A/B in
% /////////////////////////////////////////////////////////////////////////
%  "Human Robotics: Neuromechanics and Motor Control"
%   by Etienne Burdet, David Franklin W. and Theodore Milner E.
% /////////////////////////////////////////////////////////////////////////
% The simulation uses a modification of Todorov's LQG controller to
% simulate movements towards a line target. The manifestation of an
% uncontrolled manifold is seen along the line only with multiplicative
% noise, supporting the idea that our muscles are multiplicative by nature.
% --------
% Atsushi Takagi (09/09/13)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all;clc;close all;

% Time step (s)
dt = 0.01;
% Muscle time constant (s)
Tau = 0.3;
% Duration of movement (s)
N = 0.6/dt;
% Target distance (m)
T = sqrt(2)*0.25;
% Number of simulated movements
NoiseSim = 10000;

% System equations
A = [1  dt 0         0;
     0  1  dt        0;
     0  0  1-dt/Tau  dt/Tau;
     0  0  0         1-dt/Tau];

A = [A zeros(4,4);
     zeros(4,4) A];
A(9,9) = 1;

B = zeros(9,1);
B(4,1) = dt/Tau;
B(8,2) = dt/Tau;

% Control noise matrices
C = [0.3;0.3];
C0 = zeros(9,1);

% Observation matrix (position and velocity are observable)
H = zeros(4,9);
H(1:2,1:2) = eye(2);
H(3,5) = 1;
H(4,6) = 1;

% Position noise
PosNoise = 0.01;
% Velocity noise
VelNoise = 0.02;

% System noise matrix
D0 = diag(repmat([PosNoise,VelNoise],1,2));

% Control cost
R = 10^-9;
   
% State cost at endpoint
PosCost = 10;
VelCost = 0.1;
ForceCost = 0.05;
q = [PosCost zeros(1,3) PosCost zeros(1,3) -PosCost;
     0 VelCost 0 zeros(1,6);
     0 0 ForceCost zeros(1,6); zeros(1,9);
     PosCost zeros(1,3) PosCost zeros(1,3) -PosCost;
     zeros(1,5) VelCost zeros(1,3);
     zeros(1,6) ForceCost zeros(1,2); zeros(1,9);
     -PosCost zeros(1,3) -PosCost zeros(1,3) PosCost];

Q = zeros(9,9,N);
Q(:,:,N) = q;

X1 = zeros(9,1);
X1(9,1) = T;
S1 = zeros(9,9);

% Simulate systems
[xMultNoisy,UNoise] = ReachingLineLQG(A,B,C,C0,H,D0,0,Q,R,X1,S1,NoiseSim,0,0);

% Average multiplicative noise in movement
UMultAvg = mean2(abs(reshape(UNoise,1,2*NoiseSim*N)));

[xAddNoisy,UNoise] = ReachingLineLQG(A,B,C,C0,H,D0,0,Q,R,X1,S1,NoiseSim,1,UMultAvg);

% Rotate data axes
Theta = 45*pi/180;
Rotation = [cos(Theta) -sin(Theta); sin(Theta) cos(Theta)];

TempMult = Rotation*[reshape(squeeze(xMultNoisy(1,:,:)),1,N*NoiseSim);reshape(squeeze(xMultNoisy(5,:,:)),1,N*NoiseSim)];
xMultRotNoisy = reshape(TempMult(1,:),NoiseSim,N);
yMultRotNoisy = reshape(TempMult(2,:),NoiseSim,N);

TempAdd = Rotation*[reshape(squeeze(xAddNoisy(1,:,:)),1,N*NoiseSim);reshape(squeeze(xAddNoisy(5,:,:)),1,N*NoiseSim)];
xAddRotNoisy = reshape(TempAdd(1,:),NoiseSim,N);
yAddRotNoisy = reshape(TempAdd(2,:),NoiseSim,N);

% Fit a normal distribution to the endpoints of movement
[MuMult,SigmaMult] = normfit([xMultRotNoisy(:,end),yMultRotNoisy(:,end)]);
[MuAdd,SigmaAdd] = normfit([xAddRotNoisy(:,end),yAddRotNoisy(:,end)]);

% How many sigmas to draw ellipse
SigmaMultDraw = 2*SigmaMult;
SigmaAddDraw = 2*SigmaAdd;

% Standard deviation along x and y axes
STDMultX = std(squeeze(xMultRotNoisy),0,1);
STDMultY = std(squeeze(yMultRotNoisy),0,1);
STDAddX = std(squeeze(xAddRotNoisy),0,1);
STDAddY = std(squeeze(yAddRotNoisy),0,1);

% Plots
set(0,'DefaultFigureWindowStyle','docked','DefaultAxesFontsize',13);

% Number of trajectories to plot
ShowTrajectory = 15;
% Number of endpoints to plot
Endpoints = 100;
% Ellipse axes
xAxisMult = -SigmaMultDraw(1,1):0.001:SigmaMultDraw(1,1);
xAxisAdd = -SigmaAddDraw(1,1):0.001:SigmaAddDraw(1,1);

figure(1); set(gcf,'color','white');
subplot(1,2,1); hold on;
plot(xMultRotNoisy(1:ShowTrajectory,:)',yMultRotNoisy(1:ShowTrajectory,:)','k');
scatter(xMultRotNoisy(1:Endpoints,end),yMultRotNoisy(1:Endpoints,end),'.','markeredgecolor','k','sizedata',20);
plot([xAxisMult,-xAxisMult]+MuMult(1,1),[SigmaMultDraw(1,2)*real(sqrt(1-(xAxisMult/SigmaMultDraw(1,1)).^2))+MuMult(1,2),-SigmaMultDraw(1,2)*real(sqrt(1-(xAxisMult/SigmaMultDraw(1,1)).^2))+MuMult(1,2)],'k','linewidth',1.5);
axis equal;
xlabel('x [m]');
ylabel('y [m]');
title('Multiplicative noise');

subplot(1,2,2); hold on;
plot(xAddRotNoisy(1:ShowTrajectory,:)',yAddRotNoisy(1:ShowTrajectory,:)','k');
scatter(xAddRotNoisy(1:Endpoints,end),yAddRotNoisy(1:Endpoints,end),'.','markeredgecolor','k','sizedata',20);
plot([xAxisAdd,-xAxisAdd]+MuAdd(1,1),[SigmaAddDraw(1,2)*real(sqrt(1-(xAxisAdd/SigmaAddDraw(1,1)).^2))+MuAdd(1,2) -SigmaAddDraw(1,2)*real(sqrt(1-(xAxisAdd/SigmaAddDraw(1,1)).^2))+MuAdd(1,2)],'k','linewidth',1.5);
axis equal;
xlabel('x [m]');
ylabel('y [m]');
title('Additive noise');

figure(2); set(gcf,'color','white');
subplot(1,2,1); hold on;
% Time vector
Time = (0:N-1)*dt;

plot(Time,STDMultX,'b');
plot(Time,STDMultY,'r');
plot(Time,STDAddX,'--b');
plot(Time,STDAddY,'--r');
xlabel('Time [s]');
ylabel('Variability [m]');
xlim([0 N*dt]);

subplot(1,2,2); hold on;
plot(Time,STDMultX./STDMultY,'k');
plot(Time,STDAddX./STDAddY,'--k');
xlabel('Time [s]');
ylabel('Aspect ratio');
xlim([0 N*dt]);
