%%%%%%%%%%% Reaching with and without glasses %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was used to produce Figure 9.4 in
% /////////////////////////////////////////////////////////////////////////
% "Human Robotics: Neuromechanics and Motor Control"
% by Etienne Burdet, David Franklin W. and Theodore Milner E. 
% /////////////////////////////////////////////////////////////////////////
% The simulation uses an infinite horizon linear quadratic regulator as a
% controller for reaching movements in 1 dimension. The target jumps at an
% interval defined by the variable TargetMove.
%
% System equation:
% x_k+1 = A*x_k + B*u_k
% --------
% Atsushi Takagi (09/09/13)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all;clc;close all;

% Time step size (s)
dt = 0.001;
% Muscle time constant (s)
Tau = 0.4;

% State of original target
Target = [0.08;0;0;0];
% State of new target
NewTarget = [-0.08;0;0;0];
% Movement of target at different times (in seconds)
TargetMove = [0.05, 0.1, 0.15, 0.25, 0.35, 0.9]/dt;

% State vector
x = zeros(4,max(TargetMove),size(TargetMove,2));
for i=1:size(TargetMove,2)
    x(:,1,i) = [0 0 0 0];
end

% Cost of state
Q = zeros(4,4);
Q(1,1) = 50;
Q(2,2) = 0.5;

% Control cost
R(1,1) = 10^-10;

% Define system equations
A = [1  dt  0         0;
     0  1   dt        0;
     0  0   1-dt/Tau  dt/Tau;
     0  0   0         1-dt/Tau];

B = [0; 0; 0; dt/Tau];

% Solve algebraic Riccati equation (ARE) until convergence
P_store = zeros(4,4);
P = Q;
while norm(P-P_store)>0.001
    P_store = P;
    P = Q+A'*P*A-A'*P*B/(R+B'*P*B)*B'*P*A;
end
% Compute optimal control law
L = (R+B'*P*B)\B'*P*A;

% Model the system
for i=1:size(TargetMove,2)
    for k=1:max(TargetMove)-1
        if k>=TargetMove(1,i)
            Setpoint = NewTarget;
        else
            Setpoint = Target;
        end

        u = -L*(x(:,k,i)-Setpoint);
        x(:,k+1,i) = A*x(:,k,i)+B*u;
    end
end

% Plots
set(0,'DefaultFigureWindowStyle','docked','DefaultAxesFontsize',15);

figure(1); set(gcf,'color','white'); hold on;

plot((0:max(TargetMove)-1)*dt,squeeze(x(1,:,:)));
TargetMove = TargetMove(1:end-1);
h = line(repmat(TargetMove*dt,2,1),[-Target(1,ones(1,size(TargetMove,2)));Target(1,ones(1,size(TargetMove,2)))],'linewidth',2);

legend(strcat(num2str(TargetMove'*dt,'% 10.2f'),repmat({'s'},size(TargetMove,2),1)),'location','northeastoutside');
xlabel('Time (s)');
ylabel('Position (m)');
title('Target jumps at different times','fontsize',20);


