
/****************************************************************/
/*      Copyright (c) 1993 Peter D Lee                          */
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      All Rights Reserved                                     */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      continuing D Phil work from University of Oxford        */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @imperial.ac.uk for details   */
/****************************************************************/

/********************************************************************************/
/*  This version is distributed under a BSD style public license, as follows:   */
/*                                                                              */
/*  Copyright (c) 2007, Dept. of Materials, Imperial College London             */
/*  All rights reserved.                                                        */
/*  Redistribution and use in source and binary forms, with or without          */
/*  modification, are permitted provided that the following conditions          */
/*  are met:                                                                    */
/*                                                                              */
/*  * Redistributions of source code must retain the above copyright            */
/*  notice, this list of conditions and the following disclaimer.               */
/*                                                                              */
/*  * Redistributions in binary form must reproduce the above                   */
/*  copyright notice, this list of conditions and the following                 */
/*  disclaimer in the documentation and/or other materials provided             */
/*  with the distribution.                                                      */
/*                                                                              */
/*  * Neither the name of the Dept. of Materials, Imperial College London, nor  */
/*  the names of its contributors may be used to endorse or promote products    */
/*  derived from this software without specific prior written permission.       */
/*                                                                              */
/*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS         */
/*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT           */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR       */
/*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT        */
/*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,       */
/*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED    */
/*  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR      */
/*  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      */
/*  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING        */
/*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS          */
/*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                */
/********************************************************************************/
/*END of LICENSE NOTICE*/

#include <stdio.h>
#include <math.h>
#include "machine.h"

/* rand_step -- return a random number uniformly distributed between bottom and top */
CA_FLOAT rand_step (CA_FLOAT * params)
{
  CA_FLOAT bottom;
  CA_FLOAT width;

  /* make sure the parameters exist */
  if (params == NULL || (params + 1) == NULL) {
    fprintf (stderr, "ERROR: rand_step: You didn't give me enough information (NULL pointer)");
    fprintf (stderr, "ERROR: rand_step: You really need to debug this.");
    exit (1);
  }

  bottom = params[0];
  width = params[1];

  if ((bottom < 0) || (width <= 0)) {
    static int errmsg = 0;

    if (errmsg < MAX_ERR_MSG) {
      fprintf (stderr, "ERROR: rand_step: You can't ask me to do that! I quit! (bottom = %.5g width =%.5g) \n", bottom, width);
    }
#      ifdef ERROR_EXIT
    exit (1);
#      endif /*ERROR_EXIT */
  }
  return (bottom + drand48 () * width);
}                               /* end of rand_step */

/* rand_two_step  -- generate a random number distributed in two */
/* blocks, where the transition occurs at the point (middle) and changes */
/* by a factor (ratio) */
CA_FLOAT rand_two_step (CA_FLOAT * params)
{
  CA_FLOAT bottom, width1, width2, ratio;
  CA_FLOAT val, range;
  int i;

  /* make sure the parameters exist */
  for (i = 0; i < 4; i++) {
    if (params + i == NULL) {
      fprintf (stderr, "ERROR: rand_two_step: You didn't give me enough information (NULL pointer)");
      fprintf (stderr, "ERROR: rand_two_step: You really need to debug this.");
      exit (1);
    }
  }

  /* local variables */
  bottom = params[0];
  width1 = params[1];
  width2 = params[2];
  ratio = params[3];

  /* derived values */
  range = width1 + ratio * (width2);
  val = drand48 () * range;

  /* stretch the upper part by the ratio */
  if (val > width1) {
    val -= width1;
    val /= ratio;
    val += width1;
  }

  val += bottom;
  return (val);
}

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file */
char const *rcs_id_rand_step_c ()
{
  static char const rcsid[] = "$Id: rand_step.c 1339 2008-07-23 13:58:29Z  $";

  return (rcsid);
}

/* end of rcs_id_subroutine */
