/****************************************************************/
/*      Copyright (c) 1993 Peter D Lee Student Univ of Oxford   */
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      All Rights Reserved                                     */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      continuing D Phil work from University of Oxford        */
/*      This material contains CONFIDENTIAL INFORMATION that    */
/*      is the property of Imperial College. Any use,           */
/*      duplication or disclosure not specifically authorized   */
/*      by Imperial College is strictly prohibited.             */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/*      ca_wrapper.c:                                           */
/*  Main program to provide a wrapper for the portable 		*/
/*  bigblock/subblock subroutines to simulate grain		*/
/*  nucleation and growth, via the cellular automata method     */
/*  for eutectic equiaxed growth.                               */
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Wed Jul  1 18:38:31 bst 1998                 		*/
/****************************************************************/
/* 	MODIFIED by:						*/
/*  PDL: July 2, 1998						*/
/*  PDL: Aug 16, 1998						*/
/* See RCS log at bottom               */
/****************************************************************/
/*General:							*/
/* 1) 							*/
/****************************************************************/
/****** To Do List **********************************************/
/****************************************************************/

/*RCS Id:$Id: ca_wrapper.c 957 2006-11-09 20:23:51Z rcatwood $*/
/* include system headers */

/* if specified, include the headers for dbmalloc debugging */
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <sys/types.h>
#include <sys/times.h>
#include <setjmp.h>
#include <sys/signal.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>

/* header for enabling fpe traps in gnu glibc  library */
#ifdef GNU_TRAP_FPE
#  include <fenv.h>
#endif /* GNU_TRAP_FPE */

/* include header files requred by subroutines */
#include "machine.h"
#include "read_ctrl.h"          /* included for def. of READ_CTRL subroutine */
#include "blocks.h"             /* included for def. of INIT_BB, CALC_BB, FINISH_BB */
#include "handlers.h"

/*#define PRINT_RCS_ID
#define SAVE_RCS_ID*/
/*global for pressure - to be fixed*/
CA_FLOAT global_pressure;

extern int free_bb (BB_struct * bp);

extern void read_bin_blocks (BB_struct * bp, const char *fname);

/*function from rcs_id.c*/
extern void print_rcs_id (FILE * outfile);

/* functions used from ca_solid.c */
extern int ca_solid (int stat_flag, CA_FLOAT time, CA_FLOAT delt, Ctrl_str * cp, BB_struct * bp);

/* functions used from read_cp->c */
extern int read_ctrl (char *filename, Ctrl_str * cp);

jmp_buf env;                    /*jump environment buffer for signal handler */
int jflg = JFLG_END;            /* flag for behaviour of signal handler */
int the_signo;                  /* signal that was received */
int signal_change_freq = 0;     /* value to change output frequency */

/* empty stub for a procast function so that the non-procast versionwill compile */
void procast_sb_set_cells (BB_struct * bp, int sbnum)
{
  return;
}

void print_usage (char *prog_name)
{                               /* print the usage message on error */

  fprintf (stderr, "\n\n*************************************************\n");
  fprintf (stderr, "*   %s: Use the CA method to simulate grain\n", prog_name);
  fprintf (stderr, "*   nucleation and growth.\n");
  fprintf (stderr, "*   Optionally model the diffusion of gas and solute\n");
  fprintf (stderr, "*   and growth of gas porosity\n");
  fprintf (stderr, "*   possibly using the decentered square algorithm for\n");
  fprintf (stderr, "*   simulating preferred growth directions of grains\n");
  fprintf (stderr, "*   and maybe using three-component diffusion and partitioning.\n");
  fprintf (stderr, "*\n*   Usage: %s -c control_file\n", prog_name);
  fprintf (stderr, "*\tWhere the control_file, tells the\n");
  fprintf (stderr, "*\tprgramme which files to read information from.\n");
  fprintf (stderr, "*\tThe following command line options are allowed:\n");
  fprintf (stderr, "*\t-a\t\t-> test option reading and exit\n");
  fprintf (stderr, "*\t-c\t\t-> specify control file\n");
  fprintf (stderr, "*\t-r\t\t-> specify restart control file\n");
  fprintf (stderr, "*\t-f\t\t-> print compile-time options (cflags)\n");
  fprintf (stderr, "*\t-v\t\t-> print version information\n");
  fprintf (stderr, "*\t-h\t\t-> print this message\n");
  fprintf (stderr, "*\t\n");
  fprintf (stderr, "*************************************************\n\n");
}

/****************************************************************/
/* Beginning of the MAIN program!				*/
/* This is just a wrapper to simulate the grain subroutine	*/
/* being called from within CAP. It asks a few questions	*/
/* that that CAP would know, and then goes on from there.	*/
/* It gives you the option to enter all the thermophysical	*/
/* data that would be read from a file if using CAP.		*/
/* This file must be in the current directory called ca_mp.dat	*/
/****************************************************************/

/** \callgraph */
int main (int argc, char *argv[])
{

  /* declare counters and output variables for main */
  int i;
  int sigvec = 0;
  int stat_flag = -1;           /* stage in analysis: -1=before, 0=during, 1=final */

  /* variables for input */
  Ctrl_str *cp;
  int cflg, errflg;
  int finput = FALSE;           /* inquire control info from user as default */
  int restart_flag = FALSE;
  int errors = 0, tries = 0;
  char *ctrl_fname, *Cflags;

  int step, pr_step;
  CA_FLOAT ptime, del_ptime = 1.0;
  CA_FLOAT time = 0.0;
  CA_FLOAT delt = 1.0;
  char *time_string;            /* local date/time in string */
  FILE *rcs_id_file;
  static BB_struct bb;
  BB_struct *bp;
  sigset_t sigmask, newsig;

/* variables used for timing the program */
#ifdef CLOCK
  clock_t start_clock, end_clock;
  struct tms times_clock;
  double cpu_time_used, utime, stime;

  start_clock = clock ();
  {
#endif

/* variables used for debugging memory allocation */
#ifdef _DEBUG_MALLOC_INC
    unsigned long oldsize;
    unsigned long size1, size2;
    unsigned long hist0, hist1, hist2;

    oldsize = malloc_inuse (&(hist0));
#endif /*_DEBUG_MALLOC_INC*/

    bp = &bb;
    cp = (Ctrl_str *) calloc (1, sizeof (Ctrl_str));

#ifdef _DEBUG_MALLOC_INC
    malloc_inuse (&(cp->hist[0]));
#endif /*_DEBUG_MALLOC_INC*/

    /* allow external script to write compilation info into a string */
    /* this should be done in the Makefile */
    /* Stored in Cflags variable */

#include "getcflags.inc"

    fprintf (stdout, "\n*****\nCellular automata modelling of phase change phenomena.\n");
    fprintf (stdout, "Written by Peter D. Lee, Robert C. Atwood, Wei Wang, Ali Chirazi, \n");
    /*
       add your name here!
     */
    fprintf (stdout, "(c) Imperial College London, 1998--????\n\n\n");

#ifdef PRINT_RCS_ID
/*FIRST, print out all flags and rcs id strings*/
    fprintf (stderr, "%s\n", Cflags);
    print_rcs_id (stderr);
#endif /*PRINT_RCS_ID */

#define SAVE_RCS_ID
#ifdef SAVE_RCS_ID
/* and save to a file as well */
    rcs_id_file = fopen ("rcs_id.txt", "w");
    if (rcs_id_file == NULL) {
      fprintf (stderr, "ERROR: could not open file rcs_id.txt\n");
      exit (0);
    }
    fprintf (rcs_id_file, "%s\n", Cflags);
    print_rcs_id (rcs_id_file);
    fclose (rcs_id_file);
#endif /*SAVE_RCS_ID */

    /* Set up the signal handler to use SIGTERM */
    signal (SIGTERM, catchit);
    /* set up a signal handler to use SIGUSR1 (writeblock and exit) */
    signal (SIGUSR1, catchit);
    /* set up a signal handler to use SIGUSR2 (writeblock and continue) */
    signal (SIGUSR2, catchit);

    /* set up a signal handler to use SIGFPE (doesn't do anything, but you can set a trap in the debugger) */
    /* also requires linking with libfpe on the SGI , not sure how to do this on other systems (linux , cygwin) */
    /* using gcc >=2.2 : use fenv.h functions , feenableexcepts() (in handlers.c)  */

#ifdef GNU_TRAP_FPE
    enable_fpe_traps ();
    signal (SIGFPE, float_error);
#endif

/************************************************/
/* check the number parameters and print out    */
/* warning if incorrect                         */
/************************************************/
    errflg = 0;
    stat_flag = INIT_BB;
    /** \todo  waiting for someone to implement use of this feature, Header string -- general*/
    sprintf (bb.header, "\nThis space for rent :-)\n");
    sprintf (bb.tailer, "\nThis space unintentionally left blank 8-b\n");

    if (argc < 1)
      errflg++;
    #ifndef NOGETOPT
    while ((cflg = getopt (argc, argv, "avc:r:fh")) != -1) {
      switch (cflg) {
      case 'a':
        fprintf (stderr, "%s: sorry, flag 'a' is not defined\n", argv[0]);
        errflg++;
        break;
        /*get ctrl file name */
      case 'c':
        finput = TRUE;
        ctrl_fname = strdup (optarg);
        break;
        /* get restart file name */
      case 'r':
        finput = TRUE;
        ctrl_fname = strdup (optarg);
        stat_flag = RESTART_BB;
        restart_flag = TRUE;
        fprintf (stderr, "RESTART option selected\n");
        break;
        /* only print version of files and cflags */
      case 'v':
        print_rcs_id (stdout);
        /* only print compilation flags */
      case 'f':
        printf ("%s\n", Cflags);
        exit (0);
        break;

        /* print help message */
      case 'h':
      default:
        errflg++;
        break;
      }
    }
    #endif
    if (errflg) {
      print_usage (argv[0]);
      exit (0);
    }

/************************************************/
/* call the initialization subroutines          */
/************************************************/

    fprintf (stderr, "alloy exponent %.10g\n", ALLOY_EXPONENT);
    /* if no control input file was supplied,    */
    /* as the user if one should be used.        */
    if (!finput) {
      char command[MAX_STRING_LEN], answer[MAX_STRING_LEN];
      int echo = TRUE;
      int i, j;                 /* tmp counters */
      CA_FLOAT tmp;             /* tmp CA_FLOAT var. */

      fprintf (stdout, "No control file name was given.\n");
      fprintf (stdout, "Please enter the control filename:\n");
      fprintf (stdout, "or 'help' for help\n");
      fprintf (stdout, "or 'restart' for a restart\n");
      fgets (command, MAX_STRING_LEN, stdin);
      if (command[strlen (command) - 1] == '\n')
        command[strlen (command) - 1] = 0;
      if (strcasecmp (command, "restart") == 0) {
        stat_flag = RESTART_BB;
        restart_flag = TRUE;
        fprintf (stderr, "RESTART option selected\n");
        fprintf (stderr, "Now enter the restart control filename:\n");
        fgets (command, MAX_STRING_LEN, stdin);
        if (command[strlen (command) - 1] == '\n')
          command[strlen (command) - 1] = 0;
      }
      if (strcasecmp (command, "help") == 0) {
        print_usage (argv[0]);
        exit (0);
      }
      ctrl_fname = strdup (command);
    }

    fprintf (stderr, "input filename: %s.\n", ctrl_fname);
    cp->fn_ctrl = ctrl_fname;
    /************************************************/
    /* read in the control/restart file ************ */
    /************************************************/

    errors = read_ctrl (ctrl_fname, cp);
    if (errors >= 1) {
      fprintf (stdout, "ca_wrapper: exiting due to control file problems [%s]\n", ctrl_fname);
      exit (1);
    }
    /* copy compilation info into ctrl structure */
    cp->cflags = strdup (Cflags);

/************************************************/
/* loop through time calling the ca routine...  */
/*  (and later energy, diffusion, etc...)       */
/************************************************/
    step = 0;
    pr_step = 1;
    ptime = del_ptime;

    fprintf (stdout, "Calling ca_solid for initialisation.\n");

    /* call ca_solid -- initial or restart depends on the flag */
    ca_solid (stat_flag, time, delt, cp, bp);

    fprintf (stderr, "Finished ca_solid for initialisation.\n");
#ifdef _DEBUG_MALLOC_INC
    size1 = malloc_inuse (&hist1);
#endif       /*_DEBUG_MALLOC_INC*/

    /* set up the entry for signal handler */
    if (setjmp (env) != 0) {
      fprintf (stderr, "CA_WRAPPER: Handling the signal %i.  Finishing BB\n", the_signo);
    } else {                    /*signal not recieved! */

      /* call ca_solid for macro timestep */
      fprintf (stderr, "Calling ca_solid for one macro timestep.\n");
      stat_flag = CALC_BB;
      ca_solid (stat_flag, time, delt, cp, bp);
      fprintf (stderr, "Finished one macro-timestep in ca_solid.\n");

    }                           /*end of setjmp condition */
    /*program jumps to here if one of the  specified */
    /* signals is received */

    /* tell ca prgramme it has finished and to print out */
    stat_flag = FINISH_BB;
    cp->jflg = jflg;
    ca_solid (stat_flag, time, delt, cp, bp);
    fprintf (stderr, "Finished final call to ca_solid\n");

    fprintf (stderr, "\nfinished all calculations successfully.\n");
    fprintf (stderr, "be seeing you...\n");
   /************************************************/
    /* close all open input/output files and exit   */
   /************************************************/

    free (Cflags);
    free (ctrl_fname);
    if (restart_flag) {
      free (cp->fn_geo);
      free (cp->fn_mat);
      free (cp->fn_inp);
      free (cp->fn_base);
      free (cp->cflags);
      free (cp->rgbp);
      free (cp);
    }
#ifdef _DEBUG_MALLOC_INC
    size2 = malloc_inuse (&hist2);
    if (size2 != (oldsize)) {
      fprintf (stderr, "ERROR: ca_wrapper: dbMalloc test of size of allocated memory\n");
      fprintf (stderr, "\toldsize = %lu, size = %lu - should be %lu\n", oldsize, size2, oldsize);
      fprintf (stderr, "First list \n");
      malloc_list (2, hist0, hist1);
      fprintf (stderr, "Second list \n");
      malloc_list (2, hist0, hist2);
      fprintf (stderr, "Finshed dbMalloc size check \n");
    } else {
      fprintf (stderr, "OK\n");
    }
#endif /*_DEBUG_MALLOC_INC*/

#ifdef CLOCK
  }
  end_clock = clock ();
  times (&times_clock);
  cpu_time_used = ((double) (end_clock - start_clock)) / (CLOCKS_PER_SEC);
  utime = ((double) (times_clock.tms_utime)) / (THE_CLOCK);
  stime = ((double) (times_clock.tms_stime)) / (THE_CLOCK);
  fprintf (stderr, "CLOCK: cpu time used,utime,stime:, %.10g,%.10g,%.10g\n", cpu_time_used, utime, stime);
#endif

  return (0);
}                               /* end of main program, ca_wrapper */

/************************************************/
/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
/************************************************/
char const *rcs_id_ca_wrapper_c ()
{
  static char const rcsid[] = "$Id: ca_wrapper.c 957 2006-11-09 20:23:51Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_ca_wrapper_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.4  2006/11/09 20:23:51  rcatwood
RCS Log:Tried to fix some things for doxygen
RCS Log:
RCS Log:Revision 11.3  2006/11/09 13:39:40  rcatwood
RCS Log:Merged the update for ca_procast version for procast 2006.0_beta
RCS Log:
RCS Log:Revision 11.2.8.2  2006/11/06 20:18:17  rcatwood
RCS Log:Improvements for coupled version
RCS Log:Signal flag to reread output frequency from ca_newstep.in
RCS Log:
RCS Log:Revision 11.2.8.1  2006/11/02 18:48:06  rcatwood
RCS Log:Fixed up old non-diffusion ca routine to work with coupled macro
RCS Log:
RCS Log:Revision 11.2  2006/03/15 17:45:04  rcatwood
RCS Log:Added ifdefs to exclude add-in librarys if desired
RCS Log:Added more comments to procast interface routiens
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:39  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5.2.1  2006/02/22 15:56:48  rcatwood
RCS Log:Added pore output to ca_read, and extra info to print_usage
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:52  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:Revision 10.1  2005/11/03 11:56:46  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.4.8.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.6.4.5  2004/09/02 15:34:14  rcatwood
RCS Log:Prevented overflow in random seed value!
RCS Log:
RCS Log:Revision 9.6.4.4  2004/04/07 11:18:30  rcatwood
RCS Log:Fixed several division errors, added section to activate f.p.e. trapping
RCS Log:
RCS Log:Revision 9.6.4.3  2004/02/05 12:12:39  rcatwood
RCS Log:fix double-free after restart and signal USR1
RCS Log:
RCS Log:Revision 9.6.4.2  2004/01/30 19:13:53  rcatwood
RCS Log:output the phase diagram info in dump file
RCS Log:
RCS Log:Revision 9.6.4.1  2004/01/06 15:49:37  rcatwood
RCS Log:Fixed surface nucleation to work wtih Procast - moved transfer of NOT_CASTING information into open_sb
RCS Log:
RCS Log:Revision 9.6  2003/11/26 12:46:29  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.5  2003/11/05 17:00:39  rcatwood
RCS Log:Combined the ca_solid and ca_solid_procast into a single version (ca_solid.c)
RCS Log:this required some changes to wrapper and other files
RCS Log:
RCS Log:Revision 9.4  2003/10/16 13:09:39  rcatwood
RCS Log:Test rcs id file for writability to avoid seg fault
RCS Log:
RCS Log:Revision 9.3  2003/09/08 17:13:33  rcatwood
RCS Log:changed ca_feedback to use the average of the nearest cells for each node.
RCS Log:Fixed som warnings (implicit declarations)
RCS Log:
RCS Log:Revision 9.2  2003/09/02 14:01:01  rcatwood
RCS Log:Changed Makefiles so that dependencies (.h) are automatically found using GNU make
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:34  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.4.2.5  2003/03/21 18:16:41  rcatwood
RCS Log:added flat (xy-plane) mulitblock geometry
RCS Log:Tried to improve Makefile for ca_read
RCS Log:
RCS Log:Revision 8.4.2.4  2003/01/22 16:53:42  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.4.2.3  2003/01/16 18:55:31  rcatwood
RCS Log:Modified location of bb into ca_wrapper from ca_solid
RCS Log:Makefile modified to use subdirectories
RCS Log:
RCS Log:Revision 8.4.2.2  2003/01/15 19:02:00  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.3.2.1  2003/01/14 16:22:25  rcatwood
RCS Log:Removed many lint warnings from sb_decentered_step
RCS Log:Added signal function to ca_procast
RCS Log:Removed some unused files
RCS Log:Revision 8.4  2002/12/13 18:06:06  rcatwood
RCS Log:Started rewriting input section for VAR input
RCS Log:
RCS Log:Revision 8.3  2002/10/31 17:51:15  rcatwood
RCS Log:improved clock use for reporting calculation time
RCS Log:
RCS Log:Revision 8.2  2002/10/25 12:17:00  rcatwood
RCS Log:Merged cygwin portable version for Windows machines
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:01  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.8  2002/10/17 16:52:37  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.7.10.6  2002/10/17 16:03:44  rcatwood
RCS Log:Fixed a subtle problem concerning the amount of gas in the pores
RCS Log:This required adding a lot of debugging #ifdef CHECK_GAS lines
RCS Log:
RCS Log:Revision 7.7.10.5  2002/10/09 12:19:10  rcatwood
RCS Log:alpha merge of decentered and pore versions
RCS Log:with added write_(structue)_values routines and aux program files for autogenerating
RCS Log:these routines from the structure definitions.
RCS Log:
RCS Log:Revision 7.7.10.4  2002/09/27 14:38:45  rcatwood
RCS Log:Added write block structure values routines
RCS Log:Modified matprops to allow T and C for eutectic to b read in
RCS Log:Modified props to use these values
RCS Log:Modified Makefile to make the obj files in a subdirectory
RCS Log:
RCS Log:Revision 7.7.10.3  2002/09/19 18:34:57  rcatwood
RCS Log:First working restart version using bin-block file
RCS Log:
RCS Log:Revision 7.7.10.2  2002/09/17 20:51:07  rcatwood
RCS Log:Rearranged so that read/write leave the memory clean
RCS Log:Added signal 16 (SIGUSR1) catching for trapping after finishing the current timestep.
RCS Log:
RCS Log:Revision 7.7.10.1  2002/09/12 13:53:19  rcatwood
RCS Log:Added decentered mode arrays to freeblocks
RCS Log:Tested with efence and dbmalloc
RCS Log:TODO: fix curvature access bug in fs_change_diffuse (Wei)
RCS Log:
RCS Log:Revision 7.7  2002/05/23 17:28:01  rcatwood
RCS Log:Used dbmalloc to eliminate all memory leaks!
RCS Log:Started close-sb routine to improve multiblock
RCS Log:
RCS Log:Revision 7.6  2002/02/14 13:15:01  rcatwood
RCS Log:Added write_block option instead of cpp definintion.
RCS Log:
RCS Log:Revision 7.5  2001/06/05 13:39:23  rcatwood
RCS Log:Improve version , flag and help output in wrapper (needs proper Makefile)
RCS Log:
RCS Log:Revision 7.4.2.1  2001/06/05 10:11:14  rcatwood
RCS Log:Added phase diagram options
RCS Log:Improved version output for ca_wrapper
RCS Log:
RCS Log:Revision 7.4  2001/04/10 12:17:21  rcatwood
RCS Log: added listing of compilation flags to control header output (requres Makefile modification)
RCS Log:
RCS Log:Revision 7.3  2001/03/13 11:48:33  rcatwood
RCS Log:fixed some comments and added sb_line_int id subroutine
RCS Log:
RCS Log:Revision 7.2  2001/02/16 20:30:49  rcatwood
RCS Log:Added some commenting for DAS and Limrad
RCS Log:Removed some JUNK
RCS Log:
RCS Log:Revision 7.1  2001/02/15 15:15:51  rcatwood
RCS Log:Improved STAT_ONLY mode. Added pore extent output.
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:34  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.3  2000/06/29 14:31:02  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.2  2000/06/27 11:41:19  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.2  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.6  2000/01/27 12:18:48  rcatwood
RCS Log:Overgrowth addressed. Bindump of t, fs
RCS Log:
RCS Log:Revision 4.5  2000/01/20 17:41:17  rcatwood
RCS Log:Pore bug fix. Bindump of fs, ss, diff coeff
RCS Log:
RCS Log:Revision 4.4  2000/01/10 14:57:00  rcatwood
RCS Log:Quick SIGTERM handler in ca_wrapper
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.2  1999/12/21 10:34:56  rcatwood
RCS Log:included rcs_id output to file rcs_id.txt
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.4  1999/12/16 13:10:47  rcatwood
RCS Log:Still fixing rcs info.
RCS Log:
RCS Log:Revision 4.0.2.3  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
