/**\file calc_sb.c */
/****************************************************************/
/* Subroutine to perform one timestep on a subblock             */
/****************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>/* for memset prototype */

#include "machine.h"
#include "blocks.h"
#include "ca_matrix.h"
#include "writeblocks.h"
#include "read_sb.h"
#include "find_max.h"
#include "interp.h"
#include "ca_phy_calc.h"

#include "sb_head.h"
#include "sb_diffuse.h"
#include "sb_decentred_step.h"
#include "curvature.h"
#include "datatrans.h"

#ifdef CHECK_GAS
extern CA_FLOAT checkgas(BB_struct *bp,int callflag) ;
#endif


#ifdef CHIRAZI_MULTI
extern int      sb_diffuse_alloy_multi(BB_struct *bp, int sbnum);
#endif

extern int grow_octahedron_poly (BB_struct * bp, int sbnum);

/** 
* Function calc_sb: Perform all necessary calculation for one microstep on 
*   one subblock.
*
* @callgraph
* @callergraph
*/
int calc_sb (BB_struct * bp, int sbnum)
{
   int      errors = 0;
   int      i;    /* tmp counters */
  CA_FLOAT *ofs, *nfs, *ofs_pad;
   int      *ogr, *ngr;
   CA_FLOAT coor[3];   /* tmp CA_FLOAT var.*/
   int      **gv_p; /* grain block value array pointer pointer :-/ */
   SB_struct * sp;
  int iphs, iphs_tot;
  CA_FLOAT *ofs_poly[NPHAMAX],*nfs_poly[NPHAMAX];

   #ifdef CHECK_GAS
      static CA_FLOAT diffgas0=0,diffgas1=0,change=0;

      diffgas0 = checkgas(bp,5);
      change = diffgas0 - diffgas1;
      if (ABS(change) > 1e-5){
         fprintf(stderr,"WARNING: sb_pore: gas change limit exceeded! %.5g\n",change);
      }
   #endif

  iphs_tot = bp->ctrl->NUM_PHS;

   sp = bp->sb[sbnum];
   gv_p = bp->gr_array;
   ofs = bp->ftmp_one;
   nfs = sp->c_fs;
   ogr = bp->itmp_one;
   ngr = sp->gr;

  if (bp->ctrl->diffuse_alloy_poly == TRUE){
    for (iphs = 0; iphs < iphs_tot; iphs++) {
      ofs_poly[iphs]=bp->ftmp_one_poly[iphs];
      nfs_poly[iphs]=sp->c_fs_poly[iphs];
    }
  }


   coor[0] = coor[1] = coor[2] = 1.0;

   /************************************************/
   /* Calculate the new temperature of the sb...   */
   /************************************************/
   sb_temp_calc(bp, sbnum);
  
#ifdef TEST_PHY
   fprintf(stderr, "In cala_sb: cell temperature compare: %f  %f \n", bp->ca_cell_Tar[0],bp->sb[sbnum]->c_temp[0]);
#endif
   /************************************************/
   /* Copy the f_s and gr  arrays...           */
   /************************************************/
   if (bp->ctrl->scheil == TRUE) {
      /* (flag, to, from, bp, block_values,sbnum) */
      fcopy_matrix(PAD, bp->ftmp_three, sp->sch_fs, bp, bp->sch_fs_values->block_array, sbnum);
  }

  /*end of schiel test */
   /* The fraction solid is copied into a padded buffer array */
   /* (flag, to, from, bp) */
   fcopy_matrix(PAD, ofs, nfs, bp, bp->c_fs_values->block_array, sbnum);

  if (bp->ctrl->diffuse_alloy_poly == TRUE){
    for (iphs = 0; iphs < iphs_tot; iphs++) {  
      fcopy_matrix (PAD,ofs_poly[iphs],nfs_poly[iphs], bp, bp->c_fs_values[iphs].block_array, sbnum);
    }
   }
   
   /************************************************/
   /* If still liquid call nucleation and growth   */
   /************************************************/
   if (!bp->sb[sbnum]->done) {

      /*******************************************/
      /* Nucleate ...                            */
      /* This nucleation routine gets called     */
      /* for those modes where there is not      */
      /* cell-to-cell variation.                 */
      /*                                         */
      /*******************************************/

      /* this is set in combine_ctrl routine */
      if (!(bp->ctrl->use_cell_nuc)){
       sb_nuc(bp, sbnum);
      }
       /************************************************/
       /* Grow...                                      */
       /* Call the CA subroutine for one step...       */
       /*                                              */
       /************************************************/

      /* Wei's decentred square / octahedron method */
      if (bp->ctrl->decentred_octahedron == TRUE) {  /* by Wei WANG 15-07-02*/

      /* this is set in combine_ctrl routine */
      if ((bp->ctrl->use_cell_nuc) && (bp->ctrl->diffuse_alloy_poly == FALSE)) {
          cell_nucleation (bp, sbnum);                         /*cell nucleation*/
      }

#ifdef TEST_PHY
     fprintf(stderr, "After cell_nucleation: cell temperature compare: %f  %f \n", bp->ca_cell_Tar[0],bp->sb[sbnum]->c_temp[0]);
#endif

        
      if ((bp->ctrl->use_cell_nuc) && (bp->ctrl->diffuse_alloy_poly == TRUE)) {
        cell_nucleation_poly (bp, sbnum);       /*cell nucleation */
	}
 
      if ((bp->ctrl->diffuse_alloy == TRUE) && (bp->ctrl->diffuse_alloy_poly == FALSE)) {
         icopy_matrix(PAD, ogr, ngr, bp, gv_p, sbnum);
         /* need to use old GR stored in temporary buffer */
         cell_index(bp, sbnum);                                 /* cell index */
 
	cell_temp(bp, sbnum);                                /* cell temperature */

         /* need to use old FS stored in temporary buffer 
            new Curvature changed in SB */
         
	 if (bp->ctrl->curvature_3D == 1){
	    interface_normal_3D(bp,sbnum);
	    interface_curvature_3D(bp, sbnum);
	 } else if (bp->ctrl->curvature_2D == 1){
	    interface_normal(bp, sbnum); 
       interface_curvature(bp, sbnum);
	 } else if (bp->ctrl->curvature_2D == 2){
       surface_curvature(bp, sbnum); /* Nastac interface curvature */
         } else if (bp->ctrl->curvature_2D == 3){
           curvature_2d(bp, sbnum);  /*Voller's method   by lyuan  06/2008*/
	 }

         /* alloy diffusion */
         /* need to use old FS stored in temporary buffer
            old C_L is copied to temporary buffer
            old C_E is copied to temporary buffer, new C_E is stored in SB */ 
         sb_diffuse_alloy_decentred(bp, sbnum);                           /* diffuse alloy */

         /* need to use old C_E stored in temporary buffer and new C_E in SB
            need to use old Curvature in SB
            new FS is changed in ftmp_one
            new C_L is changed in SB
            new GR assoicated with melting back is changed in SB */
         fs_change_diffuse(bp, sbnum);                          /* fs change with diffusion */
 
         /* need to use new FS in ftmp_one
            new half diagonal of octahedron is stored in SB */
         grow_octahedron (bp, sbnum);                    /* grow octahedron */

         /* capture new cells */
         /* need to use GR, FS stoed in temoprary buffer
            new GR associated with captures are stored in SB
            new FS associated with captures are stored in ftmp_one
            old informations of octahedron are copied to temporary buffers, new ones are stored in SB */
         capture_octahedron_diffuse(bp, sbnum);                 /* octahedron capture cell with diffusion*/
      }

      else if (bp->ctrl->diffuse_alloy_poly == TRUE) {  /* wei wang's algorithm extension to multicomponent systems by thuinet */

        /* need to use old GR stored in temporary buffer */
        cell_index (bp, sbnum); /* cell index */

        cell_temp (bp, sbnum);  /* cell temperature */

        /* need to use old FS stored in temporary buffer
           new Curvature changed in SB */
        /* cannot use existing routine with polycomponent since the fraction-solid does not have the correct meaning */
        /**todo: modify curvature to use a meaningful value in the polycomponent case */
        /*surface_curvature (bp, sbnum);*/  /* interface curvature */
        if ((bp->ctrl->curvature_3D !=0) || (bp->ctrl->curvature_2D != 0 )){
            fprintf(stderr,"ERROR:%s: Curvature not yet implemented for poly-component option! \n",__func__);
            exit(0);
        }

        /* alloy diffusion */
        /* need to use old FS stored in temporary buffer
           old C_L is copied to temporary buffer
           old C_E is copied to temporary buffer, new C_E is stored in SB */
        sb_diffuse_alloy_decentred_poly (bp, sbnum);    /*diffuse alloy in multicomponent systems */

        /* need to use old C_E stored in temporary buffer and new C_E in SB
           need to use old Curvature in SB
           new FS is changed in ftmp_one
           new C_L is changed in SB
           new GR assoicated with melting back is changed in SB */
        fs_change_diffuse_poly (bp, sbnum);     /* fs change with diffusion in multicomponents systems */

        /* need to use new FS in ftmp_one
           new half diagonal of octahedron is stored in SB */
        grow_octahedron_poly (bp, sbnum);       /* grow octahedron */

        /* capture new cells */
        /* need to use GR, FS stoed in temoprary buffer
           new GR associated with captures are stored in SB
           new FS associated with captures are stored in ftmp_one
           old informations of octahedron are copied to temporary buffers, new ones are stored in SB */
        capture_octahedron_diffuse_poly (bp, sbnum);    /* octahedron capture cell with diffusion */
	} else { /*No diffusion */
         fs_change_nodiffuse (bp, sbnum);
         grow_octahedron (bp, sbnum);
         capture_octahedron (bp, sbnum);
	}

         /*******************************************/
         /* Diffuse gas                             */
         /*******************************************/
      if (bp->ctrl->diffuse == TRUE){
        if (bp->ctrl->diffuse_alloy_poly == FALSE) 
          errors += sb_diffuse_gas(bp, sbnum);
        if (bp->ctrl->diffuse_alloy_poly == TRUE)
          errors += sb_diffuse_gas_poly (bp, sbnum);
      }

      /******************************************/
      /* Call the pore calculation if needed    */
      /******************************************/

      if (bp->ctrl->pore == TRUE)
        errors += sb_pore (bp, sbnum);

      fcopy_mat_back (PAD, nfs, ofs, bp, bp->c_fs_values->block_array, sbnum);

      if (bp->ctrl->diffuse_alloy_poly == TRUE){
        for (iphs = 0; iphs < iphs_tot; iphs++) {
          fcopy_mat_back(PAD,nfs_poly[iphs],ofs_poly[iphs], bp, bp->c_fs_values[iphs].block_array, sbnum);
	       }
	  }
	  


      } else {  /* Robert's algorithm  -- 1-neighbour capture*/
       /*ensure new grains are included*/
       icopy_matrix(PAD, ogr, ngr, bp, gv_p, sbnum);

       /**********************************/
       /*   call the ca step algorithm   */
       /**********************************/

       /* fraction solid is updated in the buffer ftmp_one */

       /****************************************************/
       /*                                                  */
       /*             DO THE CA STEP                       */
       /*                                                  */
       /****************************************************/

       sb_ca_step(bp, sbnum);

       /* after sb_ca_step, the NEW fraction solid is in the subblock fs array */
       /* and the OLD fraction solid is still in the bp->ftmp_one array. This is */
       /* needed in order for the diffusion (non-decentered method and gas diffusion)*/
       /* to calculate the partition of the dissolved species */

       /* this is a potential problem for the multi block case -- where does the old frction */
       /* solid for another block go to ? possibly the whole old big block needs to be buffered */

         /*******************************************/
         /* Diffuse alloy                           */
         /*******************************************/
      if ((bp->ctrl->diffuse_alloy == TRUE)&&(bp->ctrl->diffuse_alloy_multi==FALSE)){
         errors += sb_diffuse_alloy(bp, sbnum);
      }

         #ifdef CHIRAZI_MULTI
         /*******************************************/
      /* Multi Diffuse alloy                           */
         /*******************************************/
      if (bp->ctrl->diffuse_alloy_multi == TRUE){
          errors += sb_diffuse_alloy_multi(bp, sbnum);
      }
      #endif

      /* needs to have OLD fs in subblock array, NEW fs in buffer! */
      if (bp->ctrl->diffuse == TRUE)
         errors += sb_diffuse_gas(bp, sbnum);

          /******************************************/
          /* Call the pore calculation if needed    */
          /******************************************/
      if (bp->ctrl->pore == TRUE)
        errors += sb_pore (bp, sbnum);

         /*******************************************/
         /* update scheil concentration             */
         /*******************************************/
      if (bp->ctrl->scheil == TRUE) {
        fcopy_mat_back (PAD, sp->sch_fs, bp->ftmp_three, bp, bp->sch_fs_values->block_array, sbnum);
        /* (flag, to, from, bp) */
      }

      /*end of schiel test */
      /*************************************/
      /* at this point the old f_s is lost */
      /*************************************/
      fcopy_mat_back(PAD, nfs, ofs, bp, bp->c_fs_values->block_array, sbnum);
      icopy_mat_back(PAD, ngr, ogr, bp, bp->gr_array, sbnum);

      i = (bp->nc[0] + 2) *  (bp->nc[1] + 2) *  (bp->nc[2] + 2);
      if (memset(bp->itmp_one,0,i*sizeof(int)) == NULL){
         fprintf(stderr,"ERROR: calc_sb: memset failed! Debug me! \n");
         exit(25);
      }
      
   } /* end Robert's algorithm*/

  }

  /*end of test for "done" condition */
   /* window moving by Wei Wang on 04-11-02 added starting time.. LY*/
   if (bp->ctrl->window_moving == TRUE && bp->ctrl->wmstarttime <= bp->sim_time) {
    bp->window_disp += bp->window_velo * bp->delt;
     if (bp->window_disp >= bp->size_c[0]) {
       if(bp->nc[2] <= 1) window_move (bp,sbnum);
       else  window_move_3d (bp,sbnum);  
       bp->window_disp -= bp->size_c[0];
       fprintf(stderr,"Window disp: %f\n", bp->window_disp);
       bp->window_steps += 1;
       fprintf(stderr,"Window step: %i\n", bp->window_steps);
     }
   }
   return (errors);
} /* end of calc_sb subroutine */

char const *rcs_id_calc_sb_c()
{
  static char const rcsid[] = "$Id: calc_sb.c 1489 2010-07-23 18:02:30Z  $";

   return(rcsid);
}

/*
RCS Log:$Log$
RCS Log:Revision 11.3  2006/11/09 20:38:29  rcatwood
RCS Log:Added caller graphs to doxygen
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:39  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3.2.8  2006/02/20 14:51:22  rcatwood
RCS Log:Correced evil log message
RCS Log:
RCS Log:Revision 10.3.2.7  2006/02/20 12:00:13  lthuinet
RCS Log: A lot of modifications : add an option to treat correctly diffusion in the case the secondary eutectic phase is stoechiometric AND add a pointer to distinguish the nature of the cell and the nature of the grain
RCS Log:
RCS Log:Revision 10.3.2.6  2006/02/01 14:36:48  rcatwood
RCS Log:solved all implicit function declarations
RCS Log:
RCS Log:Revision 10.3.2.5  2006/02/01 14:16:25  lthuinet
RCS Log:#update change of the calculation of supersaturation for gas in multicomponent
RCS Log:
RCS Log:Revision 10.3.2.4  2006/01/20 18:24:24  rcatwood
RCS Log:Merged polycomponent with remelt/curvature version
RCS Log:
RCS Log:Revision 10.3.2.3  2006/01/18 18:35:10  rcatwood
RCS Log:started makeing example input files for the new version
RCS Log:
RCS Log:Revision 10.3.2.2  2006/01/11 16:31:31  rcatwood
RCS Log:Altered the use of solute information to conform with the new solute information structure table
RCS Log:
RCS Log:Revision 10.3.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 9.4  2004/07/05 10:50:19  rcatwood
RCS Log:Improved some prototypeing
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.2  2005/11/24 13:10:54  rcatwood
RCS Log:Ran INDENT on all the files!
RCS Log:Revision 10.1.2.3  2005/12/01 13:09:37  rcatwood
RCS Log:Fixed some implicit function declarations
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:52  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:Revision 10.1  2005/11/03 11:56:46  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 9.3.10.1  2004/09/09 17:49:42  xly
RCS Log:add curvature undercooling by estimating interface normal of growing cells.
RCS Log:Revision 8.3.8.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.3  2003/10/16 11:29:24  rcatwood
RCS Log:Changed incorrect use of con_cast to use_cell_temp
RCS Log:Added icc support in Makefile
RCS Log:
RCS Log:Revision 9.2  2003/09/16 11:59:14  rcatwood
RCS Log:Improved micro/macro interpolation
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:35  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.3.2.6  2003/02/26 18:44:48  rcatwood
RCS Log:Modified (non-decentered) so that the temperature uses an array
RCS Log:calcuated before each step. Exception: Procast mode still overrides
RCS Log:and uses Ali's routine.
RCS Log:
RCS Log:Revision 8.3.2.5  2003/01/23 17:47:27  rcatwood
RCS Log:finite grid applied to decentered square,
RCS Log:works, but not checked for correct results.
RCS Log:
RCS Log:Revision 8.3.2.4  2003/01/22 16:53:43  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.3.2.3  2003/01/17 16:09:29  rcatwood
RCS Log:Before changing all CA_FLOAT to CA_CA_FLOAT
RCS Log:
RCS Log:Revision 8.3.2.2  2003/01/15 19:02:00  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:45  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:01  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.2  2002/10/17 16:52:37  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.1.12.11  2002/10/17 16:03:44  rcatwood
RCS Log:Fixed a subtle problem concerning the amount of gas in the pores
RCS Log:This required adding a lot of debugging #ifdef CHECK_GAS lines
RCS Log:
RCS Log:Revision 7.1.12.10  2002/10/07 13:24:28  ww1
RCS Log:fixed total fraction solid at each time step and remove curv from ctrl
RCS Log:
RCS Log:Revision 7.1.12.9  2002/10/07 10:37:31  ww1
RCS Log:fix the fraction solid buffer to be compatible between Robert and Wei versions
RCS Log:
RCS Log:Revision 7.1.12.8  2002/10/04 16:21:27  rcatwood
RCS Log:adjusted the reporting of fraction solid, to allow for multi block
RCS Log:
RCS Log:Revision 7.1.12.7  2002/10/04 15:00:22  rcatwood
RCS Log:improved commenting to document buffer array usage
RCS Log:
RCS Log:Revision 7.1.12.6  2002/10/04 14:38:24  ww1
RCS Log:Cell Nucleation of Decentred square method
RCS Log:
RCS Log:Revision 7.1.12.5  2002/09/27 13:45:14  ww1
RCS Log:add random nucleation to decentered square
RCS Log:fix nucleation in solid problem
RCS Log:
RCS Log:Revision 7.1.12.4  2002/09/17 16:42:10  ww1
RCS Log:add decenetered square for no diffusion option.
RCS Log:
RCS Log:Revision 7.1.12.3  2002/08/27 12:23:19  rcatwood
RCS Log:Improved comments and organized the header files
RCS Log:deleted obsolete debug ifdef's
RCS Log:
RCS Log:Revision 7.1.12.2  2002/08/23 18:08:49  ww1
RCS Log:a few change made to calc_sb by Wei WANG
RCS Log:C
RCS Log:VS: Modified Files:
RCS Log:
RCS Log:Revision 7.1.12.1  2002/08/22 15:13:43  ww1
RCS Log:Merge of Wei Wang's decentered octahedron/square method
RCS Log:
RCS Log:Revision 7.1  2001/07/06 15:18:04  rcatwood
RCS Log:fixed another bug which caused seg fault when no pore mode selected
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.1  2000/10/16 10:37:22  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.2  2000/07/11 16:33:44  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.4  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.4  2000/03/27 17:06:26  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.3  2000/03/27 11:21:22  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.3  2000/03/27 11:21:17  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.2  2000/03/24 19:33:30  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.2  2000/03/24 19:19:47  rcatwood
RCS Log:Particle dependant growth fixed.
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:10:16  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.4  2000/03/02 13:09:59  rcatwood
RCS Log:Fixed copy-mat bug.
RCS Log:
RCS Log:Revision 5.0.2.3  2000/03/01 16:48:01  rcatwood
RCS Log:fixed rcs_id sobroutine
RCS Log:
RCS Log:Revision 5.0.2.2  2000/03/01 16:23:25  rcatwood
RCS Log:added rcs stuff
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 16:10:03  rcatwood
RCS Log:split calc_sb out of subblock.c
RCS Log:
*/
