/* #define DEBUG_FIDAP */
/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*   email p.d.lee or r.atwood or x.xu @ic.ac.uk for details    */
/****************************************************************/

/****************************************************************/
/* fidap_interp_calc.c:                                         */
/* Subroutine to interpolate the temperature at a point         */
/* in the CA code using values calculated in FIDAP in a         */
/* quasi-steady state solution of the heat, mass and            */
/* momentum transfer in VAR ingots.                             */
/****************************************************************/
/****************************************************************/
/* Written by X. Xu, P.D. Lee & R.C. Atwood, Imperial College   */
/* Nov 10, 1998                                                 */
/****************************************************************/
/*      MODIFIED by:                                            */
/*  PDL: Nov 10, 1998                                           */
/****************************************************************/
/****** To Do List **********************************************/
/*General:                                                      */
/* 1)                                                           */
/****************************************************************/
/*RCS Id:$Id: fidap_interp_calc.c 887 2006-03-01 18:21:01Z rcatwood $*/
#include <stdio.h>
#include <math.h>

/* include header files requred by subroutines */
#include "machine.h"
#include "fidap.h"              /* included for def. of FGrid_struct */

/****************************************************************/
/****************************************************************/
/* fidap_interp_calc:                                           */
/* Subroutine to interpolate temp. of the cell                  */
/****************************************************************/
/* Input Variables:                                             */
/*      		*Note: currently FIDAP geom. is trans.  */
/*      		to [mm] from [m] since [mm] used in CA  */
/*   *fg:		ptr to the FGrid_str structure that     */
/*       		holds all FIDAP geometry and T's data.  */
/*   xi:		x location in the CA model (equiv. to   */
/*       		radial dist. in FIDAP.           [mm]   */
/*   yi:		y location in the CA model (equiv. to   */
/*       		height or z in FIDAP.            [mm]   */
/*   time:		time in the CA model             [s]    */
/*      		                                        */
/* Output Variables:    NONE                                    */
/*      		                                        */
/* Returned Value:                                              */
/*   T: 		Temperature for given x, y, time in     */
/*       		CA model space interpolated from FIDAP  */
/*       		results.                                */
/****************************************************************/
/****************************************************************/
CA_FLOAT fidap_interp_calc (FGrid_str * fg, CA_FLOAT xi, CA_FLOAT yi, CA_FLOAT time)
{
  CA_FLOAT r, z, h, rmax, T;
  CA_FLOAT tp1, tp2, tp3, tp4, tmpx1, tmpx2;
  int i, ib, ie, jb, je;
  CA_FLOAT mag, t_s[5], dt[5], dtstep;
  int j, nr1, nr2;

  h = fg->z[fg->nz - 1];
  rmax = fg->r[fg->nr - 1];
  /* fprintf(stderr," ***** %f %f ***** \n",h,rmax); */

/************************************************/
/* Check that the x value is in range           */
/************************************************/
  if ((xi > fg->r[fg->nr - 1]) || (xi < fg->r[0])) {
    return (fg->Tmin);
  }

  r = xi;

/* check radius range */
  for (i = 0; i < (fg->nr - 1); i++) {
    if (r > fg->r[i] && r <= fg->r[i + 1]) {
      jb = i;
      je = i + 1;
    }
    if (r == fg->r[i]) {
      jb = i;
      je = i;
    }
  }
  /* changed on nov.22,99 to see the growth direction with T */
  /* z = -yi + fg->v*time*(1+(fg->r[fg->nr-1]- xi)/fg->r[fg->nr-1])/10000.0; */
  z = -yi + fg->v * time;       /* PREVIOUS ONE */
/* first give a general situation, when treeing_on, change it */

/* Adjust z if Treerings on */
  if (fg->tring_on) {
    /*     if (!(tsta = (float *) calloc(fg->ntring, sizeof(float)))) {
       fprintf(stderr, "Error: t_s allay malloc failed \n"); }
       if (!(dtpe = (float *) calloc(fg->ntring, sizeof(float)))) {
       fprintf(stderr, "Error: dt_p allay malloc failed \n");}  */
    dtstep = time - fg->oldtime;
    /* fprintf(stderr," ** %d ***\n ",fg->ntring);  */
/************************************************/
/* Check that the x value is in range           */
/************************************************/
    for (i = 0; i < fg->ntring; i++) {
      t_s[i] = fg->tring[i].v_f[1];
      dt[i] = fg->tring[i].v_f[2];
    }                           /* end of read into time and duration for a pertubation */

#ifdef PDL_TREE

    if ((time >= t_s) && (time <= (dt + t_s))) {
      mag = fg->tring[i].v_f[0];
      nr1 = fg->tring[i].v_int[0];
      nr2 = fg->tring[i].v_int[1];
      /*  fprintf(stderr," ** %d %d ***\n ",fg->tring[i].v_int[0],fg->tring[i].v_int[1]); */

      for (j = nr1; j < nr2; j++) {
        /*            fprintf(stderr," **** %d %d ***\n ",nr1, nr2);  */
        fg->zoffset[j] += fg->v * mag * dtstep;
        /*      fprintf(stderr," **** zoff in each step %f ***\n ",fg->zoffset[j]); */
      }
    }                           /* end of pertubation time */
#endif /* end of def PDL */

    for (i = 0; i < fg->ntring; i++) {
      if (time >= t_s[i] && time < t_s[i] + dt[i]) {
        mag = fg->tring[i].v_f[0];
        z = -yi + fg->v * time * (1.0 + mag);
      }
    }                           /* end of ntring */
    fg->oldtime = time;
  }

  /* end of tring_on */
  /* If the z value is above the pool, set = Tmax */
  if (z > h) {
    T = fg->Tmin;
    return (T);

/* If the z value is below the pool, set = Tmax */
  } else if (z < 0.0) {
    T = fg->Tmax;
    return (T);

/* z in range of FIDAP calculation, so interpolate! */
  } else {

    /* check z range */
    for (i = 0; i < (fg->nz - 1); i++) {
      if (z > fg->z[i] && z <= fg->z[i + 1]) {
        ib = i;
        ie = i + 1;
      }
      if (z == fg->z[i]) {
        ib = i;
        ie = i;
      }
    }

    /* CA location identical to node location, return value at node */
    if (ib == ie && jb == je) {
      T = fg->Fidap_T[fg->nr * ib + jb];

      /* CA location identical to node location in i dir, interp j */
    } else if (ib == ie && jb != je) {
      tp1 = fg->Fidap_T[fg->nr * ib + jb];
      tp2 = fg->Fidap_T[fg->nr * ib + je];
      T = tp1 * (fg->r[je] - r) / (fg->r[je] - fg->r[jb])
        + tp2 * (r - fg->r[jb]) / (fg->r[je] - fg->r[jb]);

      /* CA location identical to node location in j dir, interp i */
    } else if (ib != ie && jb == je) {
      tp1 = fg->Fidap_T[fg->nr * ib + jb];
      tp2 = fg->Fidap_T[fg->nr * ie + jb];
      T = tp1 * (fg->z[ie] - z) / (fg->z[ie] - fg->z[ib])
        + tp2 * (z - fg->z[ib]) / (fg->z[ie] - fg->z[ib]);

      /* interp in both directions */
    } else {                    /*  if(ib!=ie && jb!=je) */
      tp1 = fg->Fidap_T[fg->nr * ib + jb];
      tp2 = fg->Fidap_T[fg->nr * ib + je];
      tp3 = fg->Fidap_T[fg->nr * ie + jb];
      tp4 = fg->Fidap_T[fg->nr * ie + je];
      tmpx1 = tp1 * (fg->r[je] - r) / (fg->r[je] - fg->r[jb])
        + tp2 * (r - fg->r[jb]) / (fg->r[je] - fg->r[jb]);
      tmpx2 = tp3 * (fg->r[je] - r) / (fg->r[je] - fg->r[jb])
        + tp4 * (r - fg->r[jb]) / (fg->r[je] - fg->r[jb]);
      T = tmpx1 * (fg->z[ie] - z) / (fg->z[ie] - fg->z[ib])
        + tmpx2 * (z - fg->z[ib]) / (fg->z[ie] - fg->z[ib]);
    }
  }

  return (T);

}                               /* End of subroutine: fidap_interp_calc */

/****************************************************************/
/****************************************************************/
/* fidap_calc_tr_offset:                                        */
/* Subroutine to calculate the z offset to simulate tree rings  */
/****************************************************************/
/* Input Variables:                                             */
/*   *fg:		ptr to the FGrid_str structure that     */
/*       		holds all FIDAP geometry and T's data.  */
/*   time:		time in the CA model             [s]    */
/*      		                                        */
/* Output Variables:    NONE                                    */
/*      		                                        */
/* Returned Value:                                              */
/*   zoffset (implicit):offset in z direction, rtn through str  */
/****************************************************************/
/****************************************************************/
void fidap_calc_tr_offset (FGrid_str * fg, CA_FLOAT time)
{
  CA_FLOAT mag, t_s, dt, dtstep;
  int i, j, nr1, nr2;

  dtstep = time - fg->oldtime;

/************************************************/
/* Check that the x value is in range           */
/************************************************/
  for (i = 0; i < fg->ntring; i++) {

    t_s = fg->tring[i].v_f[1];
    dt = fg->tring[i].v_f[2];
    if ((time >= t_s) && (time < (dt + t_s))) {
      mag = fg->tring[i].v_f[0];
      nr1 = fg->tring[i].v_int[0];
      nr2 = fg->tring[i].v_int[1];
      for (j = nr1; j < nr2; j++) {
        fg->zoffset[j] += fg->v * mag * dtstep;
      }

    }
  }

  fg->oldtime = time;

}                               /* End of subroutine: fidap_calc_tr_offset */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_fidap_interp_calc_c ()
{
  static char const rcsid[] = "$Id: fidap_interp_calc.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_fidap_interp_calc_c subroutine */

/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:39  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:46  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:35  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:43  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:00  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:46  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:01  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 5.1  2002/10/17 16:52:37  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 5.0.14.1  2002/09/12 10:48:08  rcatwood
RCS Log:Changed pore routine to avoid getxyz routine
RCS Log:
RCS Log:Revision 5.0  2000/02/22 19:02:38  rcatwood
RCS Log:Before major multiblock fixup
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.2  1999/12/16 13:49:16  rcatwood
RCS Log:Combine X.Xu version to use W. Zhang program output into main trunk.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
