/* RCS ID:$Id: props.c 887 2006-03-01 18:21:01Z rcatwood $*/

/*Props subroutines.*/
/*Required definitions.*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "machine.h"
#include "blocks.h"
/*Diffusion Coefficient Calculation.*/

/*************************************/
/*Calculate the diffusion coefficient*/
/*depending upon temperature.        */
/*-- uses degrees C as input.        */
/*-- returns in m^2/ s$.             */
/*************************************/
extern CA_FLOAT global_pressure;

CA_FLOAT get_d_arr (CA_FLOAT temp, CA_FLOAT Qa, CA_FLOAT D_o)
{
      /************************/
  /* apply arrhenius eqn  */
  /* d = D0 exp(-Q/RT)    */
  /* input Q in J/Mol     */
  /*       D_o in M^2/sec */
  /*       temp in deg C  */
  /* output in M^2 / s    */
      /************************/

  CA_FLOAT tempK, D;

  tempK = temp + STD_TEMP;
  D = (CA_FLOAT) (D_o * (exp ((double) (-Qa / (8.31441 * tempK)))));
  /*printf("%.5g,%.5g,%.5g,%.5g,%.5g\n",Qa,D_o,temp,tempK,D) */

  return (D);
}

/* hard coded encapsulations for specific diffusion models */

CA_FLOAT get_ds_ti (CA_FLOAT temp)
{
  CA_FLOAT Qa, Do;

#define W13
#ifdef R61
  Do = 8.3e-2;                  /* in cm^2/s */
  Qa = 130;                     /* in KJ/mol */
#elif defined W13
  Do = 1.6;                     /* in cm^2/s */
  Qa = 201.7;                   /* in KJ/mol */
#endif
  return get_d_arr (temp, Qa * 1e3, Do * 1e-4);
}

CA_FLOAT get_dg_ti (CA_FLOAT temp)
{
  CA_FLOAT Qa, Do;

#define W13
#ifdef R61
  Do = 8.3e-2;                  /* in cm^2/s */
  Qa = 130;                     /* in KJ/mol */
#elif defined W13
  Do = 1.6;                     /* in cm^2/s */
  Qa = 201.7;                   /* in KJ/mol */
#endif
  Qa = Qa / 10;
  return get_d_arr (temp, Qa * 1e3, Do * 1e-4);
}

CA_FLOAT get_dl_ti (CA_FLOAT temp)
{
  CA_FLOAT Qa, Do;

  Do = 100.6;                   /* in cm^2/s */
  Qa = 201.7;                   /* in KJ/mol */
  return get_d_arr (temp, Qa * 1e3, Do * 1e-4);
}

/************************************************************/
CA_FLOAT get_reac_rate (CA_FLOAT temp)
{
  CA_FLOAT tempK, A, Q;
  CA_FLOAT rate;

   /*************************/
  A = 1e-5;
  Q = 274;
  tempK = temp + STD_TEMP;
   /*************************/
  rate = A * (EXP (Q / (GAS_CONST_SI * tempK)));
  return (rate);
}

/***********************************************************/

/* see Sung, Poirer et al, J Cr. Gro 226:363 (2001) for some */
/* values that may be useful for IN718 */
#if defined CELL_DIFF_COEFF
CA_FLOAT get_ds (CA_FLOAT temp)
{

  CA_FLOAT tempK, ds_cm, ds;

  tempK = temp + STD_TEMP;
  ds_cm = 1.1e-1 * EXP (-4922 / tempK);
  ds = ds_cm * 1e-4;
  return (ds);
}                               /*end of |get_ds| funciton */

CA_FLOAT get_dl (CA_FLOAT temp)
{

  CA_FLOAT tempK, dl_cm, dl;

  tempK = temp + STD_TEMP;
  dl_cm = 3.8e-2 * EXP (-2315 / tempK);
  dl = dl_cm * 1e-4;
  return (dl);
}                               /*end of |get_dl| function */
#elif defined CELL_DIFF_ARR
CA_FLOAT get_ds (CA_FLOAT temp, Solute_props * sp)
{

  CA_FLOAT ds;

  ds = get_d_arr (temp, sp->QaSol[0], sp->DoSol[0]);
  return (ds);
}                               /*end of |get_ds| funciton */

CA_FLOAT get_dl (CA_FLOAT temp, Solute_props * sp)
{

  CA_FLOAT dl;

  dl = get_d_arr (temp, sp->QaLiq, sp->DoLiq);
  return (dl);
}                               /*end of |get_dl| function */

#else

/* Clones of the function for */
/* testing with constand D coeff.*/

CA_FLOAT get_ds (CA_FLOAT temp)
{

  return (TEST_DS);

}                               /*end of TEST |get_ds| function */

CA_FLOAT get_dl (CA_FLOAT temp)
{

  return (TEST_DL);

}                               /*end of TEST |get_dl| funciton */

#endif /*|CELL_DIFF_COEFF| */

/*Average Diff Coeff. */

/*Uses a mixture rule to calculate averaged diffusion coefficient*/
/*for the partly solid region.*/

/*  Rewritten to reduce number of division operations*/
/*  this line is the biggest single bottleneck*/
/*  it would seem!*/
CA_FLOAT getav_d (CA_FLOAT dl, CA_FLOAT ds, CA_FLOAT fs)
{
  CA_FLOAT av_d;

  /*|av_d = 1/(fs/ds + ((1-fs)/dl));| */
  av_d = (ds * dl) / (fs * dl + (1 - fs) * ds);

  return (av_d);
}                               /*end of |getav_d| function */

/*Schiel Fraction solid calculation.*/
/*The Schiel equation for fraction solid as function of temperature:*/
/*$$ f_l = \biggl({T-T_p \over mC_o}\biggr)^{1 \over {k-1}}  $$*/
/*is used here.*/

/*Terminates at eutectic $f_s$, by switching to linear over a 2 degree range.*/

/*Eutectic $f_s$ should be calculated by calling this subroutine with the*/
/*input eutectic temperature as the |Tcell|, and 1.0 as the |eut_fs|, this*/
/*should be done outside the loop and stored in the block*/
/*as it does not change!.*/

CA_FLOAT schiel (CA_FLOAT Tcell)
{
  fprintf (stderr, "Sorry Schiel mode not available");
  exit (0);
#ifdef JUNK
  CA_FLOAT schiel_fs, schiel_fl;

  if (Tcell > T_LIQ) {
    return (0.0);
  }
#ifdef NLM_FORMULA
  else if (Tcell < T_SOL) {
    schiel_fs = 1.0;
  } else {
    schiel_fs = POW ((T_LIQ - Tcell) / (T_LIQ - T_SOL), ALLOY_EXPONENT);
  }
#else /*nomral scheil equation */
  if (Tcell > T_EUT) {
    schiel_fl = POW (((Tcell - TP) / (PD_SLOPE * CINITB)), (1 / (KB - 1)));
    schiel_fs = 1.0 - schiel_fl;
  } else if (Tcell > T_EUT - 2) {
    schiel_fs = FS_EUT + (T_EUT - Tcell) * ((1 - FS_EUT) / 2);
  } else {
    schiel_fs = 1.0;
  }
#endif /*NLM_FORMULA */
  return (schiel_fs);
#endif
}                               /*end of schiel function */

/*Schiel concentration eqn.*/
/* Find the concentration of solute B (alloy) according to the Schiel eqn. */
/*assuming the fl is known. This may be calculated from subroutine |schiel| but */
/*more generally may come out of some other growth model. */
CA_FLOAT find_sch_conc (CA_FLOAT tempK, CA_FLOAT fl)
{
  fprintf (stderr, "Sorry Schiel mode not available");
  exit (0);
#ifdef JUNK
  CA_FLOAT sch_c;
  CA_FLOAT Tcell;

  Tcell = tempK - STD_TEMP;
#ifdef NLM_FORMULA
  sch_c = CINITB - (T_LIQ - Tcell) / PD_SLOPE;
#else
  sch_c = CINITB * POW ((fl), (KB - 1));
#endif
  if (sch_c < CINITB)
    sch_c = CINITB;
  if (sch_c > MAX_B_CONC)
    sch_c = MAX_B_CONC;
  return (sch_c);
#endif
}

/* Growth subroutine.*/

/*This holds the equation that is used to calculate the growth. It may*/
/*be modified..$$ \delta{}x=GT_u^2\delta{}t$$.*/
/****************************************************************/
/* Subroutine to calc. growth as fn of undercooling             */
/* and return in mm                                             */
/****************************************************************/
/*dt is the time increment $\delta{}t$ and Tunder is the undercooling $T_u$ */
/* |gg_const| is the grain growth constant $G$ supplied by the user. */
/* From Jackson and Hunt */
CA_FLOAT growth (CA_FLOAT gg_const, CA_FLOAT gg_cub, CA_FLOAT dt, CA_FLOAT Tunder)
{

  if (gg_const == 0) {
    return (dt * FLAT_GROWTH);
  } else {
    return (dt * (gg_const * Tunder * Tunder + gg_cub * Tunder * Tunder * Tunder));
  }

}                               /* end of growth function */

/*  multi-component growth */
CA_FLOAT growth_primary (CA_FLOAT gg_const, CA_FLOAT gg_cub, CA_FLOAT dt, CA_FLOAT Tunder, int eut_flag, CA_FLOAT cell_fs_n_eut,
                         CA_FLOAT cell_fs_b_eut)
{

  CA_FLOAT ins_growth_p;

  if ((gg_const == 0) && (gg_cub == 0)) {
    ins_growth_p = dt * FLAT_GROWTH;
  } else {
    ins_growth_p = dt * (gg_const * Tunder * Tunder);
  }

  return (ins_growth_p);

}                               /* end of primary growth function */

/************************************************************/

CA_FLOAT growth_eutectic (CA_FLOAT gg_const, CA_FLOAT gg_cub, CA_FLOAT dt, CA_FLOAT Tunder, int eut_flag, CA_FLOAT cell_fs_n_eut,
                          CA_FLOAT cell_fs_b_eut)
{

  CA_FLOAT ins_growth_e;

  if ((gg_const == 0) && (gg_cub == 0)) {
    ins_growth_e = dt * FLAT_GROWTH;
  } else {
    ins_growth_e = dt * ((gg_const * Tunder * Tunder) + (gg_cub * Tunder * Tunder * Tunder));
  }

  return (ins_growth_e);

}                               /* end of eutectic growth function */

/************************************************************/

/**********************/
/*Find Local Liquidus.*/
/**********************/
/*Use appropriately simplified phase diagram to find the liquidus for the cell.*/
/*Used if |phase_diag_on| option set*/
CA_FLOAT cell_liq_calc (CA_FLOAT cell_conc, Mat_str * mp)
{
  CA_FLOAT TliqCell;
 
  /* assuming the gas has no effect */
  if (mp->gasprops.m_solute[0] == 0) {
    TliqCell = mp->tp + (mp->alloyprops[0].m_solute[0] * cell_conc);
    if (TliqCell <= mp->alloyprops[0].T_eut)
    TliqCell = mp->alloyprops[0].T_eut;
  } else {
    fprintf (stderr, "SORRY:cell_liq_calc: gas effect on liquidus not implemented.\n");
    fprintf (stderr, "Please rewrite cell_liq_calc if that is what you need.\n");
    fprintf (stderr, "Otherwise set m_solute (liquidus slope) to zero for the gas.\n");
    exit (0);
  }
  return (TliqCell);
}

/* THUINET 0106 */
CA_FLOAT cell_liq_calc_poly (CA_FLOAT cell_conc, Mat_str * mp, int ele_1)
{
  CA_FLOAT TliqCell;
  int i;

  /* assuming the gas has no effect */
  if (mp->gasprops.m_solute[0] == 0) {
    TliqCell = mp->tp_poly[0];
    for (i = 0; i < ele_1; i++) {
      TliqCell += (mp->alloyprops[i].m_solute[0] * cell_conc);
    }
  } else {
    fprintf (stderr, "SORRY:cell_liq_calc: gas effect on liquidus not implemented.\n");
    fprintf (stderr, "Please rewrite cell_liq_calc if that is what you need.\n");
    fprintf (stderr, "Otherwise set m_solute (liquidus slope) to zero for the gas.\n");
    exit (0);
  }
  return (TliqCell);
}




/*Find local Liquidus using the approximated multi component phase diagram*/
/* a hyper surface is used */

#ifdef CHIRAZI_MULTI
CA_FLOAT cell_liq_calc_multi (CA_FLOAT ** conc_multi, int index, BB_struct * bp)
{
#ifdef MULTICOMP
  CA_FLOAT TliqCell;
  MultiS_struct *ms;
  FILE *liq_out;
  Ctrl_str *cp;
  int ele_num, ele_1, i;

  cp = bp->ctrl;
  ms = &(bp->MultiSvals);
  ele_num = cp->NUM_COMP;
  ele_1 = ele_num - 1;

  liq_out = fopen ("liquidus.out", "a+");

  TliqCell = TP;
  for (i = 0; i < ele_1; i++) {
    TliqCell += (conc_multi[i][index]) * (ms->slope_multi[i]);
/*fprintf(liq_out,"%f  \t %f  \t %d \t \n",TliqCell,conc_multi[i][index],index);*/
  }
  if (TliqCell <= T_EUT_1)
    TliqCell = T_EUT_1;

  fclose (liq_out);

  return (TliqCell);
#endif
}
#endif /*CHIRAZI_MULTI*/

/*Find H saturation.*/

/*This uses the formula */
/*$$ \log S_l = - {2760 \over T} + 2.796 $$ which */
/*is on page 3 (equation 2.2) of PDL thesis,*/
/*referenced to Ransley and Neufeld [1948].*/
/*and the correction factor for si:*/
/*$$ f = 10^{(-0.0119 * \%Si)}$$*/
/*eqn 2.5 PDL thesis, ref. to Doutre(1991).*/

/*Currently assuming constant partition coefficient for H hence ignoring */
/*the equation for soulubility in the solid. */
CA_FLOAT find_sat (Mat_str * mp, CA_FLOAT cell_tempK, CA_FLOAT cell_si, CA_FLOAT cell_fs)
{
  CA_FLOAT logss, sl, ss, logsl, f, cell_sat, cell_satMCUB;
  const CA_FLOAT mpm_cub = MPMETERCUB;

  f = POW (10, (DOUTRE_F * (MIN (cell_si, mp->alloyprops[0].c_eut))));
#ifdef RANSLEY
  logsl = -(2760 / cell_tempK) + 2.796;
  sl = POW (10, logsl);         /* ml/100g STP */
#else /*IMABAYASHI*/
    logsl = -(2392 / cell_tempK) + 3.256;
  sl = POW (10, logsl);         /* cc/1000g STP */
  sl *= 0.1;
#endif /*Ransley/Imabayashi */
  cell_sat = f * sl;
  cell_satMCUB = cell_sat * MPMETERCUB; /*conv to SI $mol\over m^3$ */

  cell_satMCUB *= SQRT (global_pressure);

  return (cell_satMCUB);
}

/************************the modified correction factor for********/
/************************multi component diffusion*****************/
#ifdef SENS
CA_FLOAT find_sat_multi (BB_struct * bp, CA_FLOAT cell_tempK, CA_FLOAT ** conc_multi, CA_FLOAT cell_fs, CA_FLOAT sens_P, int index)
#else
CA_FLOAT find_sat_multi (BB_struct * bp, CA_FLOAT cell_tempK, CA_FLOAT ** conc_multi, CA_FLOAT cell_fs, int index)
#endif
{
  CA_FLOAT logss, sl, ss, logsl, f, cell_sat, cell_satMCUB;
  MultiS_struct *ms;
  Ctrl_str *cp;
  int ele_num, ele_1;
  int i, j;
  CA_FLOAT sum;
  CA_FLOAT doutre[3], local_max[3];

#ifdef MULTICOMP
  local_max[0] = MAX_B_CONC_1;
  local_max[1] = MAX_B_CONC_2;
  local_max[2] = MAX_B_CONC_2;
#endif

  doutre[0] = -0.0119; /* silicon */
  doutre[1] = -0.0269; /* copper  */
  doutre[2] = 0.017;   /* magnesium*/ /* from Alcan alscan documentation */

  cp = bp->ctrl;
  ms = &(bp->MultiSvals);

  ele_num = cp->NUM_COMP;
  ele_1 = ele_num - 1;

  sum = 0.0;

/************************************************************/
  for (i = 0; i < ele_1; i++) {
    sum += doutre[i] * (MIN (conc_multi[i][index], local_max[i]));
  }
  f = POW (10, sum);
#ifdef RANSLEY
  logsl = -(2760 / cell_tempK) + 2.796;
  sl = POW (10, logsl);         /* ml/100g STP */
#else /*Imbayashi */
  logsl = -(2392 / cell_tempK) + 3.256;
  sl = POW (10, logsl);         /* cc/1000g STP */
  sl *= 0.1;
#endif /*Ransley/Imabayashi */
  cell_sat = 1.0 * f * sl;
  cell_satMCUB = cell_sat * MPMETERCUB; /*conv to SI $mol\over m^3$ */

  cell_satMCUB *= SQRT (global_pressure);
#ifdef SENS
  cell_satMCUB *= SQRT (sens_P);
#endif

  return (cell_satMCUB);
}


CA_FLOAT find_sat_poly (BB_struct * bp, int sbnum, CA_FLOAT cell_tempK)
{
  CA_FLOAT logss, sl, ss, logsl, f, cell_sat, cell_satMCUB;
  Ctrl_str *cp;
  SB_struct *sp;
  int ele_num, ele_1;
  int isol;
  CA_FLOAT sum;
  CA_FLOAT *ncl_poly[NSOLMAX];

  sp = bp->sb[sbnum];           /*subblock pointer */

  cp = bp->ctrl;
  ele_num = cp->NUM_COMP;       /* number of elements in the alloy */
  ele_1 = ele_num - 1;

  for (isol = 0; isol < ele_1; isol++) {        /*loop on solutes by Ludovic THUINET */
    ncl_poly[isol] = sp->c_sol_poly[isol];
  }  /* end loop on solutes */

  sum = 0.0;

  for (isol= 0; isol < ele_1; isol++) {
    sum +=bp->mprops.alloyprops[isol].coef_doutre *(*ncl_poly[isol]);
  }
  f = POW (10, sum);
#ifdef RANSLEY
  logsl = -(2760 / cell_tempK) + 2.796;
  sl = POW (10, logsl);         /* ml/100g STP */
#else /*Imbayashi */
  logsl = -(2392 / cell_tempK) + 3.256;
  sl = POW (10, logsl);         /* cc/1000g STP */
  sl *= 0.1;
#endif /*Ransley/Imabayashi */
  cell_sat = 1.0 * f * sl;
  cell_satMCUB = cell_sat * MPMETERCUB; /*conv to SI $mol\over m^3$ */

  cell_satMCUB *= SQRT (global_pressure);
#ifdef SENS
  cell_satMCUB *= SQRT (sens_P);
#endif

  return (cell_satMCUB);
}

/*Find cubic root.*/

/*Radius finding routine by Peter Lee using Mathematica.*/
/* DISUSED : replaced by iterative (Newton method) in findroot.c */

/* Solve for the one real root of $ar^3+br^2+c=0$   		*/
/* as given analytically from Mathematica.\par{}*/
/*{\bf Inputs:} temperature (${\rm^\circ K}$), amount of gas*/
/*(mol),surface tension $(\gamma)$ (${N\over m}$ or ${pa . m}$), old radius (m).\par{}*/
/*{\bf Output:} new radius (m).*/

/* if only it did a better job of reducing...\par */
/* {\obeylines\parindent = 1cm \codefont*/
/* x1 = -b/(3*a) + cbrt(2.0)*pow(b,2)/ (3*a*cbrt(-2*pow(b,3) 	 */
/* 	- 27*pow(a,2)*c + pow(3,1.5)*a*sqrt(4*pow(b,3)*c + 	     */
/*	27*pow(a,2)*pow(c,2)))) + cbrt(-2*pow(b,3) -		              */
/*	27*pow(a,2)*c + pow(3,1.5)*a*sqrt(4*pow(b,3)*c +	            */
/*	27*pow(a,2)*pow(c,2)))/(3*cbrt(2.0)*a);			               */
/*}*/
/* Local definitions: (Numerical constants moved to |machine.h|) */
#ifdef SENS
CA_FLOAT r_from_ideal_gas (CA_FLOAT temperature, CA_FLOAT n_gmol, CA_FLOAT gammaSI, CA_FLOAT old_rSI, CA_FLOAT sens_PRESS)
#else
CA_FLOAT r_from_ideal_gas (CA_FLOAT temperature, CA_FLOAT n_gmol, CA_FLOAT gammaSI, CA_FLOAT old_rSI)
#endif
{
#define P_ATM		1.0     /* ambient pressure in atm */
#define TWO_GAMMA_ATM	0.00177646      /* surf.ten in Atm x cm =|G_PA/101325| */
#define LMT_SQRT	1.0e-18 /* exact soln only if |tmp>LMT_SQRT| */

  double a, b, c, tmp, tcbrt, ansCM, old_r;     /* must stay as double */
  extern double sqrt (double);
  extern double cbrt (double);

/*NOTE: if |P_ATM| and |GAMMA| remain constant, this can be speed	*/
/*up quite a bit, most becomes const...			*/

/* {\bf NOTE:} If |tmp| = $(4b^3c + 27a^2c^2)\to 0$, only an approximate	*/
/* 	root is used. Also, the value may be inexact near the	*/
/*	transition!		 				*/
  old_r = (double) (old_rSI * 100);

#ifdef JUNK
  a = P_ATM;
#endif
  a = global_pressure;
#ifdef SENS
  a *= sens_PRESS;
#endif
  b = 2 * gammaSI * CONV_ATM_CM;
  c = -THREE_BY_4PI * GAS_CONST_ATM * n_gmol * temperature;
  tmp = 4 * b * b * b * c + 27 * a * a * c * c;
  if (tmp > LMT_SQRT) {
    tcbrt = cbrt (-2 * b * b * b - 27 * a * a * c + POW_3_TO_1PT5 * a * sqrt (tmp));
    ansCM = (CA_FLOAT) (-b / (3.0 * a) + CBRT2 * b * b / (3.0 * a * tcbrt) + tcbrt / (3.0 * CBRT2 * a));
  } else
    ansCM = (CA_FLOAT) sqrt ((-c - a * old_r * old_r * old_r) / b);
  return (CA_FLOAT) (ansCM / 100);
}

/*Find maturity function.*/
/*not currentl used.*/

CA_FLOAT get_mat (CA_FLOAT thresh, CA_FLOAT delta_t, CA_FLOAT ss, CA_FLOAT oldmat)
{
  CA_FLOAT mat;

  mat = oldmat + delta_t * (ss - thresh);
  return (mat);
}

#ifdef TEST_PROPS
void main (void)
{
  CA_FLOAT t;

  for (t = 300; t < 1500; t += 50) {
    printf (t, get_ds_ti (t));
  }
}
#endif /*TEST_PROPS */
/* rcs id subroutine.*/
/*Little subroutine to include the |rcs Id| in the object code.*/
/*This can also be called to print or include the Id in a file.*/

char const *rcs_id_props_c ()
{
  static char const rcsid[] = "$Id: props.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* RCS ID:$Id: props.c 887 2006-03-01 18:21:01Z rcatwood $*/
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3.2.6  2006/02/01 14:16:25  lthuinet
RCS Log:#update change of the calculation of supersaturation for gas in multicomponent
RCS Log:
RCS Log:Revision 10.3.2.5  2006/01/23 16:22:20  rcatwood
RCS Log:Comments for the Doutre coefficients
RCS Log:
RCS Log:Revision 10.3.2.4  2006/01/23 11:47:59  rcatwood
RCS Log:Changed to use reference temperature located in matprops structure
RCS Log:
RCS Log:Revision 10.3.2.2  2006/01/18 12:01:17  lthuinet
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 10.3.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.2.4.3  2004/04/07 11:18:31  rcatwood
RCS Log:Fixed several division errors, added section to activate f.p.e. trapping
RCS Log:
RCS Log:Revision 9.2.4.2  2004/03/09 11:30:53  rcatwood
RCS Log:Changed to read materials properties from separate files for alloy and gas
RCS Log:
RCS Log:Revision 9.2.4.1  2004/03/04 11:29:25  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.2  2003/10/16 12:46:33  rcatwood
RCS Log:Pore volume feedback for Procast
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:38  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.4  2003/05/06 15:48:08  rcatwood
RCS Log:Altered linear binary phase diagram usage to consistently use the values input from the control files rather than the header files.
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:45  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:01  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.4.8.4  2002/11/04 11:22:05  rcatwood
RCS Log:Check-in after Ali has gone
RCS Log:
RCS Log:Revision 7.4.8.3  2002/08/27 15:08:14  chirazi
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 7.4  2001/05/31 16:24:45  rcatwood
RCS Log:changed M macro to PD_SLOPE (Easier to find!)
RCS Log:
RCS Log:Revision 7.3  2001/03/13 10:36:46  rcatwood
RCS Log:Ready to Merge with anchou version
RCS Log:Fixed not-casting problems,
RCS Log:seperate pore stats file.
RCS Log:
RCS Log:Revision 7.2  2001/02/19 11:03:49  rcatwood
RCS Log:fixed missing dt in growth subroutine
RCS Log:
RCS Log:Revision 7.1  2001/02/16 20:30:49  rcatwood
RCS Log:Added some commenting for DAS and Limrad
RCS Log:Removed some JUNK
RCS Log:
RCS Log:Revision 7.1  2001/02/16 20:29:17  rcatwood
RCS Log:Added comments to show DAS and Limrad areas.
RCS Log:Removed some ifdef JUNK sections
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.1  2000/09/25 17:35:30  rcatwood
RCS Log:Before synch with common version, Used for PORE_00 paper
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.2  2000/06/27 11:41:19  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.4  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.2  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.7  2000/02/15 15:29:11  rcatwood
RCS Log:Version after McWasp submitted
RCS Log:
RCS Log:Revision 4.6  2000/01/06 10:48:11  rcatwood
RCS Log:Fixed bug -- prototype in sb_diffuse_gas
RCS Log:
RCS Log:Revision 4.5  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.3  1999/12/20 13:11:23  rcatwood
RCS Log:Alloy and Gas outputs
RCS Log:
RCS Log:Revision 4.2  1999/12/16 19:15:49  rcatwood
RCS Log:Alloy and gas diffusion working, pores working. 
RCS Log:Changed file name for conc. output A for alloy C for gas conc.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.3  1999/12/16 13:01:54  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
