/* #define DEBUG */
/****************************************************************/
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      is the property of Imperial College. Any use,           */
/*      All Rights Reserved                                     */
/*      THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM    */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      This material contains CONFIDENTIAL INFORMATION that    */
/*      duplication or disclosure not specifically authorized   */
/*      by Imperial College is strictly prohibited.             */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*	email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/* READ_CTRL.C:   (Part of CA code)                             */
/* Subroutine to read the main controlling info from a file.    */
/* The file is formated, using the style:                       */
/*    # as first character:     Comment                         */
/* and values are input in the format:                          */
/*    command value  #comments                                  */
/****************************************************************/
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Jul 1, 1998                                                  */
/****************************************************************/

/****************************************************************/
/* Versions maintained with RCS                                 */
/* Version 1.0: Aug. 13, pdl                                    */
/****************************************************************/
/*RCS Id:$Id: read_ctrl.c 1482 2009-11-25 21:08:47Z  $*/

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include "machine.h"
#include "blocks.h"

int combine_ctrl (Ctrl_str * cp);
int read_ctrl (char *filename, Ctrl_str * cp);
int read_ctrl_vals (char *filename, Ctrl_str * cp);
void set_ctrl_defaults (Ctrl_str * cp);

void set_ctrl_defaults (Ctrl_str * cp)
{
  /* copy the compilation flags into a variable -- requires */
  /* script in Makefile to save the flags */
  char *Cflags;
  int i;
#include "getcflags.inc"

   /*********************************************************/
  /* Set all the default values...                         */
   /*********************************************************/
  /* from read_ctrl.h */
  cp->solo = D_SOLO;
  cp->cap = D_CAP;
  cp->post = D_POST;
  cp->do_conc_prof = FALSE;
  cp->phase_diag_on = D_PHASE_DIAG;
  cp->global_undercooling = D_GLOBAL_UNDERCOOLING;
  cp->particle = D_PARTICLE;
  cp->restart = D_RESTART;
  cp->restart_gas_on = D_RESTART_PORE;
  cp->restart_gas_on = D_RESTART_GAS;
  cp->diffuse = D_DIFFUSE;
  cp->das_limrad = D_DAS_LIMRAD;
  cp->diffuse_alloy = D_DIFFUSE_ALLOY;
  cp->diffuse_alloy_multi = D_DIFFUSE_ALLOY_MULTI;
  cp->temp_dep_diff = D_TEMP_DEP_DIFF;
  cp->interpolate = D_INTERPOLATION;
  cp->thermocalc = D_THERMOCALC;
  cp->diffuse_step = D_DIFFUSE_STEP;
  cp->temp_lookup = 0;
  cp->window_moving = D_WINDOW_MOVING;  /*by Wei WANG on 11-07-02 */
  cp->init_cont = D_INIT_CONT;  /*by Wei WANG on 11-07-02 */
  cp->decentred_octahedron = D_DECENTRED_OCTAHEDRON;    /*by Wei WANG on 11-07-02 */
  cp->melt_back = D_MELT_BACK;  /* melt back option, xly 20040802 */
  cp->isotherm_curv = D_ISOTHERM_CURV;  /* option for producing transient isotherm curvature, xly 20040802 */
  cp->pr_lookup = 0;
  cp->diff_rgbmode = D_DIFF_RGBMODE;
  cp->rgbgrey = FALSE;
  cp->input = D_INP;
  cp->t_input = FALSE;
  cp->time_dump = FALSE;
  cp->time_exp = 0;
  cp->time_unit = 1;
  cp->fgrid_input = FALSE;
  cp->con_cast = D_CON_CAST;
  cp->n_neigh = NEIGH_6;
  cp->ca_method = CA_PULL;
  cp->seed = D_SEEDVAL;
  cp->scheil = FALSE;
  cp->swap_xy = 0;
  cp->curvature_2D=0;
  cp->curvature_3D=0; 
  cp->extinput = 0;


  /* default file names and allocation of strings */
  cp->fn_block_restart = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_block_restart, "%s\0", (D_BLOCK_RESTART_FILE));
  cp->fn_geo = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_geo, "%s\0", (D_GEO_FILE));
  cp->fn_mat = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_mat, "%s\0", (D_MAT_FILE));
  cp->fn_base = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_base, "%s\0", "CHOOSE_a_better_file_name");
  cp->fn_phadia = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_phadia, "%s\0", (D_PHADIA_FILE));
  cp->fn_inp = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_inp, "%s\0", (D_INP_FILE));
  cp->fn_fgrid = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_fgrid, "%s\0", (D_FG_FILE));

  cp->fn_extinp = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (cp->fn_extinp, "%s\0", (D_EXTINP_FILE));

  cp->fg_tr = TRANSIENT;

  /* output variables */
  cp->write_block = TRUE;
  cp->excel = D_EXCEL;
  cp->nsbslice = 0;
  cp->tempslice = D_TEMPSLICE;
  cp->nbbslice = 0;
  cp->grainslice = 0;
  cp->slice_dmp_freq = D_SLICE_FREQ;
  cp->blk_dmp_freq = D_BLK_FREQ;
  cp->floatdump = D_CA_FLOATDUMP;
  cp->scr_dmp_freq = D_SCREEN_FREQ;
  cp->rgbmode = D_RGB_RANDOM;

  /*plt file output + fluidflow control by lyuan*/
  cp->plt_dmp_freq = 0;
  cp->vtk_dmp_freq = 0;
  cp->point_dmp_freq = 0;
  cp->line_dmp_freq = 0;
  cp->flowdiffusion = 0;
  cp->fluidflow = 0;
  cp->ffheattran = 0;
  cp->ffstepgap = 1;
  cp->remelt = 0;
  cp->fstarttime = 0;
  /* test mode variables */
  cp->fixed_Pore = FALSE;
  cp->nfPore = 0;
  cp->fixed_nuc = FALSE;
  cp->nfnuc = 0;
  cp->coolrate = FALSE;
  cp->delT = D_DEL_TEMP;
  cp->fs_finish = D_FS_FINISH;

  /* CA - PHYSICA */
  cp->fn_ca_phy = strdup( D_CA_PHY_FILE );
  cp->physica = D_PHYSICA;

  cp->wmstarttime = 0.0;
  cp->maxucmap = 0;
  cp->maxucmap_freq = 0;
  cp->nptrack = 0;
  cp->nlntrack = 0;

  cp->lntrack = (int **)calloc(MAX_CTRL, sizeof(int *));
  for(i=0;i<MAX_CTRL;i++) cp->lntrack[i]=(int *)calloc(7, sizeof(int));
}

int read_ctrl_vals (char *filename, Ctrl_str * cp)
{
  char *line;
  char *token;
  FILE *fp;
  char *sep;
  int i, error;
  int rflag = 0;
  line = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sep = (char *) calloc (MAX_STRING_LEN, sizeof (char));
  sprintf (sep, " ,;\t\n\r");

   /*********************************************************/
  /* Open the control file                                 */
   /*********************************************************/
  if ((fp = fopen (filename, "r")) == NULL) {
    fprintf (stderr, "Error: read_ctrl: can't open input file [%s]\n", filename);
    return (1);
  }

  while (fgets (line, MAX_STRING_LEN, fp) != NULL) {
    /* ignore comment and blank lines */
    if (line[0] == '%' || line[0] == '#' || (token = strtok (line, sep)) == NULL) {
      continue;

      /*********************************************************/
      /* All values in the Ctrl_str structure                  */
      /*********************************************************/
      /* SoloMode */
    } else if (strcasecmp (token, "SoloMode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->solo = atoi (token);
      /* CAPMode */
    } else if (strcasecmp (token, "CAPMode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->cap = atoi (token);
      /* PostProcessingMode */
    } else if (strcasecmp (token, "PostProcessingMode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->post = atoi (token);
      /* CA_FEEDBACK */
    } else if (strcasecmp (token, "CA_FEEDBACK") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        cp->ca_feedback = atoi (token);

      /*Nucleation on the mould surface */
    } else if (strcasecmp (token, "MouldNuc") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->mould_nuc = atoi (token);

      /* Source of solute at the mould */
      /* uses enum variable for the type of function */
      /* typedef in machine.h */
    } else if (strcasecmp (token, "MouldSrc") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->mould_src = (atoi (token));

      /*Perturb the mould source ? */
    } else if (strcasecmp (token, "MouldSrcPert") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->mould_src_pert = atoi (token);

      /* Source magnitude of solute at the mould */
    } else if (strcasecmp (token, "MouldSourceValue") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->mould_source_value = atof (token);

      /* Source magnitude of solute prerturbation at the mould */
    } else if (strcasecmp (token, "MouldSourcePertValue") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->mould_source_pert = atof (token);

      /* Source spatial frequency of solute at the mould */
    } else if (strcasecmp (token, "MouldSourceFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->mould_source_freq = atof (token);

      /* ConCastMode -- directional solidification */
    } else if (strcasecmp (token, "ConCastMode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->con_cast = atoi (token);
      
      /*Physcia Mode -- coupled with Physica Macro*/
    } else if (strcasecmp (token, "PhysicaMode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
	cp->physica = atoi (token);			

      /*Fluidflow Modle -- by lyuan*/
    } else if (strcasecmp (token, "FluidflowMode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fluidflow = atoi (token);
      
      /*Fluidflow Modle -- by lyuan*/
    } else if (strcasecmp (token, "FFStartTime") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fstarttime = atof (token);
			
      /*Fluid flow solved each *** steps*/
    } else if (strcasecmp (token, "FFSTEPGAP") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->ffstepgap = atoi (token);
			
      /* remelt option under concast mode, default: off */
    } else if (strcasecmp (token, "REMELT") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
	cp->remelt = atoi (token);
      /* Particle */
    }else if (strcasecmp (token, "Particle") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->particle = atoi (token);
      /* das_limrad */
    } else if (strcasecmp (token, "das_limrad") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->das_limrad = atoi (token);
      /* Diffuse gas */
    } else if ((strcasecmp (token, "diffuse") == 0) ||
               /* alternate key for gas diffusion */
               (strcasecmp (token, "diffuseGas") == 0)) {
      int old_dif;

      if (cp->restart)
        old_dif = cp->diffuse;
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diffuse = atoi (token);
      /* check for restart with gas turned on */
      if (cp->restart && (old_dif == 0) && (cp->diffuse == 1)) {
        cp->restart_gas_on = 1;
      }

      /* Diffuse Alloy */
    } else if (strcasecmp (token, "diffuseAlloy") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diffuse_alloy = atoi (token);

      /* Number of alloying elements *//* L. THUINET 07/02/05 */
    } else if (strcasecmp (token, "NUMCOMP") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->NUM_COMP = atoi (token);

      /* Number of solid phases for multiphase solidification *//* L. THUINET 04/05 */
    } else if (strcasecmp (token, "NUMPHS") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->NUM_PHS = atoi (token);

      /* Number of equilibria for multiphase solidification *//* L. THUINET 04/05 */
    } else if (strcasecmp (token, "NUMEQ") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->NUM_EQ = atoi (token);

      /* Number of distinct gaussian nucleation laws *//* L. THUINET 10/05 */
    } else if (strcasecmp (token, "NUM_NUC_LAW") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->NUM_NUC_LAW = atoi (token);

      /* Diffuse Alloy Poly *//* L. THUINET 04/02/05 */
    } else if (strcasecmp (token, "diffuseAlloyPoly") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diffuse_alloy_poly = atoi (token);

      /* Phase Diagram */
    } else if (strcasecmp (token, "PhaseDiagram") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->phase_diag_on = atoi (token);

      /* Global Undercooking */
    } else if (strcasecmp (token, "FluidFlowOnDiffusion") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->flowdiffusion = atoi (token);

      /* FLuid flow on the diffusion */
    } else if (strcasecmp (token, "FFHeattran") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->ffheattran = atoi (token);

      /* Heat transfer model */
    } else if (strcasecmp (token, "GlobalUndercooling") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->global_undercooling = atoi (token);
      /* Show Eutectic */
    } else if (strcasecmp (token, "ShowEutectic") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->show_eut = atoi (token);
      /* fraction solid stop */
    } else if (strcasecmp (token, "FsFinish") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fs_finish = (CA_FLOAT) atof (token);

      /* swap-xy axis for directional */
    } else if (strcasecmp (token, "SwapXy") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->swap_xy = (CA_FLOAT) atof (token);
      /* diffuse_step */
    } else if (strcasecmp (token, "diffuse_step") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diffuse_step = atoi (token);

      /* Window Moving. by Wei WANG on 11-07-02 */
    } else if (strcasecmp (token, "Window_moving") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->window_moving = atoi (token);
      /* starting time for window moving... LY*/
    } else if (strcasecmp (token, "WM_starttime") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->wmstarttime = atof (token);	      
      /* Initial or continuing calculation. by Wei WANG on 11-07-02 */
    } else if (strcasecmp (token, "Init_cont") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->init_cont = atoi (token);

      /* Decentred Octahedron. by Wei Wang on 11-07-02 */
    } else if (strcasecmp (token, "Decentred_octahedron") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->decentred_octahedron = atoi (token);
      /* option for change pulling velocity, for melt-back only, xly 20040802 */
    } else if (strcasecmp (token, "Melt_Back") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->melt_back = atoi (token);

      /* option for producing transient isotherm curvature, xly 20040802 */
    } else if (strcasecmp (token, "Isotherm_Curv") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->isotherm_curv = atoi (token);
      /* schiel */
    } else if (strcasecmp (token, "Scheil") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->scheil = atoi (token);

      /* N_Neighbours */
    } else if (strcasecmp (token, "N_Neighbours") == 0) {
      if ((token = strtok (NULL, sep)) != NULL) {
        cp->n_neigh = atoi (token);
        switch (cp->n_neigh) {
        case NEIGH_6:
          fprintf (stderr, "N_Neighbours: set equal to 6.\n");
          break;
        case NEIGH_8:
          fprintf (stderr, "N_Neighbours: set equal to 8.\n");
          break;
        case NEIGH_10:
          fprintf (stderr, "N_Neighbours: set equal to 10.\n");
          break;
        case NEIGH_26:
          fprintf (stderr, "N_Neighbours: set equal to 27.\n");
          break;
        default:
          fprintf (stderr, "ERROR: unknown neighbourhood [%d], set to 6.\n", cp->n_neigh);
          cp->n_neigh = 6;
          break;
        }                       /* end of Flag switch */

      }
      /* CAPFileName */
    } else if (strcasecmp (token, "CAPFileName") == 0) {
      free (cp->fn_cap);
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_cap = strdup (token);
      /* Restart Block file name */
    } else if (strcasecmp (token, "BlockRestartFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        sprintf (cp->fn_block_restart, "%s\0", token);
      /* GeoFileName */
    } else if (strcasecmp (token, "GeoFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        sprintf (cp->fn_geo, "%s\0", token);
      /* MatFileName */
    } else if (strcasecmp (token, "MatFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        sprintf (cp->fn_mat, "%s\0", token);
      /* InpFileName */
    } else if (strcasecmp (token, "InpFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL) {
        sprintf (cp->fn_inp, "%s\0", token);
        cp->input = TRUE;
      }
    } else if (strcasecmp (token, "ExternalFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL) {
        sprintf (cp->fn_extinp, "%s\0", token);
        cp->extinput = TRUE;
    }
		  
      /* temperatureinput filename(string) */
    } else if (strcasecmp (token, "TemperatureInputFile") == 0) {
      cp->t_input = TRUE;
      if ((token = strtok (NULL, " ,;\t")) != NULL) {
        cp->fn_t_input = strdup (token);
        fprintf (stderr, "readgeoplus: TemperatureInputFile set to [%s].\n", cp->fn_t_input);
      }
      /* BaseFileName */
    } else if (strcasecmp (token, "BaseFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        sprintf (cp->fn_base, "%s\0", token);
      /* FgridFileName */
    } else if (strcasecmp (token, "FgridFileName") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        sprintf (cp->fn_fgrid, "%s\0", token);
      cp->fgrid_input = TRUE;
      /* FgridFileName */
    } else if (strcasecmp (token, "FgridTransient") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fg_tr = atoi (token);
      /* gas properties filename */
    } else if (strcasecmp (token, "GasPropsFile") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_gas = strdup (token);
      fprintf (stderr, "read_ctrl:gas solute properties file is %s\n", cp->fn_solprops_gas);

      /* alloy properties filename */

    } else if (strcasecmp (token, "AlloyPropsFile0") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[0] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 0  is %s\n", cp->fn_solprops_alloy[0]);
    } else if (strcasecmp (token, "AlloyPropsFile1") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[1] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 1 is %s\n", cp->fn_solprops_alloy[1]);
    } else if (strcasecmp (token, "AlloyPropsFile2") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[2] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 2 is %s\n", cp->fn_solprops_alloy[2]);
    } else if (strcasecmp (token, "AlloyPropsFile3") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[3] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 3 is %s\n", cp->fn_solprops_alloy[3]);
    } else if (strcasecmp (token, "AlloyPropsFile4") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[4] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 4 is %s\n", cp->fn_solprops_alloy[4]);
    } else if (strcasecmp (token, "AlloyPropsFile5") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[5] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 5 is %s\n", cp->fn_solprops_alloy[5]);
    } else if (strcasecmp (token, "AlloyPropsFile6") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[6] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 6 is %s\n", cp->fn_solprops_alloy[6]);
    } else if (strcasecmp (token, "AlloyPropsFile7") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[7] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 7 is %s\n", cp->fn_solprops_alloy[7]);
    } else if (strcasecmp (token, "AlloyPropsFile8") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[8] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 8 is %s\n", cp->fn_solprops_alloy[8]);
    } else if (strcasecmp (token, "AlloyPropsFile9") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->fn_solprops_alloy[9] = strdup (token);
      fprintf (stderr, "read_ctrl:alloy solute properties file 9 is %s\n", cp->fn_solprops_alloy[9]);


      /* pressure lookup? */
    } else if (strcasecmp (token, "PrLookup") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->pr_lookup = atoi (token);
      /* Porosity */
    } else if (strcasecmp (token, "Pore") == 0) {
      int old_pore;

      if (cp->restart)
        old_pore = cp->pore;
      if ((token = strtok (NULL, sep)) != NULL)
        cp->pore = atoi (token);
      /* check for restart with pore turned on */
      if (cp->restart && (old_pore == 0) && (cp->pore == 1)) {
        cp->restart_pore_on = 1;
      }

      /* RandomSeed Number */
    } else if (strcasecmp (token, "RandSeedVal") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->seed = (int32_t) (atoi (token));
      if (cp->seed == -2) {
        int fd;

        fd = open ("/dev/random", O_RDONLY);
        read (fd, &(cp->seed), sizeof (int32_t));
        fprintf (stderr, "Kernel random seed requested: using %i\n", cp->seed);
        close (fd);
      }
      /* PLIC Curvature in 3D option */
      } else if (strcasecmp(token,"Curvature_3D") == 0) {
         if ((token = strtok(NULL, sep)) != NULL)
              cp->curvature_3D = atoi(token);
      /* PLIC Curvature in 2D option */
      } else if (strcasecmp(token,"Curvature_2D") == 0) {
         if ((token = strtok(NULL, sep)) != NULL)
              cp->curvature_2D = atoi(token);


      /*********************************************************/
      /* All Output settings                                   */
      /*********************************************************/
      /* PrintExcel 1/0 (true/false) */
    } else if (strcasecmp (token, "PrintExcel") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->excel = atof (token);

      /* create 3d blocks for AVS */
    } else if (strcasecmp (token, "WriteBlock") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->write_block = atof (token);

      /* Dump pore infor for subblock */
    } else if (strcasecmp (token, "PoreDumpSb") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->pore_dump_sb = atoi (token);

      /* Dump pore infor for subblock */
    } else if (strcasecmp (token, "GrainSlice") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->grainslice = atoi (token);

      /* conc_prof sb# zslice# row# */
    } else if (strcasecmp (token, "ConcProfile") == 0) {
      error = FALSE;
      for (i = 0; i < 3; i++) {
        if ((token = strtok (NULL, sep)) != NULL) {
          cp->do_conc_prof = TRUE;
          cp->conc_prof[i] = atoi (token);
        } else {
          rflag++;
          error = TRUE;
          fprintf (stderr, "Error: ConcProfile, must have three params: sb#, slice#,row#.\n");
        }
      }

      /* PrintSlice sb# zslice# */
    } else if (strcasecmp (token, "PrintSlice") == 0) {
      if (cp->nsbslice < MAX_CTRL) {
        error = FALSE;
        for (i = 0; i < 2; i++) {
          if ((token = strtok (NULL, sep)) != NULL) {
            cp->slice[cp->nsbslice][i] = atoi (token);
          } else {
            rflag++;
            error = TRUE;
            fprintf (stderr, "Error: PrintSlice, must have two params: sb#, slice#.\n");
          }
        }
        if (!error)
          cp->nsbslice++;
      } else {                  /* exceeded MAX_CTRL print commands */
        fprintf (stderr, "Error: PrintSlice, exceed Max of %d slices.\n", MAX_CTRL);
      }

      /* PrintBBSlice zslice# */
    } else if (strcasecmp (token, "PrintBBSlice") == 0) {
      if (cp->nbbslice < MAX_CTRL) {
        if ((token = strtok (NULL, sep)) != NULL) {
          cp->bbslice[cp->nbbslice] = atoi (token);
          cp->nbbslice++;
        } else {
          fprintf (stderr, "Error: PrintBBSlice, must have two params: sb#, slice#.\n");
        }
      } else {                  /* exceeded MAX_CTRL */
        fprintf (stderr, "Error: PrintSlice, exceed Max of %d slices.\n", MAX_CTRL);
      }

      /*TempSlice int */
    } else if (strcasecmp (token, "TempSlice") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->tempslice = atoi (token);
      /*FloatDump int */
    } else if (strcasecmp (token, "BinDump") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->floatdump = atoi (token);
      /*TimeDump int */
    } else if (strcasecmp (token, "Time_Dump") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->time_dump = atoi (token);
      /*TimeExp int */
    } else if (strcasecmp (token, "Time_Exp") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->time_exp = atoi (token);
      cp->time_unit = pow (10.0, (double) cp->time_exp);
      /* SlicePFreq int */
    } else if (strcasecmp (token, "SlicePFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->slice_dmp_freq = atoi (token);
      /* BlockPFreq int */
    } else if (strcasecmp (token, "BlockPFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->blk_dmp_freq = atoi (token);
      /* SreenPFreq int -- here for backward compatibility with old ctrl files */
    } else if (strcasecmp (token, "SreenPFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL) {
        fprintf (stderr, "WARNING: read_ctrl: Old erroneous keyword (SreenPFreq) used. \n");
        cp->scr_dmp_freq = atoi (token);
      }

      /* ScreenPFreq int */
    } else if (strcasecmp (token, "ScreenPFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->scr_dmp_freq = atoi (token);

      /* RGBmode int */
    } else if (strcasecmp (token, "PltPFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->plt_dmp_freq = atoi (token);

      /* TecPlot output int */
    } else if (strcasecmp (token, "ParaviewFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
	cp->vtk_dmp_freq = atoi (token);
      /* Paraview output int*/
      
    } else if (strcasecmp (token, "PointOutputFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
	cp->point_dmp_freq = atoi (token);
      /* Point output int*/
    
    } else if (strcasecmp (token, "LineOutputFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
	cp->line_dmp_freq = atoi (token);
      /* Line output int*/
      
    } else if (strcasecmp (token, "MaxUCMapFreq") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->maxucmap_freq = atoi (token);
        if(cp->maxucmap_freq > 0) cp->maxucmap = 1; 
      /* Max undercooling map output int*/
		       
    } else if (strcasecmp (token, "RGBmode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->rgbmode = atoi (token);

      /* RGBmode int */
    } else if (strcasecmp (token, "RGBgrey") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->rgbgrey = atoi (token);

      /* RGBmode diffusion */
    } else if (strcasecmp (token, "diff_rgbmode") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diff_rgbmode = atoi (token);
      /* LogMode diffusion */
    } else if (strcasecmp (token, "diff_log_disp") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diff_log_disp = atoi (token);
      /* Ratio Disp diffusion */
    } else if (strcasecmp (token, "diff_ratio_disp") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diff_ratio_disp = atoi (token);
      /* display max for gas */
    } else if (strcasecmp (token, "gas_disp_max") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->gas_disp_max = atof (token);
      /* display max for alloy */
    } else if (strcasecmp (token, "alloy_disp_max") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->alloy_disp_max = atof (token);
      /* cap display at max for diffusion */
    } else if (strcasecmp (token, "diff_disp_cap") == 0) {
      if ((token = strtok (NULL, sep)) != NULL)
        cp->diff_disp_cap = atoi (token);
      /*********************************************************/
      /* Testing modes...                                      */
      /*********************************************************/
      /* put in only a few fixed nuclei!                       */
      /*********************************************************/
      /* FixedPore nx ny nz  */
    } else if (strcasecmp (token, "FixedPore") == 0) {
      error = FALSE;
      if (cp->nfPore < MAX_CTRL) {
        for (i = 0; i < 4; i++) {
          if ((token = strtok (NULL, sep)) != NULL) {
            cp->nPsite[cp->nfPore][i] = atof (token);
          } else {
            rflag++;
            error = TRUE;
            fprintf (stderr, "Error: FixedPore, must have four params (cell index): nx,ny,nz,thresh.\n");
          }
        }
        if (!error) {
          cp->fixed_Pore = TRUE;
          cp->nfPore++;
        }
      } else {                  /* exceeded MAX_CTRL nucleation points */
        fprintf (stderr, "Error: FixedPore, exceed Max: %d nucleation points.\n", MAX_CTRL);
      }
      /* FsPrint float */
    } else if (strcasecmp (token, "FsPrint") == 0) {
      error = FALSE;
      if (cp->nfsprint < MAX_CTRL) {
        if ((token = strtok (NULL, sep)) != NULL) {
          cp->fsprint[cp->nfsprint] = (CA_FLOAT) atof (token);
        } else {
          rflag++;
          error = TRUE;
          fprintf (stderr, "Error: FsPrint: No value \n");
        }

        if (!error) {
          cp->nfsprint++;
        }
      } else {                  /* exceeded MAX_CTRL nucleation points */
        fprintf (stderr, "Error: FsPrint, exceed Max: %d fsprint points.\n", MAX_CTRL);
      }

      /* FixedNuc nx ny nz ang0 ang1 ang2 NucThreshold *//*modified by Wei WANG on 29-01-03 */
    } else if (strcasecmp (token, "FixedNuc") == 0) {
      error = FALSE;
      if (cp->nfnuc < MAX_CTRL) {
        for (i = 0; i < 7; i++) {
          if ((token = strtok (NULL, sep)) != NULL) {
            cp->nsite[cp->nfnuc][i] = atoi (token);
          } else {
            rflag++;
            error = TRUE;
          }
        }
        if (!error) {
          cp->fixed_nuc = TRUE;
          cp->nfnuc++;
        }
      } else {                  /* exceeded MAX_CTRL nucleation points */
        fprintf (stderr, "Error: FixedNuc, exceed Max: %d nucleation points.\n", MAX_CTRL);
      }
   /*read points to track... LY*/ 
    } else if (strcasecmp (token, "PointTrack") == 0) {
      error = FALSE;
      if (cp->nptrack < MAX_CTRL) {
        for (i = 0; i < 3; i++) {
          if ((token = strtok (NULL, sep)) != NULL) {
            cp->ptrack[cp->nptrack][i] = atoi (token);
          } else {
            rflag++;
            error = TRUE;
          }
        }
        if (!error) {
          cp->nptrack ++;
        }
      } else {                  /* exceeded MAX_CTRL nucleation points */
        fprintf (stderr, "Error: PointTrack, exceed Max: %d tracking points.\n", MAX_CTRL);
      }
    }  else if (strcasecmp (token, "LineTrack") == 0) {
      error = FALSE;
      if (cp->nlntrack < MAX_CTRL) {
        for (i = 0; i < 7; i++) {
          if ((token = strtok (NULL, sep)) != NULL) {
            cp->lntrack[cp->nlntrack][i] = atoi (token);
          } else {
            rflag++;
            error = TRUE;
          }
        }
        if (!error) {
          cp->nlntrack ++;
        }
      } else {                  /* exceeded MAX_CTRL nucleation points */
        fprintf (stderr, "Error: LineTrack, exceed Max: %d tracking points.\n", MAX_CTRL);
      }
    }
    
    else if (strcasecmp (token, "CoolingRate") == 0) {
      if ((token = strtok (NULL, sep)) != NULL) {
        cp->coolrate = TRUE;
        /* allow positive or negative numbers, always cool down! */
        cp->delT = -(ABS (atof (token)));
      }

      /*********************************************************/
      /* Unknown command                                       */
      /*********************************************************/
    } else {
      fprintf (stderr, "Warning: Unknown command: %s.\n", token);
    }
  }                             /* end while */

   /***********************************************************/
  /*  check for illogical combination                        */
   /***********************************************************/
  if (cp->phase_diag_on && !cp->diffuse_alloy) {
    fprintf (stderr, "ERROR: read_ctrl: cannot use phase diagram without alloy diff\n");
    fprintf (stderr, "ERROR: read_ctrl: Turning off phase diagram\n");
    cp->phase_diag_on = 0;
  }
  if (cp->pore && !cp->diffuse) {
    fprintf (stderr, "ERROR: read_ctrl: cannot use pore without gas diff\n");
    fprintf (stderr, "ERROR: read_ctrl: Turning off pore mode.\n");
    cp->pore = 0;
  }
  /*   if (cp->decentred_octahedron  && !cp->fixed_nuc){
     fprintf(stderr,"ERROR: read_ctrl: Decentered mode needs fixed nuclei\n");
     exit(0);
     } *//* removed by Wei WANG on 17-09-02 */
  /* combine the logical values to create some useful flags */
  rflag += combine_ctrl (cp);

/* creat an array for output colours */
  if (!(cp->rgbp = (RGB_struct *) malloc (sizeof (RGB_struct)))) {
    fprintf (stderr, "ERROR: RGB_struct malloc failed\n");
    return (1);
  }
  fclose (fp);
  free (line);
  free (sep);
  return rflag;                 /* return the number of errors whilst reading file */
}                               /* end of subroutine read_ctrl */

   /*****************************************************************/
   /* subroutine to combine the logical values to create some useful flags */
   /*****************************************************************/

int combine_ctrl (Ctrl_str * cp)
{

  cp->use_global_undercooling = (cp->phase_diag_on && cp->global_undercooling);
  cp->use_csol_alloy = (cp->phase_diag_on || cp->particle);
  #ifdef CELL_NUC_OFF
  cp->use_cell_nuc = ((cp->diffuse_alloy_poly || cp->con_cast || cp->phase_diag_on || cp->fgrid_input || cp->physica) /*&& !(cp->fixed_nuc) */ );
  #else
  cp->use_cell_nuc = 1;
  #endif

  cp->use_cell_temp = (cp->con_cast || cp->fgrid_input || cp->physica || cp->ffheattran);
  if (cp->diffuse_alloy_poly == 0) {
    cp->ele_1 = 1;              /* force correct value for binary-only case */
    cp->NUM_PHS = 1;            /* pure binary/oversimplified eutectic model */
    cp->NUM_EQ = 1;             /* gibbs phase rule */
    cp->NUM_NUC_LAW = 1;        /* default case -- not nucleating eutectic */
  } else {
    cp->ele_1 = cp->NUM_COMP - 1;       /* for polycomponent THUINET model */
  }
  /** \todo  find incorrect use of con_cast and replace with use_cell_temp -- general - easy */
  return (0);
}

/*******************************************************/
/*    Entry point for reading the control file        **/
/*    Set the defaults, then read the values          **/
/*    On restart, the values are read without         **/
/*    setting the defaults -- after the restart       **/
/*    values are read in.                             **/
/*******************************************************/
int read_ctrl (char *filename, Ctrl_str * cp)
{
  int errors = 0;

  set_ctrl_defaults (cp);
  errors = read_ctrl_vals (filename, cp);
  return (errors);
}

/********************************************************/
/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
/********************************************************/
char const *rcs_id_read_ctrl_c ()
{
  static char const rcsid[] = "$Id: read_ctrl.c 1482 2009-11-25 21:08:47Z  $";

  return (rcsid);
}

/* end of rcs_id_read_ctrl_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.2  2006/11/09 13:39:40  rcatwood
RCS Log:Merged the update for ca_procast version for procast 2006.0_beta
RCS Log:
RCS Log:Revision 11.1.12.1  2006/11/02 18:48:06  rcatwood
RCS Log:Fixed up old non-diffusion ca routine to work with coupled macro
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5.2.3  2006/01/18 18:35:11  rcatwood
RCS Log:started makeing example input files for the new version
RCS Log:
RCS Log:Revision 10.5.2.2  2006/01/18 12:01:17  lthuinet
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 10.5.2.1  2006/01/10 13:58:24  rcatwood
RCS Log:Temporary branch for merging lthuinet poly-component and main branch
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 9.8.2.5.4.2  2005/01/18 11:05:35  xly
RCS Log:*** empty log message ***
RCS Log:Revision 8.5.8.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.8.2.5.4.1  2004/08/02 13:09:58  xly
RCS Log:*** empty log message ***
RCS Log:Revision 9.8.2.7  2004/09/02 17:45:33  rcatwood
RCS Log:Added option for gettign random seed from kernel urandom device
RCS Log:
RCS Log:Revision 9.8.2.5  2004/06/18 18:26:37  rcatwood
RCS Log:Added option for output freqency at uniform sim-time instead of steps
RCS Log:
RCS Log:Revision 9.8.2.4  2004/05/26 16:52:19  rcatwood
RCS Log:Removed som incorrect references to cp->mould_src
RCS Log:
RCS Log:Revision 9.8.2.3  2004/03/09 11:30:53  rcatwood
RCS Log:Changed to read materials properties from separate files for alloy and gas
RCS Log:
RCS Log:Revision 9.8.2.2  2004/03/04 11:29:25  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.8.2.1  2003/12/11 16:04:03  rcatwood
RCS Log:Branch for developing the mould solute source function
RCS Log:
RCS Log:Revision 9.8  2003/12/11 15:56:03  rcatwood
RCS Log:Changed block output : uses .blk file and separate frequecy step
RCS Log:
RCS Log:Revision 9.7  2003/11/05 17:00:39  rcatwood
RCS Log:Combined the ca_solid and ca_solid_procast into a single version (ca_solid.c)
RCS Log:this required some changes to wrapper and other files
RCS Log:
RCS Log:Revision 9.6  2003/10/27 20:01:12  rcatwood
RCS Log:Harmonized header file cpp protection
RCS Log:Fixed filename bug for restart
RCS Log:
RCS Log:Revision 9.5  2003/10/23 17:33:27  rcatwood
RCS Log:added default eutectic temp. Updated example ctrl.in file
RCS Log:
RCS Log:Revision 9.4  2003/10/23 16:32:39  rcatwood
RCS Log:chagned signal result to allow writing of the final files on USR1
RCS Log:
RCS Log:Revision 9.3  2003/10/16 12:46:33  rcatwood
RCS Log:Pore volume feedback for Procast
RCS Log:
RCS Log:Revision 9.2  2003/10/16 11:29:24  rcatwood
RCS Log:Changed incorrect use of con_cast to use_cell_temp
RCS Log:Added icc support in Makefile
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:38  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.5.2.14  2003/08/14 14:18:04  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.5.2.13  2003/06/30 16:32:54  rcatwood
RCS Log:Flux from not-casting boundary
RCS Log:
RCS Log:Revision 8.5.2.12  2003/06/16 17:19:27  rcatwood
RCS Log:Added restart with pores turned on
RCS Log:
RCS Log:Revision 8.5.2.11  2003/06/13 17:55:32  rcatwood
RCS Log:Added restart with ability to turn on gas diffusion
RCS Log:
RCS Log:Revision 8.5.2.10  2003/05/19 18:55:17  rcatwood
RCS Log:Addded option to allow horizontal or vertical directional growth
RCS Log:and flux boundary condition
RCS Log:
RCS Log:Revision 8.5.2.9  2003/05/14 11:49:53  rcatwood
RCS Log:Fixed temperature input to conform wiht new temperature calculation methods
RCS Log:and control input file
RCS Log:
RCS Log:Revision 8.5.2.8  2003/05/06 15:48:08  rcatwood
RCS Log:Altered linear binary phase diagram usage to consistently use the values input from the control files rather than the header files.
RCS Log:
RCS Log:Revision 8.5.2.7  2003/02/11 20:02:11  ww1
RCS Log:
RCS Log:add output for equivalent concentration and undercooling
RCS Log:
RCS Log:Revision 8.5.2.6  2003/02/05 13:15:21  rcatwood
RCS Log:Fixed some memory errors when list-file NOT used
RCS Log:Added ifdefs to get rid of sgi-specific image output routines
RCS Log:
RCS Log:Revision 8.5.2.5  2003/01/23 17:47:27  rcatwood
RCS Log:finite grid applied to decentered square,
RCS Log:works, but not checked for correct results.
RCS Log:
RCS Log:Revision 8.5.2.4  2003/01/22 16:53:45  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.5.2.3  2003/01/16 18:55:31  rcatwood
RCS Log:Modified location of bb into ca_wrapper from ca_solid
RCS Log:Makefile modified to use subdirectories
RCS Log:
RCS Log:Revision 8.5  2003/01/14 19:29:33  rcatwood
RCS Log:harmonised the big block with multi/procast version
RCS Log:
RCS Log:Revision 8.4  2002/12/17 13:36:40  rcatwood
RCS Log:Changed lcoation of readwrite_fg
RCS Log:
RCS Log:Revision 8.3  2002/12/13 18:06:07  rcatwood
RCS Log:Started rewriting input section for VAR input
RCS Log:
RCS Log:Revision 8.2  2002/11/04 13:35:42  rcatwood
RCS Log:Finished restart re-read reorgainization
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:03  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.3  2002/10/17 16:52:38  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.2.10.11  2002/10/09 12:19:10  rcatwood
RCS Log:alpha merge of decentered and pore versions
RCS Log:with added write_(structue)_values routines and aux program files for autogenerating
RCS Log:these routines from the structure definitions.
RCS Log:
RCS Log:Revision 7.2.10.10  2002/10/07 13:24:29  ww1
RCS Log:fixed total fraction solid at each time step and remove curv from ctrl
RCS Log:
RCS Log:Revision 7.2.10.9  2002/10/03 15:54:51  rcatwood
RCS Log:Quick fix: restart to read ctrl file and replace values
RCS Log:Fixed file name 'ca_restart.in'
RCS Log:
RCS Log:Revision 7.2.10.8  2002/09/27 13:45:14  ww1
RCS Log:add random nucleation to decentered square
RCS Log:fix nucleation in solid problem
RCS Log:
RCS Log:Revision 7.2.10.7  2002/09/19 13:25:00  rcatwood
RCS Log:added option for print at specified fraction solid
RCS Log:
RCS Log:Revision 7.2.10.6  2002/09/19 11:25:42  rcatwood
RCS Log:added control to print out at a specified fraction-solid
RCS Log:
RCS Log:Revision 7.2.10.5  2002/09/17 20:51:07  rcatwood
RCS Log:Rearranged so that read/write leave the memory clean
RCS Log:Added signal 16 (SIGUSR1) catching for trapping after finishing the current timestep.
RCS Log:
RCS Log:Revision 7.2.10.4  2002/09/12 13:53:19  rcatwood
RCS Log:Added decentered mode arrays to freeblocks
RCS Log:Tested with efence and dbmalloc
RCS Log:TODO: fix curvature access bug in fs_change_diffuse (Wei)
RCS Log:
RCS Log:Revision 7.2.10.3  2002/09/03 13:31:59  rcatwood
RCS Log:Merged with reorganized allocation routines, and adjusted nucleation to compartmentalize
RCS Log:the grain information assignment.
RCS Log:
RCS Log:Revision 7.2.10.2  2002/08/27 12:36:36  rcatwood
RCS Log:Removed some old debug ifdef's
RCS Log:Organised header files for calc_sb
RCS Log:Improved comments
RCS Log:Fixed format warnings
RCS Log:
RCS Log:Revision 7.2.10.1  2002/08/22 15:13:43  ww1
RCS Log:Merge of Wei Wang's decentered octahedron/square method
RCS Log:
RCS Log:Revision 7.2  2002/05/23 17:28:01  rcatwood
RCS Log:Used dbmalloc to eliminate all memory leaks!
RCS Log:Started close-sb routine to improve multiblock
RCS Log:
RCS Log:Revision 7.1  2002/02/14 13:15:02  rcatwood
RCS Log:Added write_block option instead of cpp definintion.
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.2  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.1  2000/10/09 16:57:04  rcatwood
RCS Log:Before changing nucleation
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:34  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.4  2000/07/11 16:32:13  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.3  2000/06/27 11:41:19  rcatwood
RCS Log:Changed pore output.
RCS Log:
RCS Log:Revision 1.2  2000/06/06 18:43:04  rcatwood
RCS Log:Changed r start and growth of pores -- pore radius equilibrium
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.7  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.6  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.6  2000/03/27 17:06:26  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.5  2000/03/27 11:21:22  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.5  2000/03/27 11:21:16  rcatwood
RCS Log:Particle dependant growth - neg. phi
RCS Log:
RCS Log:Revision 5.4  2000/03/24 19:33:30  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.4  2000/03/24 19:19:45  rcatwood
RCS Log:Particle dependant growth fixed.
RCS Log:
RCS Log:Revision 5.3  2000/03/23 19:00:16  rcatwood
RCS Log:Checkin prior to modifications for particle version
RCS Log:
RCS Log:Revision 5.3  2000/03/23 18:57:34  rcatwood
RCS Log:Particle dependant growth added.
RCS Log:
RCS Log:Revision 5.2  2000/03/06 16:51:01  rcatwood
RCS Log:added % comment
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.8  2000/01/27 12:18:48  rcatwood
RCS Log:Overgrowth addressed. Bindump of t, fs
RCS Log:
RCS Log:Revision 4.7  2000/01/13 15:32:29  rcatwood
RCS Log:Binary dump of grain and temperature,Quick temp. image.
RCS Log:
RCS Log:Revision 4.6  2000/01/10 14:57:00  rcatwood
RCS Log:Quick SIGTERM handler in ca_wrapper
RCS Log:
RCS Log:Revision 4.5  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.4  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.3  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.2  1999/12/16 19:15:49  rcatwood
RCS Log:Alloy and gas diffusion working, pores working. Changed file name for conc. output A for alloy C for gas conc.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
