/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/*RCS Id:$Id: sb_diffuse_alloy.c 1233 2008-03-12 22:03:54Z  $*/
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "blocks.h"
#include "machine.h"
#include "ca_matrix.h"
#include "sb_diffuse.h"
#include "mould_sources.h"

#define EUTECTIC  ((*newfs > 0.0 && *newfs < 1.0) && (*op >= mp->c_eut) && (*c_temp_p < mp->T_eut))

extern CA_FLOAT get_dl (CA_FLOAT temp);
extern CA_FLOAT get_ds (CA_FLOAT temp);
extern CA_FLOAT getav_d (CA_FLOAT dl, CA_FLOAT ds, CA_FLOAT fs);

int sb_diffuse_alloy (BB_struct * bp, int sbnum)
{
  int dumb = 0;
  int errflg = 0;
  int nx, ny, nz, tsteps, neut = 0;
  static int wmess = 0, emess = 0, courant_messg = 0;
  int *oni, *onip, *onend;
  SB_struct *sp;
  Solute_props *mp;
  int t, i, j, k;
  CA_FLOAT fs_av, Cinit, fsdiff;
  CA_FLOAT **sol_alloy_values;
  CA_FLOAT old_alpha, new_alpha, alpha_ratio, conc, f_ratio, f_l;       /*readability variables */
  CA_FLOAT *c_temp_p;
  CA_FLOAT *np, *op, nbconc, *nsol, *osol, nbsum;
  CA_FLOAT *oldfs, *oldfs_start, *newfs, *newfs_start;
  CA_FLOAT rmax, rmin, fstot, nbfs;
  CA_FLOAT r, rs, rl, rsrl, dtx;        /*solid, liquid */
  CA_FLOAT Clim, ClimInv, kmClimInv, fixfs;

  /** \todo Implement polycomponent when not using decentered-octahedron */

  if (bp->ctrl->diffuse_alloy_poly != FALSE){
     fprintf(stderr,"ERROR:%s: Poly-component not implemented for non-directional\n",__func__);
     exit(0);
  }

  rmin = 1;
  rmax = 0;

/* set up local neighbourhood */
/* use 6cell only for now     */
  fstot = 0;
  oni = bp->nbhd.onq;           /*padded */
  onip = oni;
  onend = oni + 6;

  nx = bp->nc[0];
  ny = bp->nc[1];
  nz = bp->nc[2];

/* set up local values and pointers */
  sp = bp->sb[sbnum];
  bp->cubeptr.curr = sbnum;
  mp = &(bp->mprops.alloyprops[0]);

  op = osol = bp->ftmp_two;
  np = nsol = sp->c_sol_alloy;
  sol_alloy_values = bp->c_sol_alloy_values->block_array;
  c_temp_p = sp->c_temp;        /* temperature of cell */
  Cinit = sp->Svals[ALLOY].Cinit;

  newfs = newfs_start = bp->ftmp_one;
  oldfs = oldfs_start = sp->c_fs;

  r = 0;
/*set up parameter values*/

  dtx = (bp->delt / (bp->size_c[0] * bp->size_c[0]));
  rs = (mp->Dsol[0] * dtx);
  rl = (mp->Dliq * dtx);
  rsrl = rs * rl;               /* precalc to save time in loop */

/* check courant stability*/
  if (rs > COURANT_LIMIT || rl > COURANT_LIMIT) {
    if (courant_messg < MAX_WARN_MSG)
      fprintf (stderr, "SB_DIFFUSE_ALLOY: WARNING: Possible instability by Courant criterion!\n Solid: , %1.2e, Liquid: %1.2e\n", rs,
               rl);
    courant_messg++;
#ifdef ERROR_EXIT
    if (courant_messg > WARN_EXIT_LIMIT) {
      fprintf (stderr, "SB_DIFFUSE_ALLOY: ERROR_EXIT: Courant Stability warning limit exceeded. %i warnings.\n", courant_messg);
      exit (courant_messg);
    }
#endif /*ERROR_EXIT */
  }
  /* make a copy of the solute concentration array */
  bp->cubeptr.ivalue = bp->Cbdy_alloy;
  errflg += fcopy_matrix (PAD, osol, nsol, bp, sol_alloy_values, sbnum);        /* (flag, to, from, bp) */
  bp->cubeptr.ivalue = 0.0;

  /* add the source at the mould surface if selected */
  if (mp->mould_src != 0) {
    sb_mould_src (bp, mp, sbnum, osol);
  }

  /* rewind array pointers */
  np = nsol;
  op = osol + bp->cubeptr.flist[0][START];      /*rewind */
  newfs = newfs_start + bp->cubeptr.flist[0][START];
  oldfs = oldfs_start;
 
  /*****  Fluid flow effect on solute diffusion   by LY  **********/
  /***     Varibles definition                                 ****/
    
    CA_FLOAT fsnb[6];
    CA_FLOAT clnb[6];
    CA_FLOAT unb[6];
    CA_FLOAT vnb[6];
    CA_FLOAT wnb[6];
	      
    int num = 0 ;
    CA_FLOAT confs;

    /*Parameters for Convection included*/
    CA_FLOAT *ophyu, *ophyv, *ophyw;
    CA_FLOAT *nphyu, *nphyv, *nphyw;
    CA_FLOAT vterm = 0.0;
    CA_FLOAT vclterm = 0.0;
    CA_FLOAT vfsterm = 0.0;
    CA_FLOAT consu, consv, consw;
    CA_FLOAT dtxx;
    CA_FLOAT vtoce;          /* Ce changes on v*/

  if(bp->phy_convection){
    
    consu=consv=consw=0;
    vtoce=0;


    nphyu = bp->ca_cell_uar;
    nphyv = bp->ca_cell_var;
    nphyw = bp->ca_cell_war;
	        
    ophyu = bp->ftmp_u;
    ophyv = bp->ftmp_v;
    ophyw = bp->ftmp_w;
		  
    fcopy_matrix (PAD, ophyu, nphyu, bp, NULL, sbnum);
    fcopy_matrix (PAD, ophyv, nphyv, bp, NULL, sbnum);
    fcopy_matrix (PAD, ophyw, nphyw, bp, NULL, sbnum);
		        
    ophyu += bp->cubeptr.flist[0][START];
    ophyv += bp->cubeptr.flist[0][START];
    ophyw += bp->cubeptr.flist[0][START];
    
    dtxx = bp->delt / bp->size_c[0];     /* dt /dx */
  }
    /**********************************************/





  
      /************************************************/
  /* now calculate the finite difference */
      /************************************************/
  /* DIFFUSION LOOP                               */
      /************************************************/
  /* Run through all cells updating as needed.    */
      /************************************************/
  for (k = 0; k < nz; k++) {    /* loop cells in z direction */
    for (j = 0; j < ny; j++) {  /* loop cells in y direction */
      for (i = 0; i < nx; i++) {        /* loop cells in x direction */
        /* skip cells that are not in the casting */
        if (*oldfs == NOT_CASTING) {
          /* do nothing */
        } else {

          old_alpha = (1.0 - (mp->km * *oldfs));
          new_alpha = (1.0 - (mp->km * *newfs));
          alpha_ratio = (old_alpha / new_alpha);

          if (EUTECTIC) {       /* this is DEFINED at the top */
            /*alpha_ratio = (1.0); */
            f_ratio = (*oldfs) / (*newfs);
            f_l = 1.0 - *newfs;
          }

          nbsum = 0;
          conc = *op;
          for (onip = oni; onip < onend; onip++) {
            nbfs = *(newfs + *onip);

            /* skip nb cells that are not in the casting */
            if (nbfs == NOT_CASTING)
              continue;
            /* averaged frac solid */
            fs_av = 0.5 * (*newfs + nbfs);

#                    define LINEAR_MEAN
#                    ifdef HARMONIC_MEAN
            r = rsrl / (rl * fs + rs * fl);
#                    endif /*HARMONIC_MEAN */
#                    ifdef LINEAR_MEAN
            r = rs * fs_av + rl * (1 - fs_av);
#                    endif /*LINEAR_MEAN */
            /* Averaged diff coeff */

            nbconc = *(op + *onip);

            nbsum += r * (nbconc - conc);
            if (r > rmax)
              rmax = r;
            if (r < rmin)
              rmin = r;
          }                     /* end of neighbour sum loop */



	  /******** Fluid Flow effect on Cl   by LY ****************/
	  if(bp->phy_convection){
	    
 		
	     num = 0;
	     confs = *newfs;	  
             consu = (*ophyu)/bp->phy_velofactor;
	     consv = (*ophyv)/bp->phy_velofactor;
	     consw = (*ophyw)/bp->phy_velofactor;
	     
	     for (onip = oni ; onip < onend; onip++){
	        
		fsnb[num] = *(newfs + *onip);
	        clnb[num] = *(op + *onip);
	        unb[num]= (*(ophyu + *onip)) / bp->phy_velofactor;
	        vnb[num]= (*(ophyv + *onip)) / bp->phy_velofactor;
		wnb[num]= (*(ophyw + *onip)) / bp->phy_velofactor;
					       	   
		num++;
	     }

	     vterm = (unb[1]-unb[0])+(unb[3]-unb[2])+(unb[5]-unb[4]);
     	     vclterm = consu * (clnb[1]-clnb[0]) + consv * (clnb[3]-clnb[2]) + consw * (clnb[5]-clnb[4]);
     	     vfsterm = consu * (fsnb[1]-fsnb[0]) + consv * (fsnb[3]-fsnb[2]) + consw * (fsnb[5]-fsnb[4]);
     	     vterm = (vterm * dtxx)/2;
    	     vclterm = (vclterm * dtxx)/2;
     	     vfsterm = (vfsterm * dtxx)/2;

     	     vtoce = (1-confs)* conc * vterm + (1-confs)*vclterm + conc * vfsterm;
	  
	     nbsum -= vtoce; 
	  }



	  /*********************************************************/

          /* eutectic rule  */
          /** \todo  improve eutectic rule -- maybe obsolote by lthuinet version */
          if (EUTECTIC) {
            *np = mp->c_eut + f_ratio * (*op - mp->c_eut) + nbsum / f_l;
            neut++;
          } else {
            *np = alpha_ratio * *op + nbsum / new_alpha;
          }
          if (*np < 0) {
            if (emess < MAX_ERR_MSG)
              fprintf (stderr, "ERROR:sb_diffuse_alloy: Possible instability *np = %g\n", *np);
            emess++;
          }
          if (*newfs > 0.0) {
            /*     calculate limiting values for this cell */
            /**  \todo  fix problem if temperature == t_pure esp. for rising liquidus slope */
            Clim = (*c_temp_p -bp->mprops.tp) * mp->m_inv_solute;
            Clim = (Clim > mp->c_eut) ? mp->c_eut : Clim;
            ClimInv = 1 / Clim;
            kmClimInv = ClimInv * mp->kminv;
            if ((Clim > Cinit) && (*np > Clim) && (*newfs < 1.0) && (*np > Cinit)) {
              /*fix frac. solid to level providing concentration Clim */
              fixfs = mp->kminv - *np * kmClimInv + *newfs * *np * ClimInv;
              if (fixfs > 0.0) {
                /* fix the fraction solid and the del_fs for latent heat calculation */
                fsdiff = *newfs - fixfs;
                sp->Tvals.del_fs -= fsdiff;
                *newfs = fixfs;
                *np = Clim;
              }
            }
          }
          fstot += *newfs;
        }                       /* end NOT_CASTING test */
        c_temp_p++;
        np++;
        oldfs++;
        newfs++;
        op++;
	if(bp->phy_convection)
	 {
	   ophyu++;
	   ophyv++;
	   ophyw++;
	 }
      }                         /*x */
      op += 2;
      newfs += 2;
      if(bp->phy_convection)
	 {
	   ophyu += 2;
	   ophyv += 2;
	   ophyw += 2;
	 }
	      
    }                           /*y */
    op += 2 * (nx + 2);
    newfs += 2 * (nx + 2);
    if(bp->phy_convection)
      {
          ophyu += 2*(nx + 2);
          ophyv += 2*(nx + 2);
          ophyw += 2*(nx + 2);
      }
	    
  }                             /*z */

  if ((bp->step % bp->ctrl->scr_dmp_freq) == 0) {
    fprintf (stderr, "sb_diffuse_alloy: rmax %.5g  rmin %.5g\n", rmax, rmin);
#ifdef VERBOSE
    fprintf (stderr, "sb_diffuse_alloy: emess: %d wmess: %d\n", emess, wmess);
    fprintf (stderr, "sb_diffuse_alloy: neut: %d\n", neut);
#endif /*VERBOSE*/
  }

  /*fix up fraction solid */

  sp->Tvals.fsavg = fstot / (bp->ncsb - sp->nmould);

  return (errflg);
}                               /* end of sb_diffuse */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_sb_diffuse_alloy_c ()
{
  static char const rcsid[] = "$Id: sb_diffuse_alloy.c 1233 2008-03-12 22:03:54Z  $";

  return (rcsid);
}

/* end of rcs_id_sb_diffuse_alloy_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5.2.2  2006/01/20 16:28:22  lthuinet
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 10.5.2.1  2006/01/11 16:31:31  rcatwood
RCS Log:Altered the use of solute information to conform with the new solute information structure table
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.3  2005/12/01 13:09:37  rcatwood
RCS Log:Fixed some implicit function declarations
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.4.4.9  2004/09/08 11:57:47  rcatwood
RCS Log:Changed verbose option in sb_diffues_alloy
RCS Log:
RCS Log:Revision 9.4.4.8  2004/05/27 11:37:49  rcatwood
RCS Log:Bug in sb_diffuse_alloy: referring to old mould_src flag
RCS Log:
RCS Log:Revision 9.4.4.7  2004/04/07 11:18:31  rcatwood
RCS Log:Fixed several division errors, added section to activate f.p.e. trapping
RCS Log:
RCS Log:Revision 9.4.4.6  2004/03/09 11:30:54  rcatwood
RCS Log:Changed to read materials properties from separate files for alloy and gas
RCS Log:
RCS Log:Revision 9.4.4.5  2004/03/04 11:29:25  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.4.4.4  2004/01/29 18:41:02  rcatwood
RCS Log:Added multiplier to piecewise-linear source
RCS Log:
RCS Log:Revision 9.4.4.3  2004/01/29 14:19:40  rcatwood
RCS Log:fixed no source (zero constant source) to work -- check speed?
RCS Log:
RCS Log:Revision 9.4.4.2  2003/12/30 23:49:30  rcatwood
RCS Log:branch fo rtesting mould source function with diffusion
RCS Log:
RCS Log:Revision 9.4.4.1  2003/12/11 16:04:03  rcatwood
RCS Log:Branch for developing the mould solute source function
RCS Log:
RCS Log:Revision 9.4  2003/11/18 13:22:40  rcatwood
RCS Log:Added routines to find and store the interface cells between the casting and the mould.
RCS Log:Added the surface cell storage structure to the subblock.
RCS Log:Improved mould source and nucleation  routines to use the surface cell structure
RCS Log:
RCS Log:Revision 9.3  2003/11/04 12:23:52  rcatwood
RCS Log:Fixed seg fault due to undefined mp pointer
RCS Log:
RCS Log:Revision 9.2  2003/10/27 20:01:12  rcatwood
RCS Log:Harmonized header file cpp protection
RCS Log:Fixed filename bug for restart
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:39  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.10  2003/08/14 14:18:04  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.1.8.9  2003/07/17 10:21:05  rcatwood
RCS Log:Fix bug from mould_src if there is no mould
RCS Log:
RCS Log:Revision 8.1.8.8  2003/06/30 16:32:54  rcatwood
RCS Log:Flux from not-casting boundary
RCS Log:
RCS Log:Revision 8.1.8.7  2003/05/19 18:55:17  rcatwood
RCS Log:Addded option to allow horizontal or vertical directional growth
RCS Log:and flux boundary condition
RCS Log:
RCS Log:Revision 8.1.8.6  2003/05/06 15:48:08  rcatwood
RCS Log:Altered linear binary phase diagram usage to consistently use the values input from the control files rather than the header files.
RCS Log:
RCS Log:Revision 8.1.8.5  2003/04/24 16:57:19  rcatwood
RCS Log:removed unnecessary warning message, which caused files to fill up.
RCS Log:
RCS Log:Revision 8.1.8.4  2003/02/27 23:04:39  rcatwood
RCS Log:Removed use of old temperature routines , all temperatures shoudl
RCS Log:be determined by checking the array c_temp in teh subblock, if the
RCS Log:subblock is open
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:46  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:02  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.3.6.1  2002/08/27 14:18:18  chirazi
RCS Log:adding files for multi-component-Procast version of CA
RCS Log:
RCS Log:Revision 7.3  2001/07/06 15:18:04  rcatwood
RCS Log:fixed another bug which caused seg fault when no pore mode selected
RCS Log:
RCS Log:Revision 7.2  2001/05/31 16:24:45  rcatwood
RCS Log:changed M macro to PD_SLOPE (Easier to find!)
RCS Log:
RCS Log:Revision 7.1  2000/12/06 21:10:40  rcatwood
RCS Log:fixed up heatfolw, tctrace
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.2  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.1.1.1  2000/10/16 17:29:10  rcatwood
RCS Log:started making non-casting cells
RCS Log:
RCS Log:Revision 6.1  2000/10/16 17:28:41  rcatwood
RCS Log:copy of 6.0 for branch point
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.3  2000/04/28 14:14:33  rcatwood
RCS Log:fixed error message to include originating routine.
RCS Log:
RCS Log:Revision 5.2  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.8  2000/02/15 15:29:11  rcatwood
RCS Log:Version after McWasp submitted
RCS Log:
RCS Log:Revision 4.7  2000/01/27 12:18:48  rcatwood
RCS Log:Overgrowth addressed. Bindump of t, fs
RCS Log:
RCS Log:Revision 4.6  2000/01/06 10:48:11  rcatwood
RCS Log:Fixed bug -- prototype in sb_diffuse_gas
RCS Log:
RCS Log:Revision 4.5  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.4  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.3  1999/12/20 13:11:23  rcatwood
RCS Log:Alloy and Gas outputs
RCS Log:
RCS Log:Revision 4.2  1999/12/16 19:15:49  rcatwood
RCS Log:Alloy and gas diffusion working, pores working. Changed file name for conc. output A for alloy C for gas conc.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
