%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% KF - Function that simulates a Kalman filter for one loop.
% Inputs:
% A - State transition matrix
% C - Observation matrix
% Q - System noise covariance matrix
% R - Observation noise covariance matrix
% xInit - Initial value of state
% PInit - Initial value of error covariance matrix
% z - Clean observations (determines length of loop)
%
% Output:
% x - State variable
% zNoise - Noisy observations
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [x,zNoise] = KF(A,C,Q,R,xInit,PInit,z)

    % Allocate memory for speed
    x = zeros(size(xInit,1),length(z));
    % Initialize Kalman filter
    x(:,1) = xInit;
    P = PInit;
    
    % Add noise to signal z
    zNoise = z + sqrt(R)*randn(size(R,1),length(z));

    % Kalman filter loop
    for i=1:length(z)-1
        % Prediction
        x_Prior = A*x(:,i);
        P = A*P*A'+Q;

        % Correction
        K = P*C'/(C*P*C'+R);
        x(:,i+1) = x_Prior + K*(zNoise(:,i)-C*x_Prior);
        P = (eye(size(xInit,1))-K*C)*P;
    end