%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute optimal controller and estimator for generalized LQG
%
% u(t)    = -L(t) x(t)
% x(t+1)  = A x(t) + B (I + Sum(C(i) rnd_1)) u(t) + C0 rnd_n
% y(t)    = H x(t) + Sum(D(i) rnd_1) x(t) + D0 rnd_n
% xhat(t+1) = A xhat(t) + B u(t) + K(t) (y(t) - H xhat(t)) + E0 rnd_n
% x(1)    ~ mean X1, covariance S1
%
% cost(t) = u(t)' R u(t) + x(t)' Q(t) x(t)
%
% K        Filter gains
% L        Control gains
% Cost     Expected cost (per iteration)
% Xa       Expected trajectory
% XSim     Simulated trajectories
% Additive 0 - multiplicative, 1 - additive noise
% UAvg     Set additive noise to average value of multiplicative noise
%
% This is an implementation of the algorithm described in:
%  Todorov, E. (2005) Stochastic optimal control and estimation
%  methods adapted to the noise characteristics of the
%  sensorimotor system. Neural Computation 17(5): 1084-1108
% The paper is available online at www.cogsci.ucsd.edu/~todorov
% Copyright (C) Emanuel Todorov, 2004-2006
% --------
% Modified by Atsushi Takagi (09/09/13)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [XSim,UNoise] = ReachingLQG(A,B,C,C0,H,D0,E0,Q,R,X1,S1,NoiseSim,Additive,UAvg)

% Numerical parameters
MaxIter = 500;
Eps = 10^-15;

% Determine sizes
szX = size(A,1);
szU = size(B,2);
szY = size(H,1);
szD0 = size(D0,2);
szE0 = size(E0,2);
N = size(Q,3);

% Replicate D into a matrix
D = zeros(szY,szX);

% If D0 or E0 are scalar, set them to 0 matrices and adjust size
if length(D0(:))==1 && D0(1)==0
    D0 = zeros(szY,1);
end
if length(E0(:))==1 && E0(1)==0
    E0 = zeros(szX,1);
end


% Initialize policy and filter
K = zeros(szX,szY,N-1);
L = zeros(szU,szX,N-1);
Cost = zeros(MaxIter);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Run iterative algorithm until convergence or MaxIter

for Iter = 1:MaxIter
   
   % Initialize covariances
   SiE = S1;
   SiX = X1*X1';
   SiXE = zeros(szX,szX);
   
   % Forward pass - recompute Kalman filter   
   for k = 1:N-1
      % Compute Kalman gain
      DSiD = D*(SiE+SiX+SiXE+SiXE')*D';
      K(:,:,k) = A*SiE*H'*pinv(H*SiE*H'+D0*D0'+DSiD);
      
      % Update SiX, SiE, SiXE
      SiX = E0*E0'+K(:,:,k)*H*SiE*A'+(A-B*L(:,:,k))*SiX*(A-B*L(:,:,k))'+...
          (A-B*L(:,:,k))*SiXE*H'*K(:,:,k)'+K(:,:,k)*H*SiXE'*(A-B*L(:,:,k))';
      SiE = E0*E0'+C0*C0'+(A-K(:,:,k)*H)*SiE*A'+B*diag(diag(L(:,:,k)*SiX*L(:,:,k)').*C.^2)*B';
      SiXE = (A-B*L(:,:,k))*SiXE*(A-K(:,:,k)*H)'-E0*E0';
   end
   
   % Initialize optimal cost-to-go function
   Sx = Q(:,:,N);
   Se = zeros(szX,szX);
   Cost(Iter) = 0;
   
   % Backward pass - recompute control policy
   for k=N-1:-1:1
      % Update cost
      Cost(Iter) = Cost(Iter)+trace(Sx*C0*C0')+trace(Se*(K(:,:,k)*D0*D0'*K(:,:,k)'+E0*E0'+C0*C0'));
      
      % Controller
      L(:,:,k) = pinv(R+B'*Sx*B+diag(diag(B'*(Sx+Se)*B).*C.^2))*B'*Sx*A;
      
      % Update Sx and Se
      Sx = Q(:,:,k)+A'*Sx*(A-B*L(:,:,k));
      KSeK = K(:,:,k)'*Se*K(:,:,k);
      Sx = Sx+D'*KSeK*D;   
      Se = A'*Sx*B*L(:,:,k)+(A-K(:,:,k)*H)'*Se*(A-K(:,:,k)*H);
   end

   % Adjust cost
   Cost(Iter) = Cost(Iter)+X1'*Sx*X1+trace((Se+Sx)*S1);     
   
   % Check convergence of Cost
   if Iter>1 && abs(Cost(Iter-1)-Cost(Iter))<Eps
      break
   end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulate noisy trajectories
   
% Square root of S1
[u,s,v] = svd(S1);
   
% Initialize
XSim = zeros(szX,NoiseSim,N);
Xhat = zeros(szX,NoiseSim,N);
UNoise = zeros(size(C,1),NoiseSim,N);
Xhat(:,:,1) = repmat(X1,[1 NoiseSim]);
XSim(:,:,1) = repmat(X1,[1 NoiseSim])+u*diag(sqrt(diag(s)))*v'*randn(szX,NoiseSim);
   
for k=1:N-1
    % Update control and cost
    U = -L(:,:,k)*Xhat(:,:,k);
      
    % Compute noisy control
    Un = U;
    if Additive==0
        UNoise(:,:,k) = U.*randn(szU,NoiseSim).*repmat(C,[1,NoiseSim]);
        Un = Un+UNoise(:,:,k);
    else
        UNoise(:,:,k) = UAvg.*randn(szU,NoiseSim).*repmat(C,[1,NoiseSim]);
        Un = Un+UNoise(:,:,k);
    end
      
    % Compute noisy observation
    y = H*XSim(:,:,k)+D0*randn(szD0,NoiseSim)+D*XSim(:,:,k).*repmat(randn(1,NoiseSim),[szY 1]);
    
      
    XSim(:,:,k+1) = A*XSim(:,:,k)+B*Un+repmat(C0,[1 NoiseSim]).*randn(size(C0,1),NoiseSim);
    Xhat(:,:,k+1) = A*Xhat(:,:,k)+B*U+K(:,:,k)*(y-H*Xhat(:,:,k))+E0*randn(szE0,NoiseSim);
end

