%%%%%%%%%%% Reaching with and without glasses %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was used to produce Figure 9.5 in
% /////////////////////////////////////////////////////////////////////////
%  "Human Robotics: Neuromechanics and Motor Control"
%   by Etienne Burdet, David Franklin W. and Theodore Milner E.
% /////////////////////////////////////////////////////////////////////////
% The simulation uses a linear quadratic regulator and set-point control
% to generate the mean trajectory of the hand that the visual and 
% proprioceptive sensors measure during a reaching movement. 
%
% Based on these mean trajectories, two cases are simulated: 
% 1. Reaching without glasses such that both vision and
%    proprioception have equal sensory noise.
% 2. Reaching with glasses where vision has significantly less noise than
%    proprioception.
% --------
% Atsushi Takagi (09/09/13)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all;clc;close all;

% Time step size
dt = 0.01;
% Length of movement
N = 0.8/dt;
% Muscle time constant
Tau = 0.3;
% Number of simulations
NoiseSim = 1000;

% State vector
x1(:,1) = [0 0 0 0];
x2(:,1) = [0 0 0 0];
% State size
SizeX = size(x1,1);

% Cost of state at END of movement
Q = zeros(SizeX,SizeX);
Q(1,1) = 1;         % Position cost
Q(2,2) = 0.5;       % Velocity cost
Q(3,3) = 0.1;       % Acceleration cost

% Control cost
R(1,1) = 10^-10;

% Define system matrices (based on Winter's model, 1990)
A = [1  dt 0         0;
     0  1  dt        0;
     0  0  1-dt/Tau  dt/Tau;
     0  0  0         1-dt/Tau];
 
B = [0; 0; 0; dt/Tau];

% Cost to go function
St = Q;

% Find optimal gain L through backwards pass of Riccati
for k=N-1:-1:1
    L(:,:,k) = pinv(R + B'*St*B)*B'*St*A;
    if k==N-1
        St = Q + A'*St*(A-B*L(:,:,k));
    else
        St = A'*St*(A-B*L(:,:,k));
    end
end

% Visual and proprioceptive state estimates at end of movement
VisionEst = [0.5;0;0;0];
PropEst = [0.3;0;0;0];

% Simulate mean trajectories of vision and proprioception estimates of hand
for k=1:N-1
    % Vision estimate
    u1 = -L(:,:,k)*(x1(:,k)-VisionEst);
    x1(:,k+1) = A*x1(:,k) + B*u1;
    
    % Proprioception estimate
    u2 = -L(:,:,k)*(x2(:,k)-PropEst);
    x2(:,k+1) = A*x2(:,k) + B*u2;
end

% Noise covariance matrices
R1 = diag([0.0005 0.0005]);     % No glasses
R2 = diag([0.0001 0.0005]);     % With glasses

% Initial error covariance matrix
PInit = 10^5*eye(SizeX);

% Mean observation
z = [x1(1,:);x2(1,:)];

% Observation matrix
C = zeros(2,size(x1(:,1),1));
C(1,1) = 1;
C(2,1) = 1;

% System noise matrix
QKalman = diag([dt^5,dt^3,dt,dt]);

% Allocate memory
xHatNoGlasses = zeros(SizeX,N,NoiseSim);
xHatGlasses = zeros(SizeX,N,NoiseSim);
zNoGlasses = zeros(2,N,NoiseSim);
zGlasses = zeros(2,N,NoiseSim);

% Simulate reaching movements with and without glasses
for s=1:NoiseSim  
    [xHatNoGlasses(:,:,s),zNoGlasses(:,:,s)] = KF(A,C,QKalman,R1,x1(:,1),PInit,z);
    [xHatGlasses(:,:,s),zGlasses(:,:,s)] = KF(A,C,QKalman,R2,x1(:,1),PInit,z);
end

% Plots
set(0,'DefaultFigureWindowStyle','docked','DefaultAxesFontsize',13);

% Define time vector for plotting
Time = (0:N-1)*dt;
% Calculate mean trajectory
TrajMean = @(x)(mean(squeeze(x),2));
% Calculate standard deviation of trajectory
TrajSTD = @(x)(std(squeeze(x),0,2));

figure(1); set(gcf,'color','white');
subplot(2,1,1); hold on;

plot(Time,TrajMean(zNoGlasses(1,:,:)),'b');
plot(Time,TrajMean(zNoGlasses(2,:,:)),'r');
plot(Time,TrajMean(xHatNoGlasses(1,:,:)),'k','linewidth',2);

X=[Time,fliplr(Time)];
Y1=[(TrajMean(zNoGlasses(1,:,:))+TrajSTD(zNoGlasses(1,:,:)))',fliplr((TrajMean(zNoGlasses(1,:,:))-TrajSTD(zNoGlasses(1,:,:)))')];
Y2=[(TrajMean(zNoGlasses(2,:,:))+TrajSTD(zNoGlasses(2,:,:)))',fliplr((TrajMean(zNoGlasses(2,:,:))-TrajSTD(zNoGlasses(2,:,:)))')];

h(1) = fill(X,Y1,'b');
h(2) = fill(X,Y2,'r');

xlabel('Time [s]');
ylabel('Position [m]');
legend('Vision','Proprioception','location','northwest');
title('Reaching WITHOUT glasses');

subplot(2,1,2); hold on;
plot(Time,TrajMean(zGlasses(1,:,:)),'b');
plot(Time,TrajMean(zGlasses(2,:,:)),'r');
plot(Time,TrajMean(xHatGlasses(1,:,:)),'k','linewidth',2);

Y3=[(TrajMean(zGlasses(1,:,:))+TrajSTD(zGlasses(1,:,:)))',fliplr((TrajMean(zGlasses(1,:,:))-TrajSTD(zGlasses(1,:,:)))')];
Y4=[(TrajMean(zGlasses(2,:,:))+TrajSTD(zGlasses(2,:,:)))',fliplr((TrajMean(zGlasses(2,:,:))-TrajSTD(zGlasses(2,:,:)))')];

h(3) = fill(X,Y3,'b');
h(4) = fill(X,Y4,'r');

xlabel('Time [s]');
ylabel('Position [m]');
title('Reaching WITH glasses');

set(h,'facealpha',0.5,'edgealpha',0);


